package com.xebialabs.deployit.task.deployment;

import java.util.Collection;
import java.util.List;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.xebialabs.deployit.Step;
import com.xebialabs.deployit.ci.Deployment;
import com.xebialabs.deployit.ci.DeploymentPackage;
import com.xebialabs.deployit.ci.Environment;
import com.xebialabs.deployit.ci.mapping.Mapping;
import com.xebialabs.deployit.plugin.PojoConverter;
import com.xebialabs.deployit.repository.ConfigurationItemEntity;
import com.xebialabs.deployit.repository.ItemAlreadyExistsException;
import com.xebialabs.deployit.repository.RepositoryObjectEntity;
import com.xebialabs.deployit.repository.RepositoryService;
import com.xebialabs.deployit.security.SecurityCallback;
import com.xebialabs.deployit.security.SecurityTemplate;
import com.xebialabs.deployit.security.UsernameAndPasswordCredentials;
import com.xebialabs.deployit.service.deployment.DeploymentService;
import com.xebialabs.deployit.task.Task;

@SuppressWarnings("serial")
public class InitialDeploymentTask extends Task {

	private final ConfigurationItemEntity sourceEntity;
	private final ConfigurationItemEntity targetEntity;
	private final Collection<ConfigurationItemEntity> mappingEntities;

	public InitialDeploymentTask(final ConfigurationItemEntity sourceEntity, final ConfigurationItemEntity targetEntity,
	        final Collection<ConfigurationItemEntity> mappingEntities, final List<Step> steps, final RepositoryService repositoryService,
	        final PojoConverter pojoConverter, final PojoConverter.Context... pojoConverterContexts) {
		super(steps, repositoryService, pojoConverter, pojoConverterContexts);
		this.sourceEntity = sourceEntity;
		this.targetEntity = targetEntity;
		this.mappingEntities = mappingEntities;
	}

	@Override
	public void doAfterTaskStateChangedToDone() {
		UsernameAndPasswordCredentials owner = getOwnerCredentials();
		SecurityTemplate.executeAs(owner.getUsername(), owner.getPassword(), new SecurityCallback<Object>() {
			@Override
			public Object doAs() {
				PojoConverter.Context context = pojoConverter.getContext();
				try {
					final DeploymentPackage pkg = context.toPojo(sourceEntity);
					final Environment env = context.toPojo(targetEntity);

					// Save the deployment without the mappings so that the deployment exists in JCR
					final Deployment newDeployment = new Deployment(pkg, env);
					String baseDeploymentLabel = DeploymentService.generateDeploymentLabel(pkg, env);
					for (int i = 0;; i++) {
						if (i == 0) {
							newDeployment.setLabel(baseDeploymentLabel);
						} else {
							newDeployment.setLabel(baseDeploymentLabel + " (" + i + ")");
						}
						try {
							repositoryService.create(context.toEntity(newDeployment));
							break;
						} catch (ItemAlreadyExistsException exc) {

						}
					}
					logger.debug("Saved deployment with label " + newDeployment.getLabel());

					// Save the mappings that are stored as subnodes of the deployment that was just created
					for (ConfigurationItemEntity each : mappingEntities) {
						String mappingId = each.getId();
						for (int i = 0;; i++) {
							if (i != 0) {
								each.setId(mappingId + " (" + i + ")");
							}
							try {
								repositoryService.create(each);
								break;
							} catch (ItemAlreadyExistsException exc) {
								// Ignored, we increment the id.
							}
						}
						logger.debug("Saved mapping with label " + each.getId());
						newDeployment.addMapping((Mapping<?, ?>) context.toPojo(each));
					}

					// Update the deployment to include the creating task id and to refer to the mappings that were just created
					RepositoryObjectEntity updatedNewDeploymentEntity = context.toEntity(newDeployment);
					updatedNewDeploymentEntity.setCreatingTaskId(getId());
					repositoryService.update(updatedNewDeploymentEntity);
				} finally {
					context.destroy();
				}
				return null;
			}
		});
	}

	@Override
	public boolean isDeploymentTask() {
		return true;
	}

	private Logger logger = LoggerFactory.getLogger(InitialDeploymentTask.class);

}
