package com.xebialabs.deployit.task;

import static com.xebialabs.deployit.task.TaskStep.StepState.DONE;
import static com.xebialabs.deployit.task.TaskStep.StepState.EXECUTING;
import static com.xebialabs.deployit.task.TaskStep.StepState.FAILED;
import static com.xebialabs.deployit.task.TaskStep.StepState.PENDING;
import static com.xebialabs.deployit.task.TaskStep.StepState.SKIPPED;
import static org.slf4j.LoggerFactory.getLogger;

import java.io.PrintWriter;
import java.io.Serializable;
import java.io.StringWriter;
import java.util.Calendar;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.xebialabs.deployit.Step;
import com.xebialabs.deployit.StepExecutionContext;

@SuppressWarnings("serial")
public class TaskStep implements Serializable {

	private final Step implementation;

	private final String description;

	private StepState state;

	private Calendar startDate;

	private Calendar completionDate;

	private String log = "";

	private Calendar lastModificationDate;

	public TaskStep(Step step) {
		this.implementation = step;
		this.description = step.getDescription();
		setState(PENDING);
	}

	/**
	 * Invoked from {@link JcrTaskArchive} to instantiate a {@link TaskStep} from the archive.
	 */
	TaskStep(String description, StepState state) {
		this.description = description;
		setState(state);
		implementation = null;
	}

	public void execute(TaskExecutionContext taskContext) {
		if ((state != PENDING && state != FAILED)) {
			logger.debug("Will not execute: {} with description: {} because it has state: {}", new Object[] { implementation, implementation.getDescription(),
			        state });
			return;
		}

		setStartDate();
		setState(EXECUTING);
		clearLog();
		logger.info("Executing: {} with description: {}", implementation, implementation.getDescription());
		final TaskStepExecutionContext context = new TaskStepExecutionContext(taskContext);
		try {
			final boolean result = implementation.execute(context);
			setState(result ? DONE : FAILED);
		} catch (Exception exc) {
			context.logError("Step failed", exc);
			setState(FAILED);
		} catch (Throwable t) {
			// Any non-exception throwable is so severe we abort the task!
			context.logError("Step failed badly, aborting!", t);
			Thread.currentThread().interrupt();
			setState(FAILED);
		}
		setCompletionDate();
	}

	public Step getImplementation() {
		return implementation;
	}

	public String getDescription() {
		return description;
	}

	public StepState getState() {
		return state;
	}

	public boolean isFailed() {
		return state == FAILED;
	}

	public boolean isSkipped() {
		return state == SKIPPED;
	}

	public boolean canSkip() {
		return state == PENDING || state == FAILED;
	}

	public void skip() {
		setState(SKIPPED);
	}

	public void unskip() {
		setState(PENDING);
	}

	void setState(StepState state) {
		this.state = state;
		setLastModificationDate();
	}

	public Calendar getStartDate() {
		return startDate;
	}

	/**
	 * Sets the start date to the current date/time.
	 */
	private void setStartDate() {
		startDate = Calendar.getInstance();
		setLastModificationDate();
	}

	/**
	 * Invoked from {@link JcrTaskArchive} to instantiate a {@link TaskStep} from the archive.
	 */
	void setStartDate(final Calendar startDate) {
		this.startDate = startDate;
	}

	public Calendar getCompletionDate() {
		return completionDate;
	}

	/**
	 * Sets the completion date to the current date/time.
	 */
	private void setCompletionDate() {
		completionDate = Calendar.getInstance();
		setLastModificationDate();
	}

	/**
	 * Invoked from {@link JcrTaskArchive} to instantiate a {@link TaskStep} from the archive.
	 */
	void setCompletionDate(final Calendar completionDate) {
		this.completionDate = completionDate;
	}

	public String getLog() {
		return log;
	}

	/**
	 * Clears the log.
	 */
	public void clearLog() {
		log = "";
	}

	/**
	 * Invoked from {@link JcrTaskArchive} to instantiate a {@link TaskStep} from the archive.
	 */
	void setLog(final String log) {
		this.log = log;
	}

	public Calendar getLastModificationDate() {
		return lastModificationDate;
	}

	protected void setLastModificationDate() {
		this.lastModificationDate = Calendar.getInstance();
	}

	public enum StepState {
		PENDING, EXECUTING, DONE, FAILED, SKIPPED
	}

	class TaskStepExecutionContext implements StepExecutionContext {

		private TaskExecutionContext taskContext;

		private Logger stepLogger;

		TaskStepExecutionContext(TaskExecutionContext taskContext) {
			this.taskContext = taskContext;
			this.stepLogger = LoggerFactory.getLogger(implementation.getClass());
		}

		@Override
		public void logOutput(String output) {
			stepLogger.info(output);
			log += output + "\n";
			setLastModificationDate();
		}

		@Override
		public void logError(String error) {
			stepLogger.error(error);
			log += error + "\n";
			setLastModificationDate();
		}

		@Override
		public void logError(String error, Throwable t) {
			stepLogger.error(error, t);
			log += error + "\n";
			StringWriter stringWriter = new StringWriter();
			t.printStackTrace(new PrintWriter(stringWriter));
			log += stringWriter.toString() + "\n";
			setLastModificationDate();
		}

		@Override
		public Object getAttribute(String name) {
			Object object = taskContext.getAttribute(name);
			stepLogger.debug("Getting value of attribute {}: {}", name, object);
			return object;
		}

		@Override
		public void setAttribute(String name, Object object) {
			stepLogger.debug("Setting value of attribute {}: {}", name, object);
			taskContext.setAttribute(name, object);
		}

	}

	private static final Logger logger = LoggerFactory.getLogger(TaskStep.class);

}
