package com.xebialabs.deployit.task;

import static com.google.common.base.Preconditions.checkState;
import static com.xebialabs.deployit.checks.Checks.checkArgument;
import static com.xebialabs.deployit.task.Task.State.CANCELLED;
import static com.xebialabs.deployit.task.Task.State.DONE;
import static com.xebialabs.deployit.task.Task.State.EXECUTING;
import static com.xebialabs.deployit.task.Task.State.PENDING;
import static com.xebialabs.deployit.task.Task.State.STOPPED;

import java.io.Serializable;
import java.util.Calendar;
import java.util.Collections;
import java.util.List;
import java.util.concurrent.FutureTask;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.xebialabs.deployit.Step;
import com.xebialabs.deployit.event.Event;
import com.xebialabs.deployit.event.EventBus;
import com.xebialabs.deployit.plugin.PojoConverter;
import com.xebialabs.deployit.repository.RepositoryService;
import com.xebialabs.deployit.security.UsernameAndPasswordCredentials;
import com.xebialabs.deployit.task.TaskStep.StepState;

@SuppressWarnings("serial")
public class Task implements Serializable, Runnable {

	private String id;

	public static String DEPLOYMENT_PREFIX = "Deployment";
	public static String UNDEPLOYMENT_PREFIX = "Undeployment";
	public static String UPGRADE_PREFIX = "Upgrade";

	// take care before changing these formats(they are used for parsing the package, env etc, for example look TasksInDateRangeReport)
	public static String DEPLOYMENT_TASK_LABEL_FORMAT = DEPLOYMENT_PREFIX + " of package:%s version:%s to env:%s";
	public static String UPGRADE_TASK_LABEL_FORMAT = UPGRADE_PREFIX + " of package:%s from version:%s to version:%s to env:%s";
	public static String UNDEPLOYMENT_TASK_LABEL_FORMAT = UNDEPLOYMENT_PREFIX + " of package:%s version:%s from env:%s";

	private String label = "Task #" + getClass();

	private String owner;

	private transient UsernameAndPasswordCredentials ownerCredentials;

	private Calendar startDate;

	private Calendar completionDate;

	private final StepList steps;

	private volatile boolean stopRequested;

	private volatile transient FutureTask<Object> wrappingTask;

	private volatile State state = PENDING;

	protected final transient RepositoryService repositoryService;

	protected final transient PojoConverter pojoConverter;

	protected final transient PojoConverter.Context[] pojoConverterContextsToDestroy;

	public enum State {
		PENDING, EXECUTING, DONE, STOPPED, CANCELLED
	}

	public Task(final List<? extends Step> steps, final RepositoryService repositoryService, final PojoConverter pojoConverter,
	        final PojoConverter.Context... pojoConverterContextsToDestroy) {
		this.steps = new StepList(steps);
		this.repositoryService = repositoryService;
		this.pojoConverter = pojoConverter;
		this.pojoConverterContextsToDestroy = pojoConverterContextsToDestroy;
	}

	Task(final StepList stepList, final State state, final RepositoryService repositoryService, final PojoConverter pojoConverter,
	        final PojoConverter.Context... pojoConverterContextsToDestroy) {
		this.steps = stepList;
		this.state = state;
		this.repositoryService = repositoryService;
		this.pojoConverter = pojoConverter;
		this.pojoConverterContextsToDestroy = pojoConverterContextsToDestroy;
	}

	protected boolean executeSteps() {
		TaskExecutionContext context = new TaskExecutionContext();
		try {
			while (steps.hasMoreSteps()) {
				TaskStep currentStep = steps.getNextStep();
				currentStep.execute(context);
				logger.info("Executed {}", currentStep);
				if (currentStep.isFailed()) {
					steps.rollback();
					return false;
				}

				// Step is DONE, but thread has been interrupted or has been requested to stop, do not continue now!
				if (isAborted() || isStopped()) {
					return false;
				}
			}
		} finally {
			context.destroy();
		}

		return true;
	}

	@Override
	public void run() {
		stopRequested = false;
		checkState(!isExecuting(), "task is already executing");
		checkState(wrappingTask != null, "task should have a wrapper set!");

		if (!isReadyForExecution())
			return;

		if (state == PENDING) {
			startDate = Calendar.getInstance();
		}

		try {
			setState(EXECUTING);
			final boolean result = executeSteps();
			logger.debug("Executed all steps of task {}", id);
			if (!result && isAborted()) {
				setState(STOPPED);
			} else if (!result && steps.hasMoreSteps()) {
				setState(STOPPED);
			} else {
				setState(DONE);
			}
		} finally {
			// I'm done, don't need my wrappingTask anymore!
			wrappingTask = null;
			// Clear the abort flag
			Thread.interrupted();
		}
	}

	/**
	 * override this method if something needs to be executed/saved/deleted after the state is changed to DONE
	 */
	public void doAfterTaskStateChangedToDone() {
	}

	/**
	 * override this method if something needs to be executed/saved/deleted after the state is changed to ABORTED
	 */
	public void doAfterTaskStateChangedToAborted() {
	}

	public void stop() {
		logger.info("Stop requested for me ({})", id);
		stopRequested = true;
	}

	/**
	 * Aborts a task when it is executing.
	 */
	public void abort() {
		wrappingTask.cancel(true);
	}

	/**
	 * Cancels a task that has not started executing or that has stopped executing (but is not DONE).
	 */
	public void cancel() {
		checkArgument(isReadyForExecution(), "Can only cancel a task that is PENDING or STOPPED");
		setState(CANCELLED);
	}

	/**
	 * Destroys a task. Cleans up any temporary files needed for execution of the task.
	 */
	public void destroy() {
		if(pojoConverterContextsToDestroy == null) {
			return;
		}

		for (PojoConverter.Context each : pojoConverterContextsToDestroy) {
			each.destroy();
		}
	}

	public boolean isReadyForExecution() {
		return state == PENDING || state == STOPPED;
	}

	public boolean isExecuting() {
		return state == EXECUTING;
	}

	private boolean isStopped() {
		return stopRequested;
	}

	private boolean isAborted() {
		return wrappingTask.isCancelled() || Thread.currentThread().isInterrupted();
	}

	public List<TaskStep> getSteps() {
		return Collections.unmodifiableList(steps);
	}

	public void setWrappingTask(final FutureTask<Object> wrappingTask) {
		this.wrappingTask = wrappingTask;
	}

	public String getId() {
		return id;
	}

	public void setId(String id) {
		this.id = id;
	}

	public String getLabel() {
		return label;
	}

	public void setLabel(String label) {
		this.label = label;
	}

	public String getOwner() {
		return owner;
	}

	public void setOwner(final String owner) {
		this.owner = owner;
	}

	public UsernameAndPasswordCredentials getOwnerCredentials() {
		return ownerCredentials;
	}

	public void setOwnerCredentials(final UsernameAndPasswordCredentials ownerCredentials) {
		this.ownerCredentials = ownerCredentials;
	}

	public Calendar getStartDate() {
		return startDate;
	}

	void setStartDate(final Calendar startDate) {
		this.startDate = startDate;
	}

	public Calendar getCompletionDate() {
		return completionDate;
	}

	void setCompletionDate(final Calendar completionDate) {
		this.completionDate = completionDate;
	}

	public int getCurrentStepNr() {
		return steps.getCurrentStepNr();
	}

	public int getNrOfSteps() {
		return steps.getNrOfSteps();
	}

	public TaskStep getStep(final int stepNr) {
		return steps.getStep(stepNr);
	}

	public FutureTask<Object> getWrappingTask() {
		return wrappingTask;
	}

	public State getState() {
		return state;
	}

	public void processAfterRecovery() {
		for (TaskStep eachStep : getSteps()) {
			if (eachStep.getState() == StepState.EXECUTING) {
				eachStep.setLog(eachStep.getLog() + "Step was executing when the server crashed. Try executing it again.\n");
				eachStep.setState(StepState.FAILED);
				steps.rollback();
				break;
			}
		}
		setState(STOPPED);
	}

	void setState(final State state) {
		if (state == this.state) {
			return;
		}

		boolean shouldDestroy = false;

		this.state = state;

		switch (state) {
		case EXECUTING:
			logger.info("Execution of task " + getId() + " has started");
			break;
		case STOPPED:
			logger.error("Execution of task " + getId() + " was stopped");
			completionDate = Calendar.getInstance();
			try {
				doAfterTaskStateChangedToAborted();
			} catch (RuntimeException e) {
				logger.error("Execution of task " + getId() + " has stopped because of an internal error", e);
				this.state = STOPPED;
				throw e;
			}
			break;
		case CANCELLED:
			logger.info("Execution of task " + getId() + " has been cancelled");
			completionDate = Calendar.getInstance();
			shouldDestroy = true;
			break;
		case DONE:
			logger.info("Execution of task " + getId() + " has completed successfully");
			completionDate = Calendar.getInstance();
			shouldDestroy = true;
			try {
				doAfterTaskStateChangedToDone();
			} catch (RuntimeException e) {
				logger.error("Execution of task " + getId() + " has stopped because of an internal error", e);
				this.state = STOPPED;
				throw e;
			}
			break;
		}

		try {
			EventBus.publish(new TaskStateChangeEvent(this, this.state));
		} catch (RuntimeException e) {
			logger.error("Exception while publishing TaskStateChangeEvent: ", e);
			throw e;
		}

		if (shouldDestroy) {
			destroy();
		}
	}

	public boolean isDeploymentTask() {
		return true;
	}

	public class TaskStateChangeEvent implements Event {
		private final Task task;
		private final State oldState;

		public TaskStateChangeEvent(Task task, final State oldState) {
			this.task = task;
			this.oldState = oldState;
		}

		public Task getTask() {
			return task;
		}

		public State getOldState() {
			return oldState;
		}
	}

	private static final Logger logger = LoggerFactory.getLogger(Task.class);
}
