package com.xebialabs.deployit.task;

import static com.xebialabs.deployit.jcr.JcrConstants.ID_PROPERTY_NAME;
import static com.xebialabs.deployit.jcr.JcrConstants.STEP_NODE_NAME_PREFIX;
import static com.xebialabs.deployit.jcr.JcrConstants.TASKS_NODE_NAME;
import static com.xebialabs.deployit.jcr.JcrConstants.TASK_NODETYPE_NAME;
import static javax.jcr.query.Query.JCR_SQL2;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Collection;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

import javax.jcr.Node;
import javax.jcr.NodeIterator;
import javax.jcr.PathNotFoundException;
import javax.jcr.RepositoryException;
import javax.jcr.Session;
import javax.jcr.Value;
import javax.jcr.ValueFactory;
import javax.jcr.ValueFormatException;
import javax.jcr.query.Query;
import javax.jcr.query.QueryManager;
import javax.jcr.query.QueryResult;
import javax.jcr.query.RowIterator;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import com.google.common.collect.Lists;
import com.xebialabs.deployit.exception.NotFoundException;
import com.xebialabs.deployit.jcr.JcrCallback;
import com.xebialabs.deployit.jcr.JcrTemplate;
import com.xebialabs.deployit.plugin.PojoConverter;
import com.xebialabs.deployit.repository.ArchivedTaskSearchParameters;
import com.xebialabs.deployit.repository.RepositoryService;

@Component
public class JcrTaskArchive implements TaskArchive {

	private static final String LABEL = "label";
	private static final String STATE = "state";
	private static final String START_DATE = "startDate";
	private static final String COMPLETION_DATE = "completionDate";
	private static final String NR_OF_STEPS = "nrOfSteps";
	private static final String CURRENT_STEP_NR = "currentStepNr";
	private static final String OWNING_USER = "ownerUser";
	private static final String LOG = "log";
	private static final String DESCRIPTION = "description";
	
	private static final String JCR_SQL_QUERY = "SELECT task.* FROM [deployit:task] AS task WHERE ( task.state = 'DONE' OR task.state = 'STOPPED' )  AND task.completionDate >= $_startDate AND task.completionDate < $_endDate ";

	private final JcrTemplate jcrTemplate;

	private final RepositoryService repositoryService;

	private final PojoConverter pojoConverter;

	@Autowired
	public JcrTaskArchive(final JcrTemplate jcrTemplate, final RepositoryService repositoryService,
			final PojoConverter pojoConverter) {
		this.jcrTemplate = jcrTemplate;
		this.repositoryService = repositoryService;
		this.pojoConverter = pojoConverter;
	}

	@Override
	public void archiveTask(final Task task) {
		jcrTemplate.execute(new JcrCallback<String>() {
			@Override
			public String doInJcr(Session session) throws IOException, RepositoryException {
				Node tasksNode = getTasksNode(session);
				Node taskNode = tasksNode.addNode(task.getId());
				taskNode.addMixin(TASK_NODETYPE_NAME);
				archiveTaskInNode(session, task, taskNode);
				session.save();
				return null;
			}
		});
	}

	private void archiveTaskInNode(Session session, final Task task, Node taskNode) throws IOException,
			RepositoryException {
		taskNode.setProperty(ID_PROPERTY_NAME, task.getId());
		taskNode.setProperty(LABEL, task.getLabel());
		taskNode.setProperty(STATE, task.getState().name());
		taskNode.setProperty(START_DATE, task.getStartDate());
		taskNode.setProperty(COMPLETION_DATE, task.getCompletionDate());
		taskNode.setProperty(CURRENT_STEP_NR, task.getCurrentStepNr());
		taskNode.setProperty(NR_OF_STEPS, task.getNrOfSteps());
		taskNode.setProperty(OWNING_USER, task.getOwner());
		for (int nr = 1; nr <= task.getNrOfSteps(); nr++) {
			TaskStep step = task.getStep(nr);
			Node stepNode = taskNode.addNode(STEP_NODE_NAME_PREFIX + nr);
			stepNode.setProperty(DESCRIPTION, step.getDescription());
			stepNode.setProperty(STATE, step.getState().name());
			stepNode.setProperty(START_DATE, step.getStartDate());
			stepNode.setProperty(COMPLETION_DATE, step.getCompletionDate());
			stepNode.setProperty(LOG, step.getLog());
		}
	}

	@Override
	public Task getTask(final String taskId) {
		return jcrTemplate.execute(new JcrCallback<Task>() {
			@Override
			public Task doInJcr(Session session) throws IOException, RepositoryException {
				try {
					Node tasksNode = getTasksNode(session);
					Node taskNode = tasksNode.getNode(taskId);

					Task task = populateTaskFromNode(taskNode);
					if (task == null) {
						throw new NotFoundException("Cannot load task " + taskId
								+ " because that object with that id is not a task");
					}
					return task;
				} catch (PathNotFoundException exc) {
					throw new NotFoundException("Cannot load task " + taskId + ": " + exc.toString(), exc);
				}
			}
		});
	}

	@Override
	public List<Task> getAllTasks() {
		return jcrTemplate.execute(new JcrCallback<List<Task>>() {
			@Override
			public List<Task> doInJcr(Session session) throws IOException, RepositoryException {
				try {
					Node tasksNode = getTasksNode(session);
					NodeIterator nodes = tasksNode.getNodes();
					List<Task> allTasks = Lists.newArrayList();
					while (nodes.hasNext()) {
						Node taskNode = nodes.nextNode();
						Task task = populateTaskFromNode(taskNode);
						if (task != null) {
							allTasks.add(task);
						}
					}

					return allTasks;
				} catch (PathNotFoundException exc) {
					throw new NotFoundException("Cannot load all tasks" + ": " + exc.toString(), exc);
				}
			}
		});
	}

	private Task populateTaskFromNode(Node taskNode) throws RepositoryException, ValueFormatException,
			PathNotFoundException {
		if (!taskNode.isNodeType(TASK_NODETYPE_NAME)) {
			return null;
		}

		StepList stepList = createStepListFromNode(taskNode);
		final Task.State state = Task.State.valueOf(taskNode.getProperty(STATE).getString());
		Task task = new Task(stepList, state, repositoryService, pojoConverter, pojoConverter.getContext());
		task.setId(taskNode.getProperty(ID_PROPERTY_NAME).getString());
		task.setLabel(taskNode.getProperty(LABEL).getString());
		if (taskNode.hasProperty(START_DATE)) {
			task.setStartDate(taskNode.getProperty(START_DATE).getDate());
		}
		if (taskNode.hasProperty(COMPLETION_DATE)) {
			task.setCompletionDate(taskNode.getProperty(COMPLETION_DATE).getDate());
		}
		if (taskNode.hasProperty(OWNING_USER)) {
			task.setOwner(taskNode.getProperty(OWNING_USER).getString());
		}
		return task;
	}

	private StepList createStepListFromNode(final Node taskNode) throws RepositoryException {
		int currentStepNr = (int) taskNode.getProperty(CURRENT_STEP_NR).getLong();
		int nrOfSteps = (int) taskNode.getProperty(NR_OF_STEPS).getLong();
		List<TaskStep> steps = Lists.newArrayList();
		for (int nr = 1; nr <= nrOfSteps; nr++) {
			Node stepNode = taskNode.getNode(STEP_NODE_NAME_PREFIX + nr);
			steps.add(createStepFromNode(stepNode));
		}
		return new StepList(steps, currentStepNr);
	}

	private TaskStep createStepFromNode(Node stepNode) throws RepositoryException {
		final String description = stepNode.getProperty(DESCRIPTION).getString();
		final TaskStep.StepState stepState = TaskStep.StepState.valueOf(stepNode.getProperty(STATE).getString());
		TaskStep step = new TaskStep(description, stepState);
		if (stepNode.hasProperty(START_DATE)) {
			step.setStartDate(stepNode.getProperty(START_DATE).getDate());
		}
		if (stepNode.hasProperty(COMPLETION_DATE)) {
			step.setCompletionDate(stepNode.getProperty(COMPLETION_DATE).getDate());
		}
		step.setLog(stepNode.getProperty(LOG).getString());
		return step;
	}

	private Node getTasksNode(Session session) throws PathNotFoundException, RepositoryException {
		return session.getRootNode().getNode(TASKS_NODE_NAME);
	}

	@Override
	public Collection<Task> searchTasks(final ArchivedTaskSearchParameters archivedTaskSearchParameters) {
		return jcrTemplate.execute(new JcrCallback<List<Task>>() {
			@Override
			public List<Task> doInJcr(final Session session) throws IOException, RepositoryException {

				QueryManager qm = session.getWorkspace().getQueryManager();
				final ValueFactory valueFactory = session.getValueFactory();
				final Query query = qm.createQuery(JCR_SQL_QUERY, JCR_SQL2);
				
				addBinds(valueFactory, query, archivedTaskSearchParameters);
				final QueryResult queryResult = query.execute();
				Map<String, Task> filteredTasks = new LinkedHashMap<String, Task>();
				final RowIterator iterator = queryResult.getRows();
				while (iterator.hasNext()) {
					try {
						Node taskNode = iterator.nextRow().getNode("task");
						Task task = populateTaskFromNode(taskNode);
						filteredTasks.put(task.getId(), task);
					} catch (RepositoryException rre) {
						// Ignore, we weren't allowed to read a node, or one of
						// it's relations.
					}
				}
				return new ArrayList<Task>(filteredTasks.values());
			}

		});
	}
	
	
	private void addBinds(final ValueFactory valueFactory, final Query query, ArchivedTaskSearchParameters archivedTaskSearchParameters) throws IllegalArgumentException, RepositoryException {
		Value startDateValue = valueFactory.createValue((Calendar)archivedTaskSearchParameters.startDate);
		query.bindValue("_startDate", startDateValue);
		Value endDateValue = valueFactory.createValue((Calendar)archivedTaskSearchParameters.endDate);
		query.bindValue("_endDate", endDateValue);
	}


}
