package com.xebialabs.deployit.task;

import static com.google.common.base.Preconditions.checkNotNull;
import static com.google.common.base.Preconditions.checkState;
import static java.lang.String.format;

import java.util.List;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.FutureTask;

import javax.annotation.PreDestroy;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.core.task.TaskExecutor;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.context.SecurityContextHolder;
import org.springframework.stereotype.Component;

import com.xebialabs.deployit.security.PermissionDeniedException;
import com.xebialabs.deployit.security.UsernameAndPasswordCredentials;

/**
 */
@Component
public class ExecutionEngine {

	@Autowired
	private TaskRegistry taskRegistry;

	@Autowired
	private TaskExecutor taskExecutor;

	public ExecutionEngine() {
	}

	/**
	 * Registers a task with the engine.
	 * 
	 * @param task
	 *            the task to register
	 * @return the ID assigned to the task
	 */
	public String register(final Task task) {
		checkNotNull(task, "Should not pass in a null task");

		final UsernameAndPasswordCredentials creds = getCurrentCredentials();
		task.setOwner(creds.getUsername());

		return taskRegistry.registerTask(task);
	}

	/**
	 * Retrieves the task with the given id
	 * 
	 * @param id
	 *            the id of the task to retrieve
	 * @return the task or <tt>null</tt> if it cannot be retrieved.
	 */
	public Task getTask(final String id) {
		return taskRegistry.getTask(id);
	}


	private UsernameAndPasswordCredentials sanityCheckTask(final String id, final Task task, final String action) {
		checkState(task != null, "Cannot %s a task that is not registered with id [%s]", action, id);

		final UsernameAndPasswordCredentials creds = getCurrentCredentials();
		if (!creds.getUsername().equals(task.getOwner())) {
			throw new PermissionDeniedException(format("Cannot %s a task registered by another user", action));
		}
		return creds;
	}

	protected UsernameAndPasswordCredentials getCurrentCredentials() {
		final Authentication authentication = SecurityContextHolder.getContext().getAuthentication();
		checkNotNull(authentication, "No authentication present, are you logged in?");
		checkState(authentication instanceof UsernameAndPasswordCredentials, "Security credentials are not of the right type");
		return (UsernameAndPasswordCredentials) authentication;
	}

    /**
     * Starts the execution of the given task
     *
     * @param id
     *            the id of the task to start that was registered before using {@link #register(Task)}
     */
    public void execute(String id) {
        final Task task = taskRegistry.getTask(id);
        final UsernameAndPasswordCredentials creds = sanityCheckTask(id, task, "start");

        setCredentialsIfNotPresent(task, creds);

        // In order for a task to be cancellable and to obtain the Thread, we need to wrap it in a FutureTask
        final FutureTask<Object> future = new FutureTask<Object>(task, null);
        task.setWrappingTask(future);

        taskExecutor.execute(future);
    }

    private void setCredentialsIfNotPresent(final Task task, final UsernameAndPasswordCredentials creds) {
        if (task.getOwnerCredentials() == null) {
            task.setOwnerCredentials(creds);
        }
    }

    public void stopExecution(final String taskId) {
		final Task task = taskRegistry.getTask(taskId);
        final UsernameAndPasswordCredentials credentials = sanityCheckTask(taskId, task, "stop");
        setCredentialsIfNotPresent(task, credentials);
        task.stop();
	}

	public void abortExecution(final String taskId) {
		final Task task = taskRegistry.getTask(taskId);
        final UsernameAndPasswordCredentials credentials = sanityCheckTask(taskId, task, "abort");
        setCredentialsIfNotPresent(task, credentials);

		task.abort();
	}

    public void cancel(String taskId) {
        final Task task = taskRegistry.getTask(taskId);
        final UsernameAndPasswordCredentials credentials = sanityCheckTask(taskId, task, "cancel");
        setCredentialsIfNotPresent(task, credentials);
        taskRegistry.cancelTask(taskId);
    }


	public List<Task> getAllIncompleteTasksForCurrentUser() {
		return taskRegistry.getIncompleteTasksForUser(getCurrentCredentials().getUsername());
	}

	@PreDestroy
	public void shutdownTasks() {
		requestStopOnTasks();
		waitForTasksToBeStopped();
		taskRegistry.destroy();
	}

	private void waitForTasksToBeStopped() {
		for (Task task : taskRegistry.getTasks()) {
			if (task.isExecuting()) {
				try {
					logger.info("Waiting for task {} to stop", task.getId());
					final FutureTask<Object> wrappingTask = task.getWrappingTask();
					if (wrappingTask != null) {
						wrappingTask.get();
					}
				} catch (InterruptedException e) {
					Thread.currentThread().interrupt();
				} catch (ExecutionException e) {
					logger.error("Could not wait for task {} to be stopped...", task.getId());
				}
			}
		}
	}

	private void requestStopOnTasks() {
		for (Task task : taskRegistry.getTasks()) {
			if (task.isExecuting()) {
				task.stop();
			}
		}
	}

	private static final Logger logger = LoggerFactory.getLogger(ExecutionEngine.class);
}
