package com.xebialabs.deployit.task;

import static com.google.common.base.Preconditions.checkNotNull;
import static com.xebialabs.deployit.task.Task.State.CANCELLED;
import static com.xebialabs.deployit.task.Task.State.DONE;
import static com.xebialabs.deployit.task.Task.State.EXECUTING;
import static com.xebialabs.deployit.task.Task.State.PENDING;
import static com.xebialabs.deployit.task.Task.State.STOPPED;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Timer;
import java.util.TimerTask;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;

import javax.annotation.PostConstruct;

import org.apache.commons.io.FileUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.DirectFieldAccessor;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Required;

import com.google.common.base.Preconditions;
import com.google.common.collect.Lists;
import com.xebialabs.deployit.event.EventBus;
import com.xebialabs.deployit.event.EventCallback;
import com.xebialabs.deployit.plugin.PojoConverter;
import com.xebialabs.deployit.repository.RepositoryService;
import com.xebialabs.deployit.security.SecurityCallback;
import com.xebialabs.deployit.security.SecurityTemplate;
import com.xebialabs.deployit.task.Task.State;
import com.xebialabs.deployit.task.Task.TaskStateChangeEvent;

public class ArchivingTaskRegistry implements TaskRegistry {

	@Autowired
	private TaskArchive taskArchive;

	// needed to be set in the deserialized tasks
	@Autowired
	private RepositoryService repositoryService;

	// needed to be set in the deserialized tasks
	@Autowired
	private PojoConverter pojoConverter;

	private File recoveryFile;

	private int recoveryFileWritingIntervalMillis;

	private ConcurrentMap<String, Task> registry = new ConcurrentHashMap<String, Task>();

	private EventCallback<TaskStateChangeEvent> taskArchivingCallback;

	private Timer recoveryFileWritingTimer;

	@Required
	public void setRecoveryFile(File recoveryFile) {
		this.recoveryFile = recoveryFile;
	}

	@Required
	public void setRecoveryFileWritingIntervalMillis(int recoveryFileWritingIntervalMillis) {
		this.recoveryFileWritingIntervalMillis = recoveryFileWritingIntervalMillis;
	}

	@Override
	public String registerTask(final Task task) {
		final String uuid = UUID.randomUUID().toString();
		Preconditions.checkState(registry.putIfAbsent(uuid, task) == null);
		task.setId(uuid);
		return uuid;
	}

	@Override
	public Task getTask(final String id) {
		Task task = registry.get(id);
		if (task == null) {
			logger.info("Retrieving task {} from archive.", id);
			task = taskArchive.getTask(id);
		}

		return task;
	}

	@Override
	public void cancelTask(String id) {
		Task task = getTask(id);
		if (task.isReadyForExecution()) {
			if (task.getState().equals(STOPPED)) {
				moveTaskFromRegistryToArchive(task);
			}
			task.cancel();
			registry.remove(id);
		}
	}

	@Override
	public Collection<Task> getTasks() {
		return registry.values();
	}

	@Override
	public Collection<Task> getAllArchivedTasks() {
		return taskArchive.getAllTasks();
	}

	@Override
	public List<Task> getIncompleteTasksForUser(String username) {
		List<Task> myTasks = Lists.newArrayList();
		for (Task eachTask : registry.values()) {
			// although the registry only contains unfinished tasks, there can
			// be a time-window when the task is finished and is yet to be moved
			// to the task archive, so better check the status
			if ((eachTask.getState().equals(STOPPED) || eachTask.getState().equals(EXECUTING)) && eachTask.getOwner().equals(username)) {
				myTasks.add(eachTask);
			}
		}
		return myTasks;
	}

	@Override
	public void updateTask(final String id, final Task task) {
		Preconditions.checkState(registry.put(id, task) != null);
	}

	@PostConstruct
	public void afterPropertiesSet() {
		loadTasksFromRecoveryFile();
		registerTaskArchivingCallback();
		startRecoveryFileWritingTimer();
	}

	@SuppressWarnings("unchecked")
	private void loadTasksFromRecoveryFile() {
		if (recoveryFile != null && recoveryFile.exists()) {
			try {
				ObjectInputStream recoveryIn = new ObjectInputStream(new FileInputStream(recoveryFile));
				try {
					Object recoveryObject = recoveryIn.readObject();
					if (recoveryObject instanceof Map<?, ?>) {
						registry = (ConcurrentMap<String, Task>) recoveryObject;
						postprocessTaskRegistryAfterRecovery();
					}
				} finally {
					recoveryIn.close();
				}
			} catch (ClassNotFoundException exc) {
				logger.error("Cannot load tasks from recovery file " + recoveryFile, exc);				
			} catch (IOException exc) {
				logger.error("Cannot load tasks from recovery file " + recoveryFile, exc);
			}
		}
	}

	private void postprocessTaskRegistryAfterRecovery() {
		Iterator<Task> tasksIterator = registry.values().iterator();
		if (tasksIterator.hasNext()) {
			logger.info("Recovering tasks after server crash");
			while (tasksIterator.hasNext()) {
				Task nextTask = tasksIterator.next();
				State taskState = nextTask.getState();
				if (taskState == PENDING) {
					logger.info("Removing {} task {}", taskState, nextTask.getId());
					tasksIterator.remove();
				} else {
					logger.info("Recovering {} task {}", taskState, nextTask.getId());
					setDepedenciesInTask(nextTask);
					nextTask.processAfterRecovery();
				}
			}
			logger.info("Recovered tasks after server crash");
		}
	}

	private void setDepedenciesInTask(Task eachTask) {
		if (!eachTask.getState().equals(PENDING)) {
			DirectFieldAccessor directFieldAccessor = new DirectFieldAccessor(eachTask);
			directFieldAccessor.setPropertyValue("repositoryService", repositoryService);
			directFieldAccessor.setPropertyValue("pojoConverter", pojoConverter);
		}
	}

	@Override
	public void destroy() {
		stopRecoveryFileWritingTimer();
		// Make sure that everything is synced up once we're destroyed!
		writeRecoveryFile();
		deregisterTaskArchivingCallback();
	}

	private void registerTaskArchivingCallback() {
		taskArchivingCallback = new EventCallback<TaskStateChangeEvent>() {
			@Override
			public void receive(TaskStateChangeEvent event) {
				Task task = event.getTask();
				if (task.getState().equals(DONE)) {
					moveTaskFromRegistryToArchive(task);
				}
			}

		};
		EventBus.registerForEvent(TaskStateChangeEvent.class, taskArchivingCallback);
	}

	private void deregisterTaskArchivingCallback() {
		EventBus.deregisterForEvent(TaskStateChangeEvent.class, taskArchivingCallback);
	}

	private void startRecoveryFileWritingTimer() {
		recoveryFileWritingTimer = new Timer(ArchivingTaskRegistry.class.getName() + "#recoveryFileWritingTimer", true);
		recoveryFileWritingTimer.schedule(new TimerTask() {
			@Override
			public void run() {
				writeRecoveryFile();
			}
		}, recoveryFileWritingIntervalMillis, recoveryFileWritingIntervalMillis);
	}

	private void stopRecoveryFileWritingTimer() {
		recoveryFileWritingTimer.cancel();
	}

	synchronized void writeRecoveryFile() {
		if (recoveryFile != null) {
			File recoveryTemp = new File(recoveryFile.getParentFile(), "recovery.tmp");
			try {
				ObjectOutputStream recoveryOut = new ObjectOutputStream(new FileOutputStream(recoveryTemp));
				try {
					recoveryOut.writeObject(registry);
				} finally {
					recoveryOut.close();
				}
				FileUtils.deleteQuietly(recoveryFile);
				FileUtils.moveFile(recoveryTemp, recoveryFile);
			} catch (IOException exc) {
				logger.error("Cannot write task registry recovery file", exc);
			}
		}
	}

	private void moveTaskFromRegistryToArchive(final Task task) {
		checkNotNull(task.getOwnerCredentials(), "Cannot archive task " + task.getId() + " because it has no owner");
		SecurityTemplate.executeAs(task.getOwnerCredentials(), new SecurityCallback<Object>() {
			@Override
			public Object doAs() {
				taskArchive.archiveTask(task);
				registry.remove(task.getId());
				return null;
			}
		});
	}
	
	public TaskArchive getTaskArchive() {
		return taskArchive;
	}

	private static final Logger logger = LoggerFactory.getLogger(ArchivingTaskRegistry.class);

}
