package com.xebialabs.deployit.service.validation;

import com.google.common.base.Function;
import com.google.common.base.Predicate;
import com.google.common.collect.Collections2;
import com.xebialabs.deployit.reflect.ConfigurationItemDescriptor;
import com.xebialabs.deployit.reflect.ConfigurationItemPropertyDescriptor;
import com.xebialabs.deployit.repository.*;
import com.xebialabs.deployit.typedescriptor.ConfigurationItemDescriptorRepositoryHolder;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.util.*;

@Component
public class Validator {

    @Autowired
    private RepositoryService repositoryService;

    @Autowired
    private Flattener flattener;

    public Validations validate(RepositoryObjectEntity ci, RepositoryObjectEntity... cisInContext) {
        Validations validations = new Validations();
	    if (!validations.checkArchetypes(ci)) {
		    return validations;
	    }
        FlattenedEntity flattened = flattener.flatten(ci);
        String type = flattened.getConfigurationItemTypeName();
        final ConfigurationItemDescriptor descriptor = ConfigurationItemDescriptorRepositoryHolder.getDescriptor(type);
        for (ConfigurationItemPropertyDescriptor propertyDescriptor : descriptor.getPropertyDescriptors()) {

            if (!flattened.isArchetype() && propertyDescriptor.isRequired() && !propertyDescriptor.asContainment()) {
                validations.checkRequired(flattened, propertyDescriptor);
            }
        }

        for (String key : flattened.getValues().keySet()) {
            if (!isPlaceholderField(key, descriptor) && validations.checkFieldExists(flattened, descriptor, key)) {
                ConfigurationItemPropertyDescriptor propertyDescriptor = descriptor.getPropertyDescriptor(key);
                final Object value = flattened.getValue(propertyDescriptor);
                validations.checkType(flattened, propertyDescriptor, value, repositoryService, cisInContext);
            }
        }

        if (descriptor.getPlaceholdersName() != null) {
            final Object placeholdersObject = flattened.getValue(descriptor.getPlaceholdersName());
	        if (!validations.checkHasSource(flattened)) {
				return validations;
	        }
            if (placeholdersObject == null || (placeholdersObject instanceof Collection && ((Collection) placeholdersObject).isEmpty())) {
                String sourceId = (String) flattened.getValue("source");
                final RepositoryObjectEntity source = repositoryService.read(sourceId);
                final Object placeholdersFromSource = source.getValue("placeholders");
                if (placeholdersFromSource != null && !((Collection<String>) placeholdersFromSource).isEmpty()) {
                    validations.add(new ValidationMessage(flattened, descriptor.getPlaceholdersName(), "Missing all placeholders from source %s (%s)", sourceId, placeholdersFromSource));
                }
            } else if (validations.checkPlaceholdersOfCorrectType(flattened, descriptor, placeholdersObject)) {
                validations.checkPlaceholdersAllFilled(flattened, descriptor, placeholdersObject);
                validations.checkPlaceholdersAllPresent(flattened, descriptor, placeholdersObject, repositoryService);
            }
        }
        return validations;
    }


    private boolean isPlaceholderField(final String key, final ConfigurationItemDescriptor descriptor) {
         return key.equals(descriptor.getPlaceholdersName()) || key.equals(descriptor.getPlaceholderFormatName());
    }

    public static class Validations extends ArrayList<ValidationMessage> {

        private void checkRequired(final RepositoryObjectEntity ci, final ConfigurationItemPropertyDescriptor propertyDescriptor) {
            final Object value = ci.getValue(propertyDescriptor);
            if (value == null || StringUtils.isBlank(value.toString())) {
                add(new ValidationMessage(ci, propertyDescriptor, "Property %s is required", propertyDescriptor.getName()));
            }
        }

        public boolean hasMessages() {
            return !isEmpty();
        }

        private void checkInteger(final RepositoryObjectEntity ci, final Object value, final ConfigurationItemPropertyDescriptor propertyDescriptor) {
            if (value != null) {
                try {
                    Integer.parseInt(value.toString());
                } catch (NumberFormatException nfe) {
                    add(new ValidationMessage(ci, propertyDescriptor, "Could not convert %s with value [%s] into an int", propertyDescriptor.getName(), value));
                }
            }
        }

        private void checkBoolean(final RepositoryObjectEntity ci, final Object value, final ConfigurationItemPropertyDescriptor propertyDescriptor) {
            if (value != null) {
                if ("true".equalsIgnoreCase(value.toString()) || "false".equalsIgnoreCase(value.toString())) {
                    return;
                }
                add(new ValidationMessage(ci, propertyDescriptor, "Could not convert %s with value [%s] into a boolean", propertyDescriptor.getName(), value));
            }
        }

        private void checkEnum(final RepositoryObjectEntity ci, final Object value, final ConfigurationItemPropertyDescriptor propertyDescriptor) {
            if (value != null) {
                for (String enumValue : propertyDescriptor.getEnumValues()) {
                    if (enumValue.equals(value)) {
                        return;
                    }
                }
                add(new ValidationMessage(ci, propertyDescriptor, "Property %s with value [%s] does not contain correct enum value", propertyDescriptor.getName(), value));
            }
        }

        private boolean checkFieldExists(final RepositoryObjectEntity ci, final ConfigurationItemDescriptor descriptor, final String key) {
            ConfigurationItemPropertyDescriptor propertyDescriptor = descriptor.getPropertyDescriptor(key);
            if (propertyDescriptor == null || !propertyDescriptor.getName().equals(key)) {
                add(new ValidationMessage(ci, key, "Property %s does not exist on configuration item", key));
                return false;
            }

            return true;
        }

        private void checkCi(final RepositoryObjectEntity ci, final Object value, final ConfigurationItemPropertyDescriptor propertyDescriptor, final RepositoryService repositoryService, RepositoryObjectEntity[] cisInContext) {
            if (value == null || propertyDescriptor.asContainment()) return;
            if (!ciExists(value.toString(), repositoryService, cisInContext)) {
                add(new ValidationMessage(ci, propertyDescriptor, "Property %s with value [%s] does not point to an existing configuration item", propertyDescriptor.getName(), value));
            }
        }

        private boolean ciExists(String ciId, RepositoryService repositoryService, RepositoryObjectEntity[] cisInContext) {
            return repositoryService.checkNodeExists(ciId) || inCisContext(ciId, cisInContext);

        }

        private boolean inCisContext(String ciId, RepositoryObjectEntity[] cisInContext) {
            for (RepositoryObjectEntity entity : cisInContext) {
                if (entity.getId().equals(ciId)) {
                    return true;
                }
            }
            return false;
        }

        private void checkSetOfStrings(final RepositoryObjectEntity ci, final Object value, final ConfigurationItemPropertyDescriptor propertyDescriptor) {
            if (value == null) return;

            boolean valid = true;
            if (!(value instanceof Collection)) {
                valid = false;
            } else {
                for (Object o : (Collection) value) {
                    valid = valid && o instanceof String;
                }
            }

            if (!valid) {
                add(new ValidationMessage(ci, propertyDescriptor, "Property %s with value %s[%s] does not contain a set of strings", propertyDescriptor.getName(), value.getClass().getSimpleName(), value));
            }
        }

        private void checkSetOfCis(final RepositoryObjectEntity ci, final Object value, final ConfigurationItemPropertyDescriptor propertyDescriptor, final RepositoryService repositoryService, RepositoryObjectEntity[] cisInContext) {
            if (value == null || propertyDescriptor.asContainment()) return;

            boolean valid = true;
            // Checks are made on Collection, as RestEASY does not always give us Sets but also Lists.
            if (!(value instanceof Collection)) {
                valid = false;
            } else {
                for (Object o : (Collection) value) {
                    valid = valid && o instanceof String && ciExists((String) o, repositoryService, cisInContext);
                }
            }

            if (!valid) {
                add(new ValidationMessage(ci, propertyDescriptor, "Property %s with value %s[%s] does not contain a set of configuration items", propertyDescriptor.getName(), value.getClass().getSimpleName(), value));
            }
        }

        private void checkListOfObjects(final RepositoryObjectEntity ci, final Object value, final ConfigurationItemPropertyDescriptor propertyDescriptor, final RepositoryService repositoryService, RepositoryObjectEntity[] cisInContext) {

            if (value == null) return;

            boolean valid = true;
            if (!(value instanceof Collection)) {
                valid = false;
            } else {
                final ConfigurationItemPropertyDescriptor[] listObjectDescriptors = propertyDescriptor.getListObjectPropertyDescriptors();
                for (Object o : (Collection) value) {
                    valid = valid && o instanceof Map;
                    if (valid) {
                        Map<String, Object> map = (Map<String, Object>) o;
                        for (ConfigurationItemPropertyDescriptor lod : listObjectDescriptors) {
                            final Object objValue = map.get(lod.getName());
                            Validations v = new Validations();
                            v.checkType(ci, lod, objValue, repositoryService, cisInContext);
                            valid = valid && !v.hasMessages();
                            addAll(v);
                        }

                        for (String key : map.keySet()) {
                            boolean found = false;
                            for (ConfigurationItemPropertyDescriptor descriptor : listObjectDescriptors) {
                                if (descriptor.getName().equals(key)) {
                                    found = true;
                                    break;
                                }
                            }

                            if (!found) {
                                add(new ValidationMessage(ci, propertyDescriptor, "Property %s does not exist on the list of objects property %s", key, propertyDescriptor.getName()));
                                valid = false;
                            }
                        }
                    }
                }
            }

            if (!valid) {
                add(new ValidationMessage(ci, propertyDescriptor, "Property %s with value [%s] does not contain a valid list of objects", propertyDescriptor.getName(), value));
            }
        }

        private void checkType(final RepositoryObjectEntity ci, final ConfigurationItemPropertyDescriptor propertyDescriptor, final Object value, final RepositoryService repositoryService, RepositoryObjectEntity[] cisInContext) {
            switch (propertyDescriptor.getType()) {
                case BOOLEAN:
                    checkBoolean(ci, value, propertyDescriptor);
                    break;
                case INTEGER:
                    checkInteger(ci, value, propertyDescriptor);
                    break;
                case STRING:
                    // no-op ;-)
                    break;
                case ENUM:
                    checkEnum(ci, value, propertyDescriptor);
                    break;
                case LIST_OF_OBJECTS:
                    checkListOfObjects(ci, value, propertyDescriptor, repositoryService, cisInContext);
                    break;
                case SET_OF_STRINGS:
                    checkSetOfStrings(ci, value, propertyDescriptor);
                    break;
                case CI:
                    checkCi(ci, value, propertyDescriptor, repositoryService, cisInContext);
                    break;
                case SET_OF_CIS:
                    checkSetOfCis(ci, value, propertyDescriptor, repositoryService, cisInContext);
                    break;
                case UNSUPPORTED:
                    break;
            }
        }

        private void checkPlaceholdersAllFilled(final FlattenedEntity flattened, final ConfigurationItemDescriptor   descriptor, final Object placeholdersObject) {
            List<Map<String, String>> placeholders = (List<Map<String, String>>) placeholdersObject;
            for (Map<String, String> placeholder : placeholders) {
                if (StringUtils.isBlank(placeholder.get("value"))) {
                    add(new ValidationMessage(flattened, descriptor.getPlaceholdersName(), "Placeholder [%s] is not mapped to any value", placeholder.get("key")));
                }
            }
        }

        public void checkPlaceholdersAllPresent(final FlattenedEntity flattened, final ConfigurationItemDescriptor descriptor, final Object placeholdersObject, final RepositoryService repositoryService) {
            List<Map<String, String>> placeholders = (List<Map<String, String>>) placeholdersObject;
            final String sourceId = (String) flattened.getValue("source");
            final RepositoryObjectEntity source = repositoryService.read(sourceId);
            final Set<String> placeholderSet = (Set<String>) source.getValue("placeholders");

	        // DEPLOYITPB-1430
	        if (placeholderSet == null) {
		        return;
	        }
            final Collection<String> transformed = Collections2.transform(placeholders, new Function<Map<String, String>, String>() {
                @Override
                public String apply(final Map<String, String> from) {
                    return from.get("key");
                }
            });

            for (String placeholder : placeholderSet) {
                if (!transformed.contains(placeholder)) {
                    add(new ValidationMessage(flattened, descriptor.getPlaceholdersName(), "Placeholders is missing key [%s]", placeholder));
                }
            }

            for (Map<String, String> placeholder : placeholders) {
                if (!placeholderSet.contains(placeholder.get("key"))) {
                    add(new ValidationMessage(flattened, descriptor.getPlaceholdersName(), "Placeholders has key [%s] which is not present in the source", placeholder));
                }
            }
        }

        private boolean checkPlaceholdersOfCorrectType(final FlattenedEntity flattened, final ConfigurationItemDescriptor descriptor, final Object placeholdersObject) {
            if (!(placeholdersObject instanceof Collection)) {
                add(new ValidationMessage(flattened, descriptor.getPlaceholdersName(), "Placeholders in field [%s] are not of correct type [%s]", descriptor.getPlaceholdersName(), placeholdersObject.getClass().getName()));
                return false;
            }

            if (Collections2.filter((Collection<Object>) placeholdersObject, new Predicate<Object>() {
                @Override
                public boolean apply(final Object input) {
                    return !(input instanceof Map);
                }
            }).size() > 0) {
                add(new ValidationMessage(flattened, descriptor.getPlaceholdersName(), "Placeholders in field [%s] are not of correct type.", descriptor.getPlaceholdersName()));
                return false;
            }
            return true;
        }

	    public boolean checkArchetypes(RepositoryObjectEntity ci) {
		    if (ci.getConfigurationItemArchetype() != null) {
			    final String archetypeType = ci.getConfigurationItemArchetype().getConfigurationItemTypeName();
			    if (!archetypeType.equals(ci.getConfigurationItemTypeName())) {
				    add(new ValidationMessage(ci, "configurationItemArchetype", "The archetype is not of type %s, but of %s", ci.getConfigurationItemTypeName(), archetypeType));
				    return false;
			    }
		    }

		    return true;
	    }

	    public boolean checkHasSource(FlattenedEntity flattened) {
		    return flattened.getValue("source") != null;
	    }
    }


}
