package com.xebialabs.deployit.service.importer;

import com.google.common.base.Function;
import com.google.common.collect.Collections2;
import com.xebialabs.deployit.ci.Application;
import com.xebialabs.deployit.ci.DeploymentPackage;
import com.xebialabs.deployit.ci.MiddlewareResource;
import com.xebialabs.deployit.exception.RuntimeIOException;
import com.xebialabs.deployit.reflect.ConfigurationItemDescriptor;
import com.xebialabs.deployit.reflect.ConfigurationItemPropertyDescriptor;
import com.xebialabs.deployit.repository.ArtifactEntity;
import com.xebialabs.deployit.repository.ConfigurationItemEntity;
import com.xebialabs.deployit.typedescriptor.ConfigurationItemDescriptorRepositoryHolder;
import com.xebialabs.deployit.typedescriptor.ConfigurationItemTypeDescriptorRepository;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.*;
import java.util.Map.Entry;
import java.util.jar.Attributes;
import java.util.jar.Manifest;

import static com.google.common.collect.Lists.newArrayList;
import static com.google.common.collect.Maps.newHashMap;
import static com.google.common.collect.Maps.newLinkedHashMap;
import static com.google.common.collect.Sets.newHashSet;
import static com.xebialabs.deployit.ConfigurationItemRoot.APPLICATIONS;

public class UnpackedPackageManifestImporter {

	public static final String PACKAGE_FORMAT_VERSION_ATTRIBUTE_NAME = "Deployit-Package-Format-Version";
	public static final String PACKAGE_FORMAT_VERSION_NUMBER = "1.2";
	public static final String APPLICATION_ATTRIBUTE_NAME = "CI-Application";
	public static final String VERSION_ATTRIBUTE_NAME = "CI-Version";
	public static final String TYPE_ATTRIBUTE_NAME = "CI-Type";
	public static final String NAME_ATTRIBUTE_NAME = "CI-Name";
	public static final String NAMESPACE_SEPARATOR = "/";
	private static final String CI = "CI-";

	public static Manifest readManifest(File manifestFile) {
		Manifest manifest;
		InputStream in = null;
		try {
			in = new FileInputStream(manifestFile);
			manifest = new Manifest(in);
		} catch (IOException exception) {
			throw new RuntimeIOException("Cannot read manifest file " + manifestFile, exception);
		} finally {
			IOUtils.closeQuietly(in);
		}
		return manifest;
	}

	public static ConfigurationItemEntity createApplicationEntity(Manifest manifest) {
		Attributes mainAttributes = manifest.getMainAttributes();

		verifyPackageFormatVersion(mainAttributes);

		ConfigurationItemEntity applicationEntity = new ConfigurationItemEntity(Application.class.getName());
		applicationEntity.setId(APPLICATIONS, getApplicationName(manifest));
		return applicationEntity;
	}

	public static ConfigurationItemEntity createDeploymentPackageEntity(Manifest manifest, ConfigurationItemEntity applicationEntity) {

		Attributes mainAttributes = manifest.getMainAttributes();

		verifyPackageFormatVersion(mainAttributes);

		String packageVersion = getApplicationVersion(manifest);

		ConfigurationItemEntity deploymentPackageEntity = new ConfigurationItemEntity(DeploymentPackage.class.getName());
		deploymentPackageEntity.setId(applicationEntity.getId(), packageVersion);
		deploymentPackageEntity.addValue("version", packageVersion);

		return deploymentPackageEntity;
	}

	protected static String getApplicationName(Manifest manifest) {
		return getAttribute(manifest, APPLICATION_ATTRIBUTE_NAME);
	}

	protected static String getApplicationVersion(Manifest manifest) {
		return getAttribute(manifest, VERSION_ATTRIBUTE_NAME);
	}

	private static String getAttribute(final Manifest manifest, final String attributeName) {
		String attributeValue = StringUtils.trim(manifest.getMainAttributes().getValue(attributeName));
		if (StringUtils.isBlank(attributeValue)) {
			throw new ImporterException("Deployment package does not declare the " + attributeName + " main attribute");
		}
		return attributeValue;
	}

	protected static void verifyPackageFormatVersion(Attributes mainAttributes) {
		String packageFormatVersion = StringUtils.trim(mainAttributes.getValue(PACKAGE_FORMAT_VERSION_ATTRIBUTE_NAME));
		if (packageFormatVersion == null) {
			packageFormatVersion = "unspecified";
		}
		if (!PACKAGE_FORMAT_VERSION_NUMBER.equals(packageFormatVersion)) {
			throw new ImporterException("Deployment package does not have the right " + PACKAGE_FORMAT_VERSION_ATTRIBUTE_NAME + " header value (actual: "
			        + packageFormatVersion + " expected: " + PACKAGE_FORMAT_VERSION_NUMBER + ")");
		}
	}

	public static boolean isMiddlewareResource(Entry<String, Attributes> entry) {
		String configurationItemTypeShortName = StringUtils.trim(entry.getValue().getValue(TYPE_ATTRIBUTE_NAME));
		String entryName = StringUtils.trim(entry.getKey()); // Is Name: path-relative-to-package-folder/SomeEarFileName.ear

		if (configurationItemTypeShortName == null) {
			return false;
		}

		ConfigurationItemDescriptor configurationItemDescriptor = getDescriptorForConfigurationItemShortName(entryName, configurationItemTypeShortName);

		return configurationItemDescriptor.getInterfaces().contains(MiddlewareResource.class.getName());
	}

	public static ArtifactEntity createArtifactEntity(ConfigurationItemEntity configurationItem, File packageToImportFile, Entry<String, Attributes> entry,
	        List<DeployableArtifactImporter> artifactImporters) {
		String entryName = StringUtils.trim(entry.getKey()); // Is Name: path-relative-to-package-folder/SomeEarFileName.ear
		Attributes entryAttributes = entry.getValue();

		String configurationItemTypeShortName = readCIShortType(entryAttributes, entryName);

		ConfigurationItemDescriptor configurationItemDescriptor = getDescriptorForConfigurationItemShortName(entryName, configurationItemTypeShortName);

		ArtifactEntity artifact = new ArtifactEntity(configurationItemDescriptor.getType());
		final String baseId = configurationItem.getId();
		artifact.setId(baseId, extractArtifactNameFromPath(entryName));
		// Location is required but we do not have a physical location, using a fake jcr location.
		artifact.addValue("location", "jcr://" + artifact.getId());
		File artifactFile = new File(packageToImportFile, entryName);
		for(DeployableArtifactImporter importer : artifactImporters) {
			if(importer.canHandle(configurationItemDescriptor)) {
				importer.importArtifact(artifact, artifactFile, configurationItemDescriptor);
			}
		}
		fillAttributes(artifact, configurationItemDescriptor, entryAttributes, baseId);
		return artifact;
	}

    private static String extractArtifactNameFromPath(String path) {
        int index = path.lastIndexOf(NAMESPACE_SEPARATOR);
        if (index > -1) {
            index++;
            if (index >= path.length()) {
               throw new ImporterException("Could not determine artifact Name: %s because entry ends with '/'", path);
            }
            return path.substring(index);
        }
        return path;

    }

	private static String readCIShortType(final Attributes entryAttributes, final String entryName) {
		String configurationItemTypeShortName = StringUtils.trim(entryAttributes.getValue(TYPE_ATTRIBUTE_NAME));
		if (configurationItemTypeShortName == null) {
			throw new ImporterException("Could not import Name: %s because of missing %s entry in MANIFEST.MF", entryName, TYPE_ATTRIBUTE_NAME);
		}
		return configurationItemTypeShortName;
	}

	private static ConfigurationItemDescriptor getDescriptorForConfigurationItemShortName(String entryName, String configurationItemTypeShortName) {
		ConfigurationItemTypeDescriptorRepository typeDescriptorRepository = ConfigurationItemDescriptorRepositoryHolder.getDescriptorRepository();
		ConfigurationItemDescriptor configurationItemDescriptor = typeDescriptorRepository.getDescriptorByShortName(configurationItemTypeShortName);
		if (configurationItemDescriptor == null) {
			throw new ImporterException("Could not import Name: %s because %s is not a known CI type", entryName, configurationItemTypeShortName);
		}
		return configurationItemDescriptor;
	}

	public static ConfigurationItemEntity createMiddlewareResourceEntity(final ConfigurationItemEntity deploymentPackage, final File explodedPackage,
	        final Entry<String, Attributes> entry) {
		String entryName = StringUtils.trim(entry.getKey());
		Attributes entryAttributes = entry.getValue();

		String configurationItemTypeShortName = readCIShortType(entryAttributes, entryName);

		ConfigurationItemDescriptor configurationItemDescriptor = getDescriptorForConfigurationItemShortName(entryName, configurationItemTypeShortName);

		final ConfigurationItemEntity resource = new ConfigurationItemEntity(configurationItemDescriptor.getType());
		final String baseId = deploymentPackage.getId();
		resource.setId(baseId, extractArtifactNameFromPath(entryName));
		fillAttributes(resource, configurationItemDescriptor, entryAttributes, baseId);
		return resource;
	}

	private static void fillAttributes(final ConfigurationItemEntity entity, final ConfigurationItemDescriptor configurationItemDescriptor,
	                                   final Attributes entryAttributes, final String baseId) {
		for (ConfigurationItemPropertyDescriptor propertyDescriptor : configurationItemDescriptor.getPropertyDescriptors()) {
			final String name = propertyDescriptor.getName();
			switch (propertyDescriptor.getType()) {
			case BOOLEAN:
			case INTEGER:
			case STRING:
			case ENUM:
				final String value = entryAttributes.getValue(CI + name);
				if (value != null) {
					entity.addValue(name, value);
				}
				break;
			case CI:
				final String ci = entryAttributes.getValue(CI + name);
				if (ci != null) {
					entity.addValue(name, baseId + NAMESPACE_SEPARATOR + ci);
				}
				break;
			case LIST_OF_OBJECTS:
				List<Map<String, String>> objects = handleListOfObjects(propertyDescriptor, entryAttributes);
				if (!objects.isEmpty()) {
					entity.addValue(name, objects);
				}
				break;
			case SET_OF_STRINGS:
				Set<String> strings = handleSetOfStrings(propertyDescriptor, entryAttributes);
				if (!strings.isEmpty()) {
					entity.addValue(name, strings);
				}
				break;
			case SET_OF_CIS:
				Set<String> cis = new HashSet<String>(Collections2.transform(handleSetOfStrings(propertyDescriptor, entryAttributes), new Function<String, String>() {
					@Override
					public String apply(String from) {
						return baseId + NAMESPACE_SEPARATOR + from;
					}
				}));
				if (!cis.isEmpty()) {
					entity.addValue(name, cis);
				}
				break;
			}
		}
	}

	private static Set<String> handleSetOfStrings(ConfigurationItemPropertyDescriptor propertyDescriptor, Attributes entryAttributes) {
		String key = CI + propertyDescriptor.getName() + "-EntryValue";
		final Set<String> strings = newHashSet();
		for (Entry<Object, Object> entry : entryAttributes.entrySet()) {
			if (entry.getKey().toString().trim().startsWith(key)) {
				strings.add(entry.getValue().toString().trim());
			}
		}

		return strings;
	}

	private static List<Map<String, String>> handleListOfObjects(final ConfigurationItemPropertyDescriptor propertyDescriptor, final Attributes entryAttributes) {
		final String name = CI + propertyDescriptor.getName() + "-Entry";
		List<Map<String, String>> objects = newArrayList();
		Map<String, Map<String, String>> keyValues = gatherObjectEntriesFromAttributes(entryAttributes, name);

		for (String key : new TreeSet<String>(keyValues.keySet())) {
			Map<String, String> values = keyValues.get(key);
			Map<String, String> object = newHashMap();
			final ConfigurationItemPropertyDescriptor[] listObjectDescriptors = propertyDescriptor.getListObjectPropertyDescriptors();
			for (ConfigurationItemPropertyDescriptor listObjectDescriptor : listObjectDescriptors) {
				final String descriptorName = listObjectDescriptor.getName();
				if (values.get(descriptorName) != null) {
					object.put(descriptorName, values.get(descriptorName));
				}
			}
			if (object.size() > 0) {
				objects.add(object);
			}
		}
		return objects;
	}

	private static Map<String, Map<String, String>> gatherObjectEntriesFromAttributes(final Attributes entryAttributes, final String name) {
		Map<String, Map<String, String>> keyValues = newLinkedHashMap();
		for (Entry<Object, Object> entry : entryAttributes.entrySet()) {
			final String key = entry.getKey().toString().trim();
			if (key.startsWith(name) && entryAttributes.getValue(key) != null) {
				// key is of the form: name-Entry<Property>-<Number>
				String entryKey = key.substring(name.length());
				// entryKey is now of the form: <Property>-<Number>
				String entryNumber = entryKey.substring(entryKey.indexOf('-') + 1);
				entryKey = StringUtils.uncapitalize(entryKey.substring(0, entryKey.indexOf('-')));
				// entryKey is now of the form: <property>

				Map<String, String> properties = keyValues.get(entryNumber);
				if (properties == null) {
					properties = newHashMap();
					keyValues.put(entryNumber, properties);
				}

				final String entryValue = ((String) entry.getValue()).trim();
				properties.put(entryKey, entryValue);
			}
		}
		return keyValues;
	}

	private static final Logger logger = LoggerFactory.getLogger(UnpackedPackageManifestImporter.class);
}
