package com.xebialabs.deployit.service.importer;

import com.google.common.collect.Lists;
import com.xebialabs.deployit.exception.DeployitException;
import com.xebialabs.deployit.exception.HttpResponseCodeResult;
import com.xebialabs.deployit.exception.NotFoundException;
import com.xebialabs.deployit.exception.RuntimeIOException;
import com.xebialabs.deployit.repository.ArtifactEntity;
import com.xebialabs.deployit.repository.ConfigurationItemEntity;
import com.xebialabs.deployit.repository.RepositoryService;
import com.xebialabs.deployit.service.validation.ValidationMessage;
import com.xebialabs.deployit.service.validation.Validator;
import org.apache.commons.httpclient.HttpClient;
import org.apache.commons.httpclient.HttpException;
import org.apache.commons.httpclient.HttpMethod;
import org.apache.commons.httpclient.HttpStatus;
import org.apache.commons.httpclient.methods.GetMethod;
import org.apache.commons.io.FileUtils;
import org.apache.commons.io.IOUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import java.io.*;
import java.net.URISyntaxException;
import java.net.URL;
import java.util.List;
import java.util.Map;
import java.util.jar.Attributes;
import java.util.jar.Manifest;

import static com.google.common.collect.Iterables.toArray;
import static com.google.common.collect.Lists.newArrayList;
import static com.xebialabs.deployit.service.importer.Exploder.explode;
import static java.util.Collections.sort;

public class ImporterServiceImpl implements ImporterService {

    @Autowired
    private RepositoryService repositoryService;

    @Autowired
    private Validator validator;

    private File importablePackageDirectory;

    private List<DeployableArtifactImporter> artifactImporters = Lists.newArrayList();

    public void setImportablePackageDirectory(final File importablePackageDirectory) {
        this.importablePackageDirectory = importablePackageDirectory;
    }

    public void setArtifactImporters(List<DeployableArtifactImporter> artifactImporters) {
        this.artifactImporters = artifactImporters;
    }

    @Override
    public List<String> list() {
        List<String> newArrayList = newArrayList(PackageScanner.scan(importablePackageDirectory));
        sort(newArrayList);
        return newArrayList;
    }

    @Override
    public PackageInfo preparePackage(URL url) {
        try {
            logger.debug("Preparing to download package from {}", url);
            HttpClient client = new HttpClient();
            HttpMethod method = new GetMethod(url.toURI().toString());
            int statusCode = client.executeMethod(method);
            if (statusCode != HttpStatus.SC_OK) {
                throw new RuntimeIOException("Failed to download package,status="+ statusCode +", from url " + url);
            }
           return preparePackage(method.getResponseBodyAsStream());
        } catch (URISyntaxException e) {
            throw new IllegalArgumentException("Invalid URL", e);
        } catch (HttpException e) {
            throw new RuntimeIOException(e);
        } catch (IOException e) {
            throw new RuntimeIOException(e);
        }
    }

    @Override
    public PackageInfo preparePackage(final String file) {
        final PackageInfo packageInfo = new PackageInfo();
        packageInfo.originalPackage = file;

        File packageToImport = new File(importablePackageDirectory, file);

        if (PackageScanner.isExplodedPackage(packageToImport)) {
            logger.debug("Preparing unpacked package {}", packageToImport);
            packageInfo.explodedPackage = packageToImport;
        } else if (PackageScanner.isDarPackage(packageToImport)) {
            packageInfo.explodedPackage = explode(packageToImport);
        } else {
            throw new NotFoundException("Cannot import " + file + " because it does not exist or is not a .dar or unpacked package");
        }

        updatePackageInfoWithManifestValues(packageInfo);

        return packageInfo;
    }

    private void updatePackageInfoWithManifestValues(final PackageInfo packageInfo) {
        File manifestFileForUnpackedPackage = PackageScanner.getManifestFileForUnpackedPackage(packageInfo.explodedPackage);
        Manifest manifest = UnpackedPackageManifestImporter.readManifest(manifestFileForUnpackedPackage);

        packageInfo.applicationName = UnpackedPackageManifestImporter.getApplicationName(manifest);
        packageInfo.applicationVersion = UnpackedPackageManifestImporter.getApplicationVersion(manifest);
    }

    @Override
    public PackageInfo preparePackage(final byte[] uploadedPackage) {
        return preparePackage(new ByteArrayInputStream(uploadedPackage));
    }

    private PackageInfo preparePackage(final InputStream uploadedPackageStream) {
        PackageInfo packageInfo = new PackageInfo();
        try {
            File archive = File.createTempFile("uploaded-package", ".dar");
            IOUtils.copy(uploadedPackageStream, new FileOutputStream(archive));
            if (PackageScanner.isDarPackage(archive)) {
                logger.debug("Preparing uploaded packed package {}", archive);
                packageInfo.originalPackage = archive.getAbsolutePath();
                packageInfo.explodedPackage = explode(archive);
                updatePackageInfoWithManifestValues(packageInfo);
            }

        } catch (Exception e) {
            throw new IllegalArgumentException("Cannot import uploaded package because it does not exist or is not a .dar or unpacked package", e);
        } finally {
            IOUtils.closeQuietly(uploadedPackageStream);
        }
        return packageInfo;
    }

    @Override
    public String importPackage(final PackageInfo packageInfo) {
        File manifestFileForUnpackedPackage = PackageScanner.getManifestFileForUnpackedPackage(packageInfo.explodedPackage);
        Manifest manifest = UnpackedPackageManifestImporter.readManifest(manifestFileForUnpackedPackage);

        List<ConfigurationItemEntity> toCreate = newArrayList();

        ConfigurationItemEntity applicationEntity = UnpackedPackageManifestImporter.createApplicationEntity(manifest);
        if (repositoryService.checkNodeExists(applicationEntity.getId())) {
            applicationEntity = repositoryService.read(applicationEntity.getId());
        } else {
            toCreate.add(applicationEntity);
        }

        ConfigurationItemEntity deploymentPackage = UnpackedPackageManifestImporter.createDeploymentPackageEntity(manifest, applicationEntity);
        if (repositoryService.checkNodeExists(deploymentPackage.getId())) {
            throw new ImportFailedException("The package %s is already imported.", deploymentPackage.getId());
        }
        toCreate.add(deploymentPackage);

        for (Map.Entry<String, Attributes> entry : manifest.getEntries().entrySet()) {
            if (UnpackedPackageManifestImporter.isMiddlewareResource(entry)) {
                ConfigurationItemEntity resource = UnpackedPackageManifestImporter.createMiddlewareResourceEntity(deploymentPackage, packageInfo.explodedPackage, entry);
                toCreate.add(resource);
            } else {
                ArtifactEntity artifact = UnpackedPackageManifestImporter.createArtifactEntity(deploymentPackage, packageInfo.explodedPackage, entry, this.artifactImporters);
                toCreate.add(artifact);
            }
        }

        final ConfigurationItemEntity[] toCreateArray = toArray(toCreate, ConfigurationItemEntity.class);
        List<ValidationMessage> messages = newArrayList();
        for (ConfigurationItemEntity entity : toCreate) {
            final Validator.Validations validations = validator.validate(entity, toCreateArray);
            messages.addAll(validations);
        }

        if (!messages.isEmpty()) {
            throw new ImportFailedException("Import failed with the following validation errors %s", messages.toString());
        }

        repositoryService.create(toCreateArray);

        return deploymentPackage.getId();
    }

    @Override
    public void cleanUp(final PackageInfo packageInfo) {
        if (packageInfo.originalPackage.endsWith("dar")) {
            FileUtils.deleteQuietly(packageInfo.explodedPackage);
        }
    }

    @SuppressWarnings("serial")
    @HttpResponseCodeResult(statusCode = 400)
    class ImportFailedException extends DeployitException {
        public ImportFailedException(String template, Object... args) {
            super(template, args);
        }
    }

    private static final Logger logger = LoggerFactory.getLogger(ImporterServiceImpl.class);

}
