package com.xebialabs.deployit.service.importer;

import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FilenameFilter;
import java.io.IOException;
import java.util.Set;

import org.apache.commons.lang.ArrayUtils;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.collect.Sets;
import com.xebialabs.deployit.ci.artifact.Folder;
import com.xebialabs.deployit.reflect.ConfigurationItemDescriptor;
import com.xebialabs.deployit.repository.ArtifactEntity;

public class FolderTypeArtifactImporter extends DeployableArtifactImporter {

	private String[] ALLOWED_FILE_EXTENSIONS = { ".xml", ".properties", ".txt", ".cmd", ".bat", ".sh", ".sql" };

	@Override
	public boolean canHandle(ConfigurationItemDescriptor typeDescriptor) {
		return typeDescriptor.getSuperClasses().contains(Folder.class.getName());
	}

	public void doImport(final ArtifactEntity artifact, final File artifactFolder) {
		byte[] bytes;
		try {
			bytes = Imploder.implode(artifactFolder);
		} catch (IOException e) {
			throw new ImporterException(e, "Could not import folder Artifact: %s", artifactFolder);
		}

		artifact.setData(new ByteArrayInputStream(bytes));
	}

	protected Set<String> parsePlaceholders(final ArtifactEntity artifact, final File toBeParsedFile) {
		Set<String> placeholders = Sets.newHashSet();
		parseFilesForPlaceholderRecursively(artifact, toBeParsedFile, placeholders);
		return placeholders;
	}

	/**
	 * Parses a directory recursively to look for placeholders in the files(only text files as of now!)
	 */
	private void parseFilesForPlaceholderRecursively(ArtifactEntity artifact, File toBeParsedFile, Set<String> placeholders) {
		for (File eachFile : getValidFiles(toBeParsedFile)) {
			try {
				if (eachFile.isDirectory()) {
					parseFilesForPlaceholderRecursively(artifact, eachFile, placeholders);
				} else {
					placeholders.addAll(PlaceHolderHelper.parsePlaceholders(new FileInputStream(eachFile)));
				}
			} catch (FileNotFoundException e) {
				logger.warn("Problems found while parsing placeholders in file:" + eachFile.getName());
				logger.warn(e.getMessage());
			}
		}
	}

	protected File[] getValidFiles(File toBeParsedFile) {
		FilenameFilter filter = new FilenameFilter() {
			@Override
			public boolean accept(File dir, String fileName) {
				int lastIndexOfDot = StringUtils.lastIndexOf(fileName, ".");
				if (lastIndexOfDot == -1) {
					File f = new File(dir, fileName);
					return f.isDirectory();
				} else {
					String extension = fileName.substring(lastIndexOfDot);
					return ArrayUtils.contains(ALLOWED_FILE_EXTENSIONS, extension);
				}
			}
		};
		return toBeParsedFile.listFiles(filter);
	}

	private static final Logger logger = LoggerFactory.getLogger(FolderTypeArtifactImporter.class);

}
