package com.xebialabs.deployit.service.importer;

import static org.springframework.util.Assert.isTrue;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.zip.ZipEntry;
import java.util.zip.ZipInputStream;

import org.apache.commons.io.FileUtils;
import org.apache.commons.io.IOUtils;

/**
 */
public class Exploder {

	/**
	 * Extract a ZIP/DAR file.<br/>
	 * <b>N.B.:</b> You should cleanup after yourself, the temp directory won't be automatically deleted after this call.
	 * 
	 * @param zip
	 *            the zip file to extract
	 * @return the temp directory where the zip file is extracted
	 */
	public static File explode(File zip) {
		try {
			FileInputStream zipIn = new FileInputStream(zip);
			try {
				return explode(zipIn);
			} finally {
				zipIn.close();
			}
		} catch (IOException exc) {
			throw new ImporterException(exc, "Cannot import darFile: %s", zip.getName());
		}
	}

	/**
	 * Extracts a ZIP/DAR input stream.<br/>
	 * <b>N.B.:</b> You should cleanup after yourself, the temp directory won't be automatically deleted after this call.
	 * 
	 * @param zipIn
	 *            the input stream with the ZIP/DAR.
	 * @return the temp directory where the zip file is extracted
	 * @throws IOException
	 *             if an I/O exception occured while extracting.
	 */
	public static File explode(InputStream zipIn) throws IOException {
		File tempDir = null;
		try {
			tempDir = createTempDir();

			final ZipInputStream zipEntryStream = new ZipInputStream(zipIn);

			for (;;) {
				ZipEntry entry = zipEntryStream.getNextEntry();
				if (entry == null) {
					return tempDir;
				}

				try {

					if (entry.isDirectory()) {
						final File file = new File(tempDir, entry.getName());
						if(!file.exists()){
							isTrue(file.mkdirs(), "Could not create directory: " + entry.getName());
						}
						continue;
					}

					final File destinationFile = new File(tempDir, entry.getName());
                    final String parentDirPath = destinationFile.getParent();
                    if (parentDirPath != null) {
                        final File destinationDir = new File(parentDirPath);
                        if (!destinationDir.exists()) {
                            isTrue(destinationDir.mkdirs(), "Could not create directory: " + entry.getName());
                        }
					}
                    isTrue(destinationFile.createNewFile(), "Could not create file: " + entry.getName());
					IOUtils.copy(zipEntryStream, new FileOutputStream(destinationFile));
				} finally {
					zipEntryStream.closeEntry();
				}
			}
		} catch (IOException exc) {
			// On exception, clean up!
			FileUtils.deleteQuietly(tempDir);
			throw exc;
		}
	}

	/**
	 * Ugly way to create temporary directory because Java only creates temporary files...
	 * 
	 * @return the created temporary directory
	 * @throws IOException
	 *             if we could not create a temporary directory
	 */
	private static File createTempDir() throws IOException {
		File tempDir = File.createTempFile("explodedDar", "deployit");
		isTrue(tempDir.delete(), "Could not delete temporary file");
		isTrue(tempDir.mkdir(), "Could not create temporary directory");
		return tempDir;
	}

}
