package com.xebialabs.deployit.service.discovery;

import static com.google.common.collect.Lists.newArrayList;
import static com.xebialabs.deployit.checks.Checks.checkArgument;

import java.io.Serializable;
import java.util.Collection;
import java.util.Collections;
import java.util.List;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.xebialabs.deployit.ConfigurationItem;
import com.xebialabs.deployit.Discoverable;
import com.xebialabs.deployit.plugin.PojoConverter;
import com.xebialabs.deployit.repository.ConfigurationItemEntity;
import com.xebialabs.deployit.task.TaskExecutionContext;
import com.xebialabs.deployit.translation.DiscoveryChangePlan;
import com.xebialabs.deployit.translation.DiscoveryIdGenerator;

@Component
public class DiscoveryService {

	@Autowired
	private PojoConverter pojoConverter;

	@Autowired
	private DiscoveryIdGenerator labelGenerator;

	public ConfigurationItemEntity fetchConfiguration(final ConfigurationItemEntity middlewareEntity) {
		PojoConverter.Context pojoConverterContext = pojoConverter.getContext();
		try {
			TaskExecutionContext tec = new TaskExecutionContext(Maps.<String, Object> newHashMap());
			try {
				Discoverable<?> discoverable = getDiscoverable(middlewareEntity, pojoConverterContext);
				DiscoveryChangePlan changePlan = new DiscoveryChangePlan();
				Object discoveredObject = discoverable.discover(tec.getAttributes(), changePlan);
				return pojoConverterContext.toEntity(discoveredObject);
			} finally {
				tec.destroy();
			}
		} finally {
			pojoConverterContext.destroy();
		}
	}

	@SuppressWarnings("unchecked")
	public List<ConfigurationItemEntity> fetchConfigurationWithChildren(final ConfigurationItemEntity middleware) {
		PojoConverter.Context pojoConverterContext = pojoConverter.getContext();
		try {
			// TODO: validate that the discoveryRequired fields are filled in

			TaskExecutionContext tec = new TaskExecutionContext(Maps.<String, Object> newHashMap());
			try {
				final DiscoveryChangePlan changePlan = new DiscoveryChangePlan();

				final Discoverable<Serializable> discoverable = getDiscoverable(middleware, pojoConverterContext);
				Serializable discoveredObject = discoverable.discover(tec.getAttributes(), changePlan);
				discoverable.discoverChildrenInfo(tec.getAttributes(), changePlan);

				List<Serializable> discoveredItems = Lists.newArrayList();
				discoveredItems.add(discoveredObject);

				// sorting the keys because we expect the plugin to return the discovered objects in numbered keys (1,2,3,..) that specify the order in which
				// the
				// objects need to be added
				/*
				 * TODO: replace with some topological graph ordering code (i.e. make a graph where the objects are the nodes, and there is an edge between
				 * nodes X and Y iff there is a relation from X to Y (Y is a child of X or Y has a pointer to X). A (reverse) topological order of this graph
				 * gives the order in which objects need to be saved. This will also catch circular references!
				 */
				List<String> discovertedItemKeyList = newArrayList(tec.getAttributes().keySet());
				Collections.sort(discovertedItemKeyList);

				for (String eachKey : discovertedItemKeyList) {
					Object possibleDiscoveredObjects = tec.getAttributes().get(eachKey);
					if (!(possibleDiscoveredObjects instanceof Collection)) {
						continue;
					}

					for (Object eachPossibleDiscoveredObject : (Collection<Object>) possibleDiscoveredObjects) {
						if (eachPossibleDiscoveredObject.getClass().isAnnotationPresent(ConfigurationItem.class)) {
							discoveredItems.add((Serializable) eachPossibleDiscoveredObject);
						}
					}
				}

				labelGenerator.generateCorrectIds(discoveredItems);

				List<ConfigurationItemEntity> discoveredEntities = pojoConverterContext.toEntity(discoveredItems);

				return discoveredEntities;
			} finally {
				tec.destroy();
			}

		} finally {
			pojoConverterContext.destroy();
		}
	}

	@SuppressWarnings("unchecked")
	private Discoverable<Serializable> getDiscoverable(final ConfigurationItemEntity middlewareEntity, PojoConverter.Context pojoConverterContext) {
		final Serializable ci = pojoConverterContext.toPojo(middlewareEntity);
		checkArgument(ci instanceof Discoverable, "The configuration item [%s] is not discoverable.", middlewareEntity.getId());
		return (Discoverable<Serializable>) ci;
	}

}
