package com.xebialabs.deployit.service.deployment;

import static com.google.common.collect.Lists.newArrayList;

import java.io.Serializable;
import java.util.Collection;
import java.util.List;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import com.xebialabs.deployit.ChangeResolution;
import com.xebialabs.deployit.MappingInfo;
import com.xebialabs.deployit.Mappings;
import com.xebialabs.deployit.ResolutionException;
import com.xebialabs.deployit.RunBook;
import com.xebialabs.deployit.Step;
import com.xebialabs.deployit.checks.Checks;
import com.xebialabs.deployit.ci.Deployment;
import com.xebialabs.deployit.ci.DeploymentPackage;
import com.xebialabs.deployit.ci.Environment;
import com.xebialabs.deployit.ci.MiddlewareResource;
import com.xebialabs.deployit.ci.artifact.DeployableArtifact;
import com.xebialabs.deployit.ci.mapping.Mapping;
import com.xebialabs.deployit.exception.DeployitException;
import com.xebialabs.deployit.exception.HttpResponseCodeResult;
import com.xebialabs.deployit.plugin.PluginLoader;
import com.xebialabs.deployit.plugin.PojoConverter;
import com.xebialabs.deployit.reflect.ConfigurationItemDescriptor;
import com.xebialabs.deployit.repository.ConfigurationItemEntity;
import com.xebialabs.deployit.repository.RepositoryService;
import com.xebialabs.deployit.task.Task;
import com.xebialabs.deployit.task.deployment.InitialDeploymentTask;
import com.xebialabs.deployit.task.deployment.UndeploymentTask;
import com.xebialabs.deployit.task.deployment.UpgradeDeploymentTask;
import com.xebialabs.deployit.translation.DefaultChange;
import com.xebialabs.deployit.translation.MappingGenerator;
import com.xebialabs.deployit.translation.SimpleChangePlan;
import com.xebialabs.deployit.typedescriptor.ConfigurationItemDescriptorRepositoryHolder;

@Component
public class DeploymentService {

	@Autowired
	private PojoConverter pojoConverter;

	@Autowired
	private MappingGenerator defaultMappingGenerator;

	@Autowired
	private PluginLoader pluginLoader;

	@Autowired
	private RepositoryService repositoryService;

	public Mappings generatePartialDefaultMappings(ConfigurationItemEntity source, List<ConfigurationItemEntity> sources, ConfigurationItemEntity target,
	        String mappingType) {
		final PojoConverter.Context pojoConverterContext = pojoConverter.getContext();
		try {
			final DeploymentPackage deploymentPackage = pojoConverterContext.toPojo(source);

			final DeploymentPackage dummyDeploymentPackage = new DeploymentPackage();
			dummyDeploymentPackage.setApplication(deploymentPackage.getApplication());

			for (ConfigurationItemEntity each : sources) {
				final Object pojo = pojoConverterContext.toPojo(each);
				if (pojo instanceof DeployableArtifact) {
					dummyDeploymentPackage.addDeployableArtifact((DeployableArtifact) pojo);
				} else if (pojo instanceof MiddlewareResource) {
					dummyDeploymentPackage.addMiddlewareResource((MiddlewareResource) pojo);
				} else {
					throw new Checks.IncorrectArgumentException("{} is not a DeployableArtifact or a MiddlewareResource.", pojo);
				}
			}
			final String targetType = target.getConfigurationItemTypeName();
			final ConfigurationItemDescriptor targetDescriptor = ConfigurationItemDescriptorRepositoryHolder.getDescriptor(targetType);
			final Environment env;
			if (isAssignable(targetDescriptor, Environment.class)) {
				env = pojoConverterContext.toPojo(target);
			} else {
				env = new Environment();
				final Object targetPojo = pojoConverterContext.toPojo(target);
				env.addMember((Serializable) targetPojo);
			}

			return convertMappings(defaultMappingGenerator.generateMappingsForInitialDeployment(dummyDeploymentPackage, env, mappingType));
		} finally {
			pojoConverterContext.destroy();
		}
	}

	private boolean isAssignable(ConfigurationItemDescriptor desc, Class<?> clazz) {
		return clazz.isAssignableFrom(desc.getTypeClass());
	}

	public Mappings generateDefaultMappings(ConfigurationItemEntity source, ConfigurationItemEntity target) {
		final PojoConverter.Context pojoConverterContext = pojoConverter.getContext();
		try {
			final DeploymentPackage pkg = pojoConverterContext.toPojo(source);
			final Environment env = pojoConverterContext.toPojo(target);
			final Collection<MappingInfo> mappings = defaultMappingGenerator.generateMappingsForInitialDeployment(pkg, env, null);
			return convertMappings(mappings);
		} finally {
			pojoConverterContext.destroy();
		}
	}

	public Mappings generateMappingsForUpgradeDeployment(ConfigurationItemEntity newSourceEntity, ConfigurationItemEntity deploymentEntity) {
		final PojoConverter.Context pojoConverterContext = pojoConverter.getContext();
		try {
			final Deployment deployment = pojoConverterContext.toPojo(deploymentEntity);
			final DeploymentPackage newPackage = pojoConverterContext.toPojo(newSourceEntity);
			final Collection<MappingInfo> generatedMappingsForUpgrade = defaultMappingGenerator.generateMappingsForUpgradeDeployment(deployment, newPackage);
			return convertMappings(generatedMappingsForUpgrade);
		} finally {
			pojoConverterContext.destroy();
		}
	}

	private Mappings convertMappings(final Collection<MappingInfo> mappingsInfo) {
		final Mappings mappings = new Mappings();
		final List<ConfigurationItemEntity> validMappings = newArrayList();
		final List<ConfigurationItemEntity> invalidMappings = newArrayList();
		for (MappingInfo mappingInfo : mappingsInfo) {
			ConfigurationItemEntity mappingEntity = pojoConverter.toEntity(mappingInfo.getMapping());
			if (mappingInfo.getValidMapping()) {
				validMappings.add(mappingEntity);
			} else {
				invalidMappings.add(mappingEntity);
			}
		}
		mappings.setValidMappings(validMappings);
		mappings.setInvalidMappings(invalidMappings);
		return mappings;
	}

	@SuppressWarnings("rawtypes")
	public Task prepareInitialDeployment(final ConfigurationItemEntity sourceEntity, final ConfigurationItemEntity targetEntity,
	        final Collection<ConfigurationItemEntity> mappingEntities) {
		final PojoConverter.Context pojoConverterContext = pojoConverter.getContext();
		final DeploymentPackage pkg = pojoConverterContext.toPojo(sourceEntity);
		final Environment env = pojoConverterContext.toPojo(targetEntity);
		final Deployment newDeployment = new Deployment(pkg, env);
		newDeployment.setLabel(generateDeploymentLabel(pkg, env));
		for (ConfigurationItemEntity each : mappingEntities) {
			String sourceId = (String) each.getValue("source");
			String generatedId = (String) each.getValue("target") + "/" + sourceId.substring(sourceId.lastIndexOf("/") + 1);
			each.setId(generatedId);
			newDeployment.addMapping((Mapping) pojoConverterContext.toPojo(each));
		}
		List<Step> steps = getSteps(DefaultChange.with(null, newDeployment));

		InitialDeploymentTask initialDeploymentTask = new InitialDeploymentTask(sourceEntity, targetEntity, mappingEntities, steps, repositoryService,
		        pojoConverter, pojoConverterContext);
		initialDeploymentTask.setLabel(String.format(Task.DEPLOYMENT_TASK_LABEL_FORMAT, pkg.getLabel(), pkg.getVersion(), env));
		return initialDeploymentTask;
	}

	public static String generateDeploymentLabel(final DeploymentPackage pkg, final Environment env) {
		String applicationId = pkg.getApplication().getLabel();
		String applicationName = applicationId.substring(applicationId.lastIndexOf("/") + 1);

		return env.getLabel() + "/" + applicationName;
	}

	public Task prepareUpgradeDeployment(final ConfigurationItemEntity newSourceEntity, final ConfigurationItemEntity existingDeploymentEntity,
	        final Collection<ConfigurationItemEntity> mappingEntities) {
		final PojoConverter.Context repositoryContext = pojoConverter.getContext();
		final DeploymentPackage newPkg = repositoryContext.toPojo(newSourceEntity);
		final Deployment existingDeployment = repositoryContext.toPojo(existingDeploymentEntity);
		final Environment env = existingDeployment.getTarget();

		final Deployment newDeployment = new Deployment(newPkg, env);
		newDeployment.setLabel(existingDeployment.getLabel());
		for (ConfigurationItemEntity each : mappingEntities) {
			String sourceId = (String) each.getValue("source");
			String generatedId = (String) each.getValue("target") + "/" + sourceId.substring(sourceId.lastIndexOf("/") + 1);
			each.setId(generatedId);
		}

		final PojoConverter.Context mappingsContext = pojoConverter.getContext();
		final Collection<Mapping<?, ?>> mappings = mappingsContext.toPojo(mappingEntities);
		for (Mapping<?, ?> each : mappings) {
			newDeployment.addMapping(each);
		}
		final ConfigurationItemEntity newDeploymentEntity = pojoConverter.toEntity(newDeployment);

		List<Step> steps = getSteps(DefaultChange.with(existingDeployment, newDeployment));
		UpgradeDeploymentTask upgradeDeploymentTask = new UpgradeDeploymentTask(mappingEntities, newDeploymentEntity, steps, repositoryService, pojoConverter,
		        repositoryContext, mappingsContext);
		upgradeDeploymentTask.setLabel(String.format(Task.UPGRADE_TASK_LABEL_FORMAT, newPkg.getLabel(), existingDeployment.getSource().getVersion(),
		        newPkg.getVersion(), env));
		return upgradeDeploymentTask;
	}

	protected List<Step> getSteps(final DefaultChange<Deployment> change) {
		final SimpleChangePlan plan = new SimpleChangePlan(change);
		final Collection<RunBook> runBookCollection = pluginLoader.getRunBooks();
		try {
			for (RunBook runBook : runBookCollection) {
				Collection<ChangeResolution> resolutions = runBook.resolve(plan);
				for (ChangeResolution each : resolutions) {
					plan.addSteps(each.getSteps());
				}
			}
		} catch (ResolutionException re) {
			logger.error("Failed to resolve all the mappings into steps.", re);
			throw new DeploymentException(re.getMessage());
		}
		return plan.getSteps();
	}

	public Task prepareUndeployment(final ConfigurationItemEntity source) {
		final PojoConverter.Context pojoConverterContext = pojoConverter.getContext();
		final Deployment deployment = pojoConverterContext.toPojo(source);
		List<Step> steps = getSteps(DefaultChange.with(deployment, null));
		UndeploymentTask undeploymentTask = new UndeploymentTask(deployment, steps, repositoryService, pojoConverter, pojoConverterContext);
		undeploymentTask.setLabel(String.format(Task.UNDEPLOYMENT_TASK_LABEL_FORMAT, deployment.getSource().getLabel(), deployment.getSource().getVersion(),
		        deployment.getTarget()));
		return undeploymentTask;
	}

	@SuppressWarnings("serial")
	@HttpResponseCodeResult(statusCode = 400)
	public static class DeploymentException extends DeployitException {
		public DeploymentException(final String message) {
			super(message);
		}
	}

	private static final Logger logger = LoggerFactory.getLogger(DeploymentService.class);

	public Task prepareRedeployment(ConfigurationItemEntity source, ConfigurationItemEntity target, Collection<ConfigurationItemEntity> mappingEntities) {
		// TODO Auto-generated method stub
		return null;
	}
}
