package com.xebialabs.deployit.service.comparison;

import static com.xebialabs.deployit.checks.Checks.checkArgument;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;

import org.apache.commons.lang.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import com.google.common.collect.ArrayListMultimap;
import com.google.common.collect.ListMultimap;
import com.google.common.collect.Lists;
import com.xebialabs.deployit.reflect.ConfigurationItemDescriptor;
import com.xebialabs.deployit.reflect.ConfigurationItemPropertyDescriptor;
import com.xebialabs.deployit.repository.FlattenedEntity;
import com.xebialabs.deployit.repository.Flattener;
import com.xebialabs.deployit.repository.RepositoryObjectEntity;
import com.xebialabs.deployit.typedescriptor.ConfigurationItemDescriptorRepositoryHolder;

@Component
public class Comparator {

    private static final String PASSWORD_OBFUSCATION = "********";

    @Autowired
    private Flattener flattener;

    public ListMultimap<String, String> compare(RepositoryObjectEntity reference, RepositoryObjectEntity... entities) {
        ListMultimap<String, String> comparison = ArrayListMultimap.create();

        checkCiTypes(reference.getConfigurationItemTypeName(), entities);

        final ConfigurationItemDescriptor descriptor = ConfigurationItemDescriptorRepositoryHolder.getDescriptor(reference);
        int nrCi = 0;

        List<RepositoryObjectEntity> entitiesToCompare = Lists.newArrayList();
        entitiesToCompare.add(reference);
        entitiesToCompare.addAll(Arrays.asList(entities));
        for (RepositoryObjectEntity compareEntity : entitiesToCompare) {
          FlattenedEntity flattened = flattener.flatten(compareEntity);
          writeFields(comparison, flattened);
          writeProperties(comparison, flattened, descriptor, nrCi);
          nrCi++;
        }

        return comparison;
    }

    @SuppressWarnings("unchecked")
    private void writeProperties(final ListMultimap<String, String> comparison, final RepositoryObjectEntity entity, final ConfigurationItemDescriptor descriptor, final int nrCi) {
        final ConfigurationItemPropertyDescriptor[] descriptors = descriptor.getPropertyDescriptors();
        for (ConfigurationItemPropertyDescriptor propertyDescriptor : descriptors) {
            final String key = propertyDescriptor.getName();
            Object value = entity.getValue(key);
            if (propertyDescriptor.isPassword()) {
                value = PASSWORD_OBFUSCATION;
            }
            switch (propertyDescriptor.getType()) {
                case BOOLEAN:
                case INTEGER:
                case STRING:
                case ENUM:
                case CI:
                    comparison.put(key, (String) value);
                    break;
                case SET_OF_STRINGS:
                case SET_OF_CIS:
                    handleSetOfString(comparison, key, (Collection<String>) value, nrCi);
                    break;
                case LIST_OF_OBJECTS:
                    if (value != null) {
                        handleListOfObjects(comparison, key, (List<Map<String, Object>>) value, propertyDescriptor, nrCi);
                    }
                    break;
                case UNSUPPORTED:
                    break;
            }
        }
    }

    private void handleSetOfString(final ListMultimap<String, String> comparison, final String key, final Collection<String> value, final int nrCi) {
        if (value != null) {
            final ArrayList<String> list = new ArrayList<String>(value);
            Collections.sort(list);
            setOrAdd(comparison, key, StringUtils.join(list, ","), nrCi);
        } else {
            setOrAdd(comparison, key, null, nrCi);
        }
    }

    private void setOrAdd(final ListMultimap<String, String> comparison, final String key, final String value, final int index) {
        List<String> stringList = comparison.get(key);
        if (stringList.size() > index) {
            stringList.set(index, value);
        } else if (stringList.size() == index) {
            stringList.add(value);
        } else {
            // Fill her up with null to the correct size...
            for (int i = stringList.size(); i < index; i++) {
                stringList.add(null);
            }
            stringList.add(value);
        }
    }

    @SuppressWarnings("unchecked")
    private void handleListOfObjects(final ListMultimap<String, String> comparison, final String key, final List<Map<String, Object>> objectList, final ConfigurationItemPropertyDescriptor propertyDescriptor, final int nrCi) {
        int index = 0;
        final ConfigurationItemPropertyDescriptor[] descriptors = propertyDescriptor.getListObjectPropertyDescriptors();
        for (Map<String, Object> map : objectList) {
            String keyPrefix = key + index + ".";
            for (ConfigurationItemPropertyDescriptor descriptor : descriptors) {
                switch (descriptor.getType()) {
                    case BOOLEAN:
                    case INTEGER:
                    case STRING:
                    case ENUM:
                    case CI:
                        setOrAdd(comparison, keyPrefix + descriptor.getName(), (String) map.get(descriptor.getName()), nrCi);
                        break;
                    case SET_OF_STRINGS:
                    case SET_OF_CIS:
                        handleSetOfString(comparison, keyPrefix + descriptor.getName(), (Collection<String>) map.get(descriptor.getName()), nrCi);
                        break;
                    case LIST_OF_OBJECTS:
                        break;
                    case UNSUPPORTED:
                        break;
                }
            }
            index++;
        }
    }

    private void writeFields(final ListMultimap<String, String> comparison, final FlattenedEntity entity) {
        comparison.put("id", entity.getId());
        comparison.put("type", entity.getConfigurationItemTypeName());
        comparison.put("archetypeId", entity.getArchetypeId());
    }

    private void checkCiTypes(final String referenceType, final RepositoryObjectEntity... entities) {
        for (RepositoryObjectEntity entity : entities) {
            checkArgument(entity.getConfigurationItemTypeName().equals(referenceType), "Not all configuration items are of type %s", referenceType);
        }
    }

}
