package com.xebialabs.deployit.security;

import static com.google.common.base.Preconditions.checkNotNull;
import static com.google.common.collect.Lists.newArrayList;
import static com.xebialabs.deployit.jcr.JcrConstants.CONFIGURATION_NODE_ID;
import static com.xebialabs.deployit.jcr.JcrConstants.PERMISSION_PROPERTY_NAME_PREFIX;
import static org.apache.commons.collections.CollectionUtils.addAll;

import java.io.IOException;
import java.security.Principal;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Set;

import javax.jcr.Node;
import javax.jcr.PathNotFoundException;
import javax.jcr.Property;
import javax.jcr.RepositoryException;
import javax.jcr.Session;
import javax.jcr.Value;
import javax.jcr.ValueFormatException;
import javax.security.auth.Subject;

import org.apache.commons.lang.ArrayUtils;
import org.apache.jackrabbit.api.JackrabbitSession;
import org.apache.jackrabbit.api.security.principal.PrincipalIterator;
import org.apache.jackrabbit.api.security.principal.PrincipalManager;
import org.apache.jackrabbit.core.SessionImpl;
import org.apache.jackrabbit.core.security.principal.AdminPrincipal;
import org.apache.jackrabbit.value.StringValue;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.context.SecurityContextHolder;
import org.springframework.stereotype.Component;

import com.google.common.base.Function;
import com.google.common.collect.Collections2;
import com.google.common.collect.Sets;
import com.xebialabs.deployit.jcr.JcrCallback;
import com.xebialabs.deployit.jcr.JcrTemplate;

@Component("permissionService")
public class JcrPermissionService implements PermissionService {

	protected JcrTemplate jcrTemplate;

    @Autowired
    public JcrPermissionService(JcrTemplate jcrTemplate) {
        this.jcrTemplate = jcrTemplate;
    }

    @Override
	public boolean hasPermission(final String permissionName, final String principalName) {
		return jcrTemplate.execute(new JcrCallback<Boolean>() {
			@Override
			public Boolean doInJcr(Session session) throws IOException, RepositoryException {
				final PrincipalManager principalManager = ((JackrabbitSession) session).getPrincipalManager();
				final Principal principal = principalManager.getPrincipal(principalName);
				if (isAdmin(principal, permissionName)) return true;
				List<String> groups = getGroupMemberships(principalManager, principal);

				return checkHasPermission(session, groups, principalName, permissionName);
			}
		});
	}

	@Override
	public boolean hasPermission(final String permissionName) {
		final Authentication auth = SecurityContextHolder.getContext().getAuthentication();

		return jcrTemplate.execute(new JcrCallback<Boolean>() {
			@Override
			public Boolean doInJcr(Session session) throws IOException, RepositoryException {
				// N.B.: Casting the session to SessionImpl is the only way to get the Subject.
				final Subject subject = ((SessionImpl) session).getSubject();

				final PrincipalManager principalManager = ((JackrabbitSession) session).getPrincipalManager();
				final List<String> groups = newArrayList();
				for (Principal eachPrincipal : subject.getPrincipals()) {
					if (isAdmin(eachPrincipal, permissionName)) return true;
					groups.addAll(getGroupMemberships(principalManager, eachPrincipal));
				}

				final String principalName = auth.getName();

				return checkHasPermission(session, groups, principalName, permissionName);
			}
		});
	}

	private boolean isAdmin(Principal eachPrincipal, String permissionName) {
		if (eachPrincipal instanceof AdminPrincipal) {
			logger.debug("Granted {} because user is admin.", permissionName);
			return true;
		}
		return false;
	}

	private Boolean checkHasPermission(Session session, List<String> groups, String principalName, String permissionName) throws RepositoryException {
		final Node configNode = checkNotNull(session.getNode(CONFIGURATION_NODE_ID));
		final Value[] allowedPrincipals = getAllowedPrincipals(configNode, permissionName);
		for (Value each : allowedPrincipals) {
			final String stringValue = each.getString();
			if (stringValue.equals(principalName)) {
				logger.debug("Granted {} because user {} has the permission.", permissionName, principalName);
				return true;
			} else if (groups.contains(stringValue)) {
				logger.debug("Granted {} because group {} has the permission.", permissionName, stringValue);
				return true;
			}
		}

		return false;
	}

	@Override
	public void grantPermission(final String permissionName, final String... principals) {
		logger.debug("Granting {} to {}", permissionName, Arrays.toString(principals));
		jcrTemplate.execute(new JcrCallback<Object>() {
			@Override
			public Object doInJcr(Session session) throws IOException, RepositoryException {
				Node configNode = checkNotNull(session.getNode(CONFIGURATION_NODE_ID));

				Set<String> newPrincipalsList = Sets.newHashSet();
				for (Value each : getAllowedPrincipals(configNode, permissionName)) {
					newPrincipalsList.add(each.getString());
				}
				addAll(newPrincipalsList, principals);

				Collection<Value> newValueCollection = Collections2.transform(newPrincipalsList, new Function<String, Value>() {
					@Override
					public Value apply(String from) {
						return new StringValue(from);
					}
				});
				Value[] newValues = newValueCollection.toArray(new Value[newValueCollection.size()]);

				configNode.setProperty(PERMISSION_PROPERTY_NAME_PREFIX + permissionName, newValues);

				session.save();
				return null;
			}
		});
	}

	@Override
	public void denyPermission(final String permissionName, final String... principals) {
		jcrTemplate.execute(new JcrCallback<Object>() {
			@Override
			public Object doInJcr(Session session) throws IOException, RepositoryException {
				Node configNode = checkNotNull(session.getNode(CONFIGURATION_NODE_ID));
				Value[] oldValues = getAllowedPrincipals(configNode, permissionName);

				List<Value> newValueList = newArrayList();
				for (Value oldValue : oldValues) {
					if (!ArrayUtils.contains(principals, oldValue.getString())) {
						newValueList.add(oldValue);
					}
				}

				Value[] newValues = newValueList.toArray(new Value[newValueList.size()]);
				configNode.setProperty(PERMISSION_PROPERTY_NAME_PREFIX + permissionName, newValues);
				session.save();
				return null;
			}
		});
	}

	protected Value[] getAllowedPrincipals(Node configNode, final String permissionName) throws RepositoryException, ValueFormatException {
		Value[] oldValues;
		try {
			Property permissionProperty = configNode.getProperty(PERMISSION_PROPERTY_NAME_PREFIX + permissionName);
			oldValues = permissionProperty.getValues();
		} catch (PathNotFoundException exc) {
			oldValues = new Value[0];
		}
		return oldValues;
	}

	private static final Logger logger = LoggerFactory.getLogger(JcrPermissionService.class);

	private List<String> getGroupMemberships(PrincipalManager principalManager, Principal principal) {
		final PrincipalIterator groupPrincipals = principalManager.getGroupMembership(principal);
		final List<String> groups = newArrayList();
		while (groupPrincipals.hasNext()) {
			final Principal group = groupPrincipals.nextPrincipal();
			groups.add(group.getName());
		}

		return groups;
	}

}
