package com.xebialabs.deployit.security;

import com.xebialabs.deployit.jcr.JcrCallback;
import com.xebialabs.deployit.jcr.JcrTemplate;
import org.apache.jackrabbit.api.JackrabbitSession;
import org.apache.jackrabbit.api.security.JackrabbitAccessControlEntry;
import org.apache.jackrabbit.api.security.JackrabbitAccessControlList;
import org.apache.jackrabbit.core.security.principal.PrincipalImpl;
import org.apache.jackrabbit.value.ValueFactoryImpl;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import javax.jcr.RepositoryException;
import javax.jcr.Session;
import javax.jcr.Value;
import javax.jcr.security.*;
import java.io.IOException;
import java.security.Principal;
import java.util.Collection;
import java.util.List;
import java.util.Map;

import static com.google.common.collect.Iterables.toArray;
import static com.google.common.collect.Lists.newArrayList;
import static com.google.common.collect.Maps.newHashMap;
import static com.xebialabs.deployit.repository.JcrPathHelper.getAbsolutePathFromId;

@Component("accessControlService")
public class JcrAccessControlService implements AccessControlService {

	JcrTemplate jcrTemplate;

    @Autowired
    public JcrAccessControlService(JcrTemplate jcrTemplate) {
        this.jcrTemplate = jcrTemplate;
    }

    @Override
	public Collection<JcrAccessControlEntry> getAccessControlEntries(final String id) {
		return jcrTemplate.execute(new JcrCallback<Collection<JcrAccessControlEntry>>() {
			@Override
			public Collection<JcrAccessControlEntry> doInJcr(Session session) throws IOException, RepositoryException {
				AccessControlManager acm = session.getAccessControlManager();
				List<JcrAccessControlEntry> entries = newArrayList();
				for (AccessControlPolicy eachPolicy : acm.getPolicies(getAbsolutePathFromId(id))) {
					if (!(eachPolicy instanceof JackrabbitAccessControlList))
						continue;

					JackrabbitAccessControlList acl = (JackrabbitAccessControlList) eachPolicy;
					for (AccessControlEntry eachEntry : acl.getAccessControlEntries()) {
						JackrabbitAccessControlEntry entry = (JackrabbitAccessControlEntry) eachEntry;
						List<String> permissions = newArrayList();
						for (Privilege eachPrivilege : entry.getPrivileges()) {
							permissions.add(eachPrivilege.getName());
						}
						boolean isTransitive = true;
						if (((JackrabbitAccessControlEntry) eachEntry).getRestriction("rep:glob") != null) {
							isTransitive = false;
						}
						entries.add(new JcrAccessControlEntry(entry.getPrincipal().getName(), entry.isAllow(), permissions, isTransitive));
					}
				}
				return entries;
			}
		});
	}

	@Override
	public void setAccessControlEntries(final String id, final JcrAccessControlEntry... entries) {
		final String path = getAbsolutePathFromId(id);
		jcrTemplate.execute(new JcrCallback<Object>() {
			@Override
			public Object doInJcr(Session session) throws IOException, RepositoryException {
				AccessControlManager acm = session.getAccessControlManager();

				boolean accessControlPolicyHasBeenSet = false;
				for (AccessControlPolicy eachPolicy : acm.getPolicies(path)) {
					if (tryAndSetAccessControlPolicy(path, session, eachPolicy, entries)) {
						accessControlPolicyHasBeenSet = true;
					}
				}

				if (!accessControlPolicyHasBeenSet) {
					AccessControlPolicyIterator applicablePolicies = acm.getApplicablePolicies(path);
					while (applicablePolicies.hasNext()) {
						AccessControlPolicy eachPolicy = applicablePolicies.nextAccessControlPolicy();
						if (tryAndSetAccessControlPolicy(path, session, eachPolicy, entries)) {
							accessControlPolicyHasBeenSet = true;
						}
					}
				}
	
				if (!accessControlPolicyHasBeenSet) {
					throw new IllegalStateException("Unable to set access control entries on " + id);
				}

				session.save();
				return null;
			}
		});

	}

	@Override
	public boolean hasPrivilege(final String id, final String privilege) {
		return jcrTemplate.execute(new JcrCallback<Boolean>() {
			@Override
			public Boolean doInJcr(Session session) throws IOException, RepositoryException {
				AccessControlManager acm = session.getAccessControlManager();
				final Privilege aPrivilege = acm.privilegeFromName(privilege);
				return acm.hasPrivileges(getAbsolutePathFromId(id), new Privilege[]{aPrivilege});
			}
		});
	}

	private boolean tryAndSetAccessControlPolicy(final String path, final Session session, final AccessControlPolicy policy,
	        final JcrAccessControlEntry... entries) throws RepositoryException {
		if (!(policy instanceof JackrabbitAccessControlList))
			return false;

        final AccessControlManager acm = session.getAccessControlManager();

        JackrabbitAccessControlList acl = (JackrabbitAccessControlList) policy;
		final String[] strings = acl.getRestrictionNames();
		
		for (AccessControlEntry eachEntry : acl.getAccessControlEntries()) {
			acl.removeAccessControlEntry(eachEntry);
		}
		for (JcrAccessControlEntry eachEntry : entries) {
            final String principalName = eachEntry.getPrincipalName();
            if (((JackrabbitSession) session).getPrincipalManager().hasPrincipal(principalName)) {
                Principal principal = new PrincipalImpl(principalName);
                List<Privilege> privilegeList = newArrayList();
                for (String eachPrivilege : eachEntry.getPrivileges()) {
                    privilegeList.add(acm.privilegeFromName(eachPrivilege));
                }
                Privilege[] privileges = toArray(privilegeList, Privilege.class);
                if (logger.isDebugEnabled()) {
                    for (Privilege privilege : privileges) {
                        logger.debug("{} {} for {} on {}", new Object[] {eachEntry.isAllow() ? "Adding" : "Denying", privilege.getName(), principalName, path});
                    }
                }
	            if (privileges.length > 0) {
		            if (eachEntry.isTransitive()) {
                        acl.addEntry(principal, privileges, eachEntry.isAllow());
		            } else {
			            Map<String, Value> restrictions = newHashMap();
			            restrictions.put("rep:glob", ValueFactoryImpl.getInstance().createValue(""));
			            acl.addEntry(principal, privileges, eachEntry.isAllow(), restrictions);
		            }
	            }
            } else {
                logger.warn("Skipping {} because {} is not known.", eachEntry, principalName);
            }
		}
		acm.setPolicy(path, acl);
		return true;
	}

    private static final Logger logger = LoggerFactory.getLogger(JcrAccessControlService.class);
}
