package com.xebialabs.deployit.repository;

import static com.google.common.collect.Lists.newArrayList;
import static com.xebialabs.deployit.jcr.JcrConstants.ARCHETYPE_NODETYPE_NAME;
import static com.xebialabs.deployit.jcr.JcrConstants.CONFIGURATION_ITEM_NODETYPE_NAME;
import static com.xebialabs.deployit.jcr.JcrConstants.CONFIGURATION_ITEM_TYPE_PROPERTY_NAME;
import static com.xebialabs.deployit.jcr.JcrConstants.ID_PROPERTY_NAME;
import static com.xebialabs.deployit.jcr.JcrConstants.LAST_MODIFIED_DATE_PROPERTY_NAME;
import static com.xebialabs.deployit.reflect.ConfigurationItemPropertyType.CI;
import static com.xebialabs.deployit.reflect.ConfigurationItemPropertyType.SET_OF_CIS;
import static java.lang.String.format;
import static javax.jcr.nodetype.NodeType.NT_BASE;
import static javax.jcr.query.Query.JCR_SQL2;
import static org.apache.commons.lang.StringUtils.isNotBlank;
import static org.apache.commons.lang.StringUtils.startsWith;

import java.util.Calendar;
import java.util.List;
import java.util.Map;

import javax.jcr.RepositoryException;
import javax.jcr.Session;
import javax.jcr.Value;
import javax.jcr.ValueFactory;
import javax.jcr.query.Query;
import javax.jcr.query.QueryManager;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.Transformer;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.collect.Lists;
import com.xebialabs.deployit.jcr.JcrConstants;
import com.xebialabs.deployit.reflect.ConfigurationItemDescriptor;
import com.xebialabs.deployit.reflect.ConfigurationItemPropertyDescriptor;
import com.xebialabs.deployit.typedescriptor.ConfigurationItemTypeDescriptorRepository;

/**
 */
public class SearchQueryBuilder {
	public static final String CI_SELECTOR_NAME = "ci";

	private static final String EQUALITY_OPERATOR = "=";
	private static final String LIKE_OPERATOR = "LIKE";
	private static final String ISCHILDNODE_OPERATOR = "ISCHILDNODE";

	private final ConfigurationItemTypeDescriptorRepository descriptorRepository;
	private final SearchParameters parameters;
	private final StringBuilder joins;
	private final List<Condition> conditions;
	private int nextSelectorId = 1;

	public SearchQueryBuilder(final ConfigurationItemTypeDescriptorRepository descriptorRepository, final SearchParameters parameters) {
		this.descriptorRepository = descriptorRepository;
		this.parameters = parameters;
		this.joins = new StringBuilder();
		this.conditions = newArrayList();
	}

    @SuppressWarnings("deprecated")
	public Query build(final Session session) throws RepositoryException {
		createJoinsAndConditions();

		QueryManager qm = session.getWorkspace().getQueryManager();
		final ValueFactory valueFactory = session.getValueFactory();
		final String q = constructQueryString();
		final Query query = qm.createQuery(q, JCR_SQL2);

		addBinds(valueFactory, query);
		addPagingInfo(query);

        logger.debug("Query built: {}", query.getStatement());
		return query;
	}

	String constructQueryString() {
		StringBuilder builder = new StringBuilder();
		builder.append("SELECT " + CI_SELECTOR_NAME + ".* FROM ");

		builder.append("[");
		if (parameters.at == null) {
			builder.append(parameters.archetypes ? ARCHETYPE_NODETYPE_NAME : CONFIGURATION_ITEM_NODETYPE_NAME);
		} else {
			builder.append(NT_BASE);
		}
		builder.append("]  AS " + CI_SELECTOR_NAME);

		builder.append(joins.toString());

		if (!conditions.isEmpty()) {
			builder.append(" WHERE ");
			builder.append(StringUtils.join(CollectionUtils.collect(conditions, new Transformer() {
				@Override
				public Object transform(final Object input) {
					return ((Condition) input).build();
				}
			}), " AND "));
		}

		builder.append(" ORDER BY " + CI_SELECTOR_NAME + ".[" + ID_PROPERTY_NAME + "]");

		logger.debug("Built query: {}", builder.toString());

		return builder.toString();
	}

	private void addBinds(final ValueFactory valueFactory, final Query query) throws RepositoryException {
		for (Condition eachCondition : conditions) {
			if (eachCondition.parameter == null)
				continue;

			for (int i = 0; i < eachCondition.values.length; i++) {
				Object eachValue = eachCondition.values[i];
				Value v;
				if (eachValue instanceof String) {
					v = valueFactory.createValue((String) eachValue);
				} else if (eachValue instanceof Calendar) {
					v = valueFactory.createValue((Calendar) eachValue);
				} else {
					throw new IllegalArgumentException("Value of Condition is not a String or a Calendar but a " + eachValue.getClass().getName());
				}
				query.bindValue(eachCondition.parameter + i, v);

				logger.debug("Bound {} to {}", eachCondition.parameter + i, v);
			}
		}
	}

	void addPagingInfo(final Query query) {
		if (parameters.resultsPerPage > 0) {
			query.setLimit(parameters.resultsPerPage);
			query.setOffset(parameters.page * parameters.resultsPerPage);
		}
	}

	private void createJoinsAndConditions() {
        createConditionForParent();

        createConditionForId();

        createConditionForConfigurationItemTypeName();

        createConditionForAt();

		createConditionForCreatingTaskId();

		createConditionsForProperties();
	}

	private void createConditionForConfigurationItemTypeName() {
		if (isNotBlank(parameters.configurationItemTypeName)) {
			List<ConfigurationItemDescriptor> descriptor = descriptorRepository.getDescriptorsBySuperType(parameters.configurationItemTypeName);
			if (descriptor.isEmpty()) {
				logger.warn("Cannot search for configuration item type " + parameters.configurationItemTypeName + " because it is not a known type");
				conditions.add(Condition.from(CI_SELECTOR_NAME, CONFIGURATION_ITEM_TYPE_PROPERTY_NAME, "_configurationItemTypeName", EQUALITY_OPERATOR, false,
				        parameters.configurationItemTypeName));
				return;
			}

			List<String> typenames = Lists.newArrayList();
			for (ConfigurationItemDescriptor each : descriptor) {
				typenames.add(each.getType());
			}
			conditions.add(Condition.from(CI_SELECTOR_NAME, CONFIGURATION_ITEM_TYPE_PROPERTY_NAME, "_configurationItemTypeName", EQUALITY_OPERATOR, false,
			        (Object[]) typenames.toArray(new String[typenames.size()])));
		}
	}

	private void createConditionForParent() {
		if (isNotBlank(parameters.parent)) {
			if (parameters.at != null) {
				// We cannot use ISCHILDNODE for frozen nodes
				throw new IllegalArgumentException("Cannot combine fromPath and at search parameters");
			}

			String parent = parameters.parent;
			if (!startsWith(parent, "/")) {
				parent = "/" + parent;
			}
			conditions.add(Condition.from(CI_SELECTOR_NAME, ID_PROPERTY_NAME, null, ISCHILDNODE_OPERATOR, parent));
		}
	}

	private void createConditionForId() {
		if (isNotBlank(parameters.id)) {
			if (parameters.id.contains("%")) {
				conditions.add(Condition.imatch(CI_SELECTOR_NAME, ID_PROPERTY_NAME, "_id", parameters.id));
			} else {
				conditions.add(Condition.match(CI_SELECTOR_NAME, ID_PROPERTY_NAME, "_id", parameters.id));
			}
		}
	}

	private void createConditionForAt() {
		if (parameters.at != null) {
			conditions.add(Condition.from(CI_SELECTOR_NAME, LAST_MODIFIED_DATE_PROPERTY_NAME, "_at", "<=", parameters.at));
		}
	}

	private void createConditionForCreatingTaskId() {
		if (parameters.creatingTaskId != null) {
			conditions.add(Condition.from(CI_SELECTOR_NAME, JcrConstants.CREATING_TASK_ID_PROPERTY_NAME, "_creatingTaskId", EQUALITY_OPERATOR,
			        parameters.creatingTaskId));
		}
	}

	private void createConditionsForProperties() {
		for (Map.Entry<String, String> entry : parameters.properties.entrySet()) {
			createConditionForProperty(entry);
		}
	}

	protected void createConditionForProperty(Map.Entry<String, String> entry) {
		if (descriptorRepository != null && isNotBlank(parameters.configurationItemTypeName)) {
			ConfigurationItemDescriptor descriptor = descriptorRepository.getDescriptorByType(parameters.configurationItemTypeName);
			if (descriptor != null) {
				ConfigurationItemPropertyDescriptor propertyDescriptor = descriptor.getPropertyDescriptor(entry.getKey());
				if (propertyDescriptor != null) {
					if (propertyDescriptor.getType() == CI) {
						String selectorName = "referenced" + nextSelectorId;
						joins.append(" INNER JOIN [" + CONFIGURATION_ITEM_NODETYPE_NAME + "] AS " + selectorName + " ON " + CI_SELECTOR_NAME + ".["
						        + entry.getKey() + "] = " + selectorName + ".[jcr:uuid]");
						conditions.add(Condition.match(selectorName, ID_PROPERTY_NAME, entry.getKey(), entry.getValue()));
						nextSelectorId++;
						return;
					} else if (propertyDescriptor.getType() == SET_OF_CIS) {
						// FIXME: Find out how to make the JOIN works on this case. The INNER JOIN used for the single CI property causes unpredictable results.
						// The results of the JOIN are different every time if the multi-value property has more than one value.
						throw new IllegalArgumentException("Cannot query property " + propertyDescriptor + " because it is of type SET_OF_CIS");
					}
				}
			}
		}
		conditions.add(Condition.match(CI_SELECTOR_NAME, entry.getKey(), entry.getKey(), entry.getValue()));
	}

	private static class Condition {
		String selector;
		String field;
		String operator;
		String parameter;
		Object[] values;
		boolean caseInsensitive;

		private static final String CASE_SENSITIVE = "%s.[%s] %s $%s";
		private static final String CASE_INSENSITIVE = "LOWER(%s.[%s]) %s $%s";
		private static final String ISCHILDNODE_FORMAT = "ISCHILDNODE(%s, [\'%s\'])";

		String build() {
			StringBuilder conditionString = new StringBuilder();
			conditionString.append("(");
			for (int i = 0; i < values.length; i++) {
				if (i > 0) {
					conditionString.append(" OR ");
				}

				if (operator.equals(ISCHILDNODE_OPERATOR)) {
					conditionString.append(format(ISCHILDNODE_FORMAT, selector, values[i]));
				} else {
					conditionString.append(format(caseInsensitive ? CASE_INSENSITIVE : CASE_SENSITIVE, selector, field, operator, parameter + i));
				}
			}
			conditionString.append(")");
			return conditionString.toString();
		}

		static Condition match(final String selector, final String field, final String parameter, final String value) {
			return match(selector, field, parameter, value, false);
		}

		static Condition imatch(final String selector, final String field, final String parameter, final String value) {
			return match(selector, field, parameter, value.toLowerCase(), true);
		}

		private static Condition match(final String selector, final String field, final String parameter, final String value, final boolean caseInsensitive) {
			String operator;
			if (value.contains("%")) {
				operator = LIKE_OPERATOR;
			} else {
				operator = EQUALITY_OPERATOR;
			}
			return from(selector, field, parameter, operator, caseInsensitive, value);
		}

		static Condition from(final String selector, final String field, final String parameter, final String operator, final Object value) {
			return from(selector, field, parameter, operator, false, value);
		}

		static Condition from(final String selector, final String field, final String parameter, final String operator, final boolean caseInsensitive,
		        final Object... values) {
			Condition t = new Condition();
			t.selector = selector;
			t.field = field;
			t.parameter = parameter;
			t.operator = operator;
			t.caseInsensitive = caseInsensitive;
			t.values = values;
			return t;
		}
	}

	private static final Logger logger = LoggerFactory.getLogger(SearchQueryBuilder.class);
}
