package com.xebialabs.deployit.repository;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Preconditions.checkNotNull;
import static com.google.common.collect.Lists.newArrayList;
import static com.xebialabs.deployit.ConfigurationItemRoot.NESTED;
import static com.xebialabs.deployit.jcr.JcrConstants.ARCHETYPE_NODETYPE_NAME;
import static com.xebialabs.deployit.jcr.JcrConstants.ARCHETYPE_ROOT_NODE_NAME;
import static com.xebialabs.deployit.jcr.JcrConstants.ARTIFACT_NODETYPE_NAME;
import static com.xebialabs.deployit.jcr.JcrConstants.CONFIGURATION_ITEM_NODETYPE_NAME;
import static com.xebialabs.deployit.repository.JcrPathHelper.getAbsolutePathFromId;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Comparator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

import javax.jcr.Node;
import javax.jcr.PathNotFoundException;
import javax.jcr.ReferentialIntegrityException;
import javax.jcr.RepositoryException;
import javax.jcr.Session;
import javax.jcr.nodetype.NodeType;
import javax.jcr.query.Query;
import javax.jcr.query.QueryResult;
import javax.jcr.query.RowIterator;
import javax.jcr.version.VersionManager;

import org.apache.commons.lang.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import com.google.common.collect.Lists;
import com.xebialabs.deployit.checks.Checks;
import com.xebialabs.deployit.exception.NotFoundException;
import com.xebialabs.deployit.jcr.JcrCallback;
import com.xebialabs.deployit.jcr.JcrConstants;
import com.xebialabs.deployit.jcr.JcrTemplate;
import com.xebialabs.deployit.reflect.ConfigurationItemDescriptor;
import com.xebialabs.deployit.reflect.ConfigurationItemPropertyDescriptor;
import com.xebialabs.deployit.reflect.ConfigurationItemPropertyType;
import com.xebialabs.deployit.typedescriptor.ConfigurationItemTypeDescriptorRepository;

@Component
public class JcrRepositoryService implements RepositoryService {

	@Autowired
	private JcrTemplate jcrTemplate;

	@Autowired
	private ConfigurationItemTypeDescriptorRepository descriptorRepository;

	@Override
	public <T extends RepositoryObjectEntity> void create(final T... entities) {
		verifyEntitiesMeetCreationPreconditions(entities);
		createEntities(entities);
	}

	private void verifyEntitiesMeetCreationPreconditions(final RepositoryObjectEntity[] entities) {
		for (int i = 0; i < entities.length; i++) {
			RepositoryObjectEntity entity = entities[i];
			checkNotNull(entity, "entity at index %s is null.", i);
			checkNotNull(entity.getId(), "entity at index %s has null id.", i);

			if (entity instanceof ArtifactEntity) {
				checkArgument(((ArtifactEntity) entity).containsData(), "Artifact %s should contain data", entity
						.getId());
			}
		}
	}

	private <T extends RepositoryObjectEntity> void createEntities(final T[] entitiesParam) {
		final List<T> entities = Arrays.asList(entitiesParam);
		Collections.sort(entities, new Comparator<T>() {
			@Override
			public int compare(T o1, T o2) {
				final int nrSlashes1 = StringUtils.countMatches(o1.getId(), "/");
				final int nrSlashes2 = StringUtils.countMatches(o2.getId(), "/");
				return nrSlashes1 - nrSlashes2;
			}
		});
		jcrTemplate.execute(new JcrCallback<Object>() {
			@Override
			public Object doInJcr(Session session) throws IOException, RepositoryException {
				List<EntityNodeWriter> writers = newArrayList();
				for (RepositoryObjectEntity entity : entities) {
					Node node = createNode(session, entity);
					final EntityNodeWriter<RepositoryObjectEntity> nodeWriter = new EntityNodeWriter<RepositoryObjectEntity>(
							session, entity, node);
					nodeWriter.writeBasics();
					writers.add(nodeWriter);
				}
				writeEntities(writers);
				session.save();
				return null;
			}
		});
	}

	private void writeEntities(List<EntityNodeWriter> writers) throws RepositoryException {
		for (EntityNodeWriter writer : writers) {
			writer.write();
		}
	}

	private Node createNode(Session session, RepositoryObjectEntity entity) throws RepositoryException {
		validateNodeStoredInCorrectPath(session, entity);
		final Node node = session.getRootNode().addNode(entity.getId());
		int index = node.getIndex();
		if (index != 1) {
			node.remove();
			throw new ItemAlreadyExistsException("Repository entity with ID [%s] already exists", entity.getId());
		}
		setMixins(entity, node);
		return node;
	}

	@Override
	public <T extends RepositoryObjectEntity> T read(final String id) {
		checkNotNull(id, "id is null");

		return this.<T> read(id, false);
	}

	@Override
	public <T extends RepositoryObjectEntity> T readFully(String id) {
		checkNotNull(id, "id is null");

		return this.<T> read(id, true);
	}

	private <T extends RepositoryObjectEntity> T read(final String id, final boolean fully) {
		checkNotNull(id, "id is null");

		return jcrTemplate.execute(new JcrCallback<T>() {
			@Override
			public T doInJcr(Session session) throws IOException, RepositoryException {
				try {
					Node node = session.getNode(getAbsolutePathFromId(id));
					return new EntityNodeReader(session, node, fully).<T> getEntity();
				} catch (PathNotFoundException exc) {
					throw new NotFoundException(exc, "Repository entity [%s] not found", id);
				}
			}
		});
	}

	@Override
	public <T extends RepositoryObjectEntity> void update(final T entity) {
		checkNotNull(entity, "entity is null");
		checkNotNull(entity.getId(), "id is null");

		jcrTemplate.execute(new JcrCallback<Object>() {
			@Override
			public Object doInJcr(final Session session) throws IOException, RepositoryException {
				try {
					final Node node = session.getNode(getAbsolutePathFromId(entity.getId()));
					checkpoint(session, node);
					new EntityNodeWriter<T>(session, entity, node).write();
					session.save();
					return null;
				} catch (PathNotFoundException exc) {
					throw new NotFoundException("Repository entity [%s] not found", entity.getId());
				}
			}
		});
	}

	@Override
	public boolean delete(final String id) {
		checkNotNull(id, "id is null");

		return jcrTemplate.execute(new JcrCallback<Boolean>() {
			@Override
			public Boolean doInJcr(Session session) throws IOException, RepositoryException {
				try {
					Node node = session.getNode(getAbsolutePathFromId(id));
					boolean isArchetype = isAnArchetype(node); // Need to check
																// this before
																// delete...
					checkpoint(session, node);
					try {
						node.remove();
						session.save();
					} catch (ReferentialIntegrityException integrityException) {
						if (isArchetype) {
							throw new ItemInUseException(
									"Can't delete archetype [%s] which is in use by any configuration items", id);
						} else {
							throw new ItemInUseException(integrityException,
									"ConfigurationItem with id [%s] is still referenced, cannot be deleted.", id);
						}
					}
				} catch (PathNotFoundException ignored) {
					return false;
				}
				return true;
			}
		});
	}

	@Override
	public List<String> list(final SearchParameters parameters) {
		checkNotNull(parameters);

		List<RepositoryObjectEntity> entities = listEntities(parameters);
		List<String> entityIds = Lists.newArrayList();
		for (RepositoryObjectEntity each : entities) {
			entityIds.add(each.getId());
		}
		return entityIds;
	}

	@Override
	public <T extends RepositoryObjectEntity> List<T> listEntities(final SearchParameters parameters) {
		checkNotNull(parameters, "parameters is null");

		return jcrTemplate.execute(new JcrCallback<List<T>>() {
			@Override
			public List<T> doInJcr(final Session session) throws IOException, RepositoryException {
				final Query query = new SearchQueryBuilder(descriptorRepository, parameters).build(session);
				final QueryResult queryResult = query.execute();
				Map<String, T> entities = new LinkedHashMap<String, T>();
				final RowIterator iterator = queryResult.getRows();
				while (iterator.hasNext()) {
					// TODO temporary fix!
					// We need to fix the security model for deployment to
					// inherit the privileges from both the source and the
					// target.
					// Remove the try-catch once that is implemented.
					try {
						Node each = iterator.nextRow().getNode(SearchQueryBuilder.CI_SELECTOR_NAME);
						T entity = new EntityNodeReader(session, each, false).<T> getEntity();
						if (entityNotFoundYetOrNewerThenEntityAlreadyFound(entities, entity)) {
							entities.put(entity.getId(), entity);
						}
					} catch (RepositoryException rre) {
						// Ignore, we weren't allowed to read a node, or one of
						// it's relations.
					}
				}
				return new ArrayList<T>(entities.values());
			}

		});
	}

	private <T extends RepositoryObjectEntity> boolean entityNotFoundYetOrNewerThenEntityAlreadyFound(
			Map<String, T> entities, T entity) {
		return !entities.containsKey(entity.getId())
				|| entities.get(entity.getId()).getLastModified().before(entity.getLastModified());
	}

	@Override
	public boolean checkNodeExists(final String id) {
		return jcrTemplate.execute(new JcrCallback<Boolean>() {
			@Override
			public Boolean doInJcr(final Session session) throws IOException, RepositoryException {
				return session.itemExists(getAbsolutePathFromId(id));
			}
		});
	}

	private <T extends RepositoryObjectEntity> void validateNodeStoredInCorrectPath(final Session session,
			final T entity) throws RepositoryException {
		final String id = entity.getId();
		final String typeName = entity.getConfigurationItemTypeName();
		final ConfigurationItemDescriptor desc = descriptorRepository.getDescriptorByType(typeName);
		checkNotNull(desc, "Unknown configuration item type %s", typeName);

		final String[] pathElements = id.split("/");
		if (entity instanceof ArchetypeEntity) {
			Checks.checkArgument(pathElements.length == 2 && pathElements[0].equals(ARCHETYPE_ROOT_NODE_NAME),
					"Archetype cannot be stored at %s. It should be stored under " + ARCHETYPE_ROOT_NODE_NAME, id);
		} else if (desc.getRoot() != NESTED) {
			Checks.checkArgument(pathElements.length == 2 && pathElements[0].equals(desc.getRoot().getRootNodeName()),
					"Configuration item of type %s cannot be stored at %s. It should be stored under %s", typeName, id,
					desc.getRoot().getRootNodeName());
		} else {
			Checks
					.checkArgument(
							pathElements.length >= 3,
							"Configuration item of type %s cannot be stored at %s. It should be stored under a valid parent node",
							typeName, id);
			final String parentId = id.substring(0, id.lastIndexOf('/'));
			final Node parentNode = session.getNode(getAbsolutePathFromId(parentId));
			Checks.checkArgument(parentNode != null,
					"Configuration item of type %s cannot be stored at %s. The parent does not exist", typeName, id);
			Checks.checkArgument(parentNode.isNodeType(CONFIGURATION_ITEM_NODETYPE_NAME),
					"Configuration item of type %s cannot be stored at %s. The parent is not a configuration item",
					typeName, id);
			final String parentTypeName = parentNode.getProperty(JcrConstants.CONFIGURATION_ITEM_TYPE_PROPERTY_NAME)
					.getString();
			final ConfigurationItemDescriptor parentDescriptor = descriptorRepository
					.getDescriptorByType(parentTypeName);

			// Check parent relation
			for (ConfigurationItemPropertyDescriptor aPD : desc.getPropertyDescriptors()) {
				if (aPD.getType() == ConfigurationItemPropertyType.CI && aPD.asContainment()) {
					if (matchesType(parentDescriptor, aPD.getPropertyClassname())) {
						return;
					}
				}
			}

			// Check child relation
			for (ConfigurationItemPropertyDescriptor aPD : parentDescriptor.getPropertyDescriptors()) {
				if (aPD.getType() == ConfigurationItemPropertyType.SET_OF_CIS && aPD.asContainment()) {
					if (matchesType(desc, aPD.getCollectionMemberClassname())) {
						return;
					}
				}
			}

			throw new Checks.IncorrectArgumentException(
					"Configuration item of type %s cannot be stored at %s. The parent cannot contain configuration items of this type",
					typeName, id);
		}
	}

	private boolean matchesType(final ConfigurationItemDescriptor desc, String wantedType) {
		boolean isExactClass = wantedType.equals(desc.getType());
		boolean isSubClass = desc.getSuperClasses().contains(wantedType);
		boolean implementsInterface = desc.getInterfaces().contains(wantedType);
		return isExactClass || isSubClass || implementsInterface;
	}

	private <T extends RepositoryObjectEntity> void setMixins(final T entity, final Node node)
			throws RepositoryException {
		if (entity instanceof ArtifactEntity) {
			node.addMixin(ARTIFACT_NODETYPE_NAME);
			node.addMixin(CONFIGURATION_ITEM_NODETYPE_NAME);
		} else if (entity instanceof ConfigurationItemEntity) {
			node.addMixin(CONFIGURATION_ITEM_NODETYPE_NAME);
		} else if (entity instanceof ArchetypeEntity) {
			node.addMixin(ARCHETYPE_NODETYPE_NAME);
		} else {
			throw new IllegalArgumentException("Not supporting RepositoryEntity type " + entity.getClass().getName());
		}
		node.addMixin(NodeType.MIX_REFERENCEABLE);
		node.addMixin(NodeType.MIX_VERSIONABLE);
	}

	private boolean isAnArchetype(Node node) throws RepositoryException {
		NodeType[] mixinNodeTypes = node.getMixinNodeTypes();
		for (NodeType type : mixinNodeTypes) {
			if (type.getName().equals(ARCHETYPE_NODETYPE_NAME)) {
				return true;
			}
		}
		return false;
	}

	private void checkpoint(final Session session, final Node node) throws RepositoryException {
		VersionManager versionManager = session.getWorkspace().getVersionManager();
		versionManager.checkpoint(node.getPath());
	}

	static String generateJcrPropertyNameForNestedProperty(ConfigurationItemPropertyDescriptor nestedPd) {
		return nestedPd.getOwningPropertyDescriptor().getName() + "_" + nestedPd.getName();
	}

}
