package com.xebialabs.deployit.repository;

import static com.google.common.collect.Lists.newArrayList;
import static com.xebialabs.deployit.jcr.JcrConstants.ARCHETYPE_PROPERTY_NAME;
import static com.xebialabs.deployit.jcr.JcrConstants.CONFIGURATION_ITEM_TYPE_PROPERTY_NAME;
import static com.xebialabs.deployit.jcr.JcrConstants.CREATING_TASK_ID_PROPERTY_NAME;
import static com.xebialabs.deployit.jcr.JcrConstants.DATA_PROPERTY_NAME;
import static com.xebialabs.deployit.jcr.JcrConstants.ID_PROPERTY_NAME;
import static com.xebialabs.deployit.jcr.JcrConstants.LAST_MODIFIED_DATE_PROPERTY_NAME;
import static com.xebialabs.deployit.repository.JcrPathHelper.getAbsolutePathFromId;

import java.util.Calendar;
import java.util.Collection;
import java.util.List;
import java.util.Map;

import javax.jcr.Binary;
import javax.jcr.Node;
import javax.jcr.PathNotFoundException;
import javax.jcr.RepositoryException;
import javax.jcr.Session;
import javax.jcr.Value;
import javax.jcr.ValueFactory;

import org.apache.jackrabbit.value.ReferenceValue;

import com.google.common.collect.Lists;
import com.xebialabs.deployit.reflect.ConfigurationItemDescriptor;
import com.xebialabs.deployit.reflect.ConfigurationItemPropertyDescriptor;
import com.xebialabs.deployit.reflect.ConfigurationItemPropertyType;
import com.xebialabs.deployit.typedescriptor.ConfigurationItemDescriptorRepositoryHolder;

/**
 * Writes an {@link RepositoryObjectEntity} to a JCR {@link Node}.
 */
class EntityNodeWriter<T extends RepositoryObjectEntity> {

	private final Session session;
	private final T entity;
	private final Node node;
	private boolean basicsWritten = false;

	public EntityNodeWriter(final Session session, final T entity, final Node node) {
		this.session = session;
		this.entity = entity;
		this.node = node;
	}

	public void writeBasics() throws RepositoryException {
		copyConfigurationItemTypeName();
		setArchetypeReferenceProperty();
		basicsWritten = true;
	}

	public void write() throws RepositoryException {
		if (!basicsWritten) {
			writeBasics();
		}
		copyData();
		copyValuesIntoNode();
		copyMetadata();
	}

	private void copyConfigurationItemTypeName() throws RepositoryException {
		node.setProperty(CONFIGURATION_ITEM_TYPE_PROPERTY_NAME, entity.getConfigurationItemTypeName());
	}

	private void setArchetypeReferenceProperty() throws RepositoryException {
		ArchetypeEntity archetype = entity.getConfigurationItemArchetype();
		if (archetype != null) {
			String archetypeId = archetype.getId();
			Node archetypeNode = session.getNode(JcrPathHelper.getAbsolutePathFromId(archetypeId));
			node.setProperty(ARCHETYPE_PROPERTY_NAME, archetypeNode);
		} else if (node.hasProperty(ARCHETYPE_PROPERTY_NAME)) {
			node.getProperty(ARCHETYPE_PROPERTY_NAME).remove();
		}
	}

	private void copyData() throws RepositoryException {
		if (entity instanceof ArtifactEntity) {
			ArtifactEntity artifact = (ArtifactEntity) entity;
			if (artifact.containsData()) {
				Binary binary = session.getValueFactory().createBinary(artifact.getData());
				node.setProperty(DATA_PROPERTY_NAME, binary);
			}
		}
	}

	private void copyValuesIntoNode() throws RepositoryException {
		ConfigurationItemDescriptor ciDescriptor = ConfigurationItemDescriptorRepositoryHolder.getDescriptor(entity);
		for (ConfigurationItemPropertyDescriptor pd : ciDescriptor.getPropertyDescriptors()) {

			if (entity.getValue(pd.getName()) == null) {
				removePropertyFromNode(pd);
				continue;
			}

			switch (pd.getType()) {
			case BOOLEAN:
			case INTEGER:
			case STRING:
                if (pd.isPassword()) {
                    copyPasswordIntoNode(pd);
                    break;
                }
			case ENUM:
				copyPrimitivePropertyIntoNode(pd);
				break;
			case LIST_OF_OBJECTS:
				copyListOfObjectsPropertyIntoNode(pd);
				break;
			case SET_OF_STRINGS:
				copySetOfStringsPropertyIntoNode(pd);
				break;
			case CI:
				copyConfigurationItemPropertyIntoNode(pd);
				break;
			case SET_OF_CIS:
				copySetOfConfigurationItemsPropertyIntoNode(pd);
				break;
			default:
				throw new IllegalArgumentException("Cannot convert property " + pd.getName() + " because it is of unsupported type " + pd.getType());

			}
		}
	}

    private void copyPasswordIntoNode(final ConfigurationItemPropertyDescriptor pd) throws RepositoryException {
        final String password = convertObjectValueToString(pd, entity.getValue(pd.getName()));
        if ("********".equals(password)) {
            // no-op
        } else {
            node.setProperty(pd.getName(), password);
        }
    }

    private void copyMetadata() throws RepositoryException {
		node.setProperty(ID_PROPERTY_NAME, entity.getId());
		if (entity.getOverrideLastModified() != null) {
			node.setProperty(LAST_MODIFIED_DATE_PROPERTY_NAME, entity.getOverrideLastModified());
		} else {
			node.setProperty(LAST_MODIFIED_DATE_PROPERTY_NAME, Calendar.getInstance());
		}
		if (entity.getCreatingTaskId() != null) {
			node.setProperty(CREATING_TASK_ID_PROPERTY_NAME, entity.getCreatingTaskId());
		}
	}

	private void removePropertyFromNode(ConfigurationItemPropertyDescriptor pd) throws RepositoryException {
		try {
			node.getProperty(pd.getName()).remove();
		} catch (PathNotFoundException ignored) {
		}

		if (pd.getType() == ConfigurationItemPropertyType.LIST_OF_OBJECTS) {
			for (ConfigurationItemPropertyDescriptor eachNestedPd : pd.getPropertyDescriptors()) {
				try {
					node.getProperty(JcrRepositoryService.generateJcrPropertyNameForNestedProperty(eachNestedPd)).remove();
				} catch (PathNotFoundException ignored) {
				}
			}
		}
	}

	private void copyPrimitivePropertyIntoNode(ConfigurationItemPropertyDescriptor pd) throws RepositoryException {
		String valueAsString = convertObjectValueToString(pd, entity.getValue(pd.getName()));
		node.setProperty(pd.getName(), valueAsString);
	}

	@SuppressWarnings("rawtypes")
	private void copyListOfObjectsPropertyIntoNode(ConfigurationItemPropertyDescriptor pd) throws RepositoryException {
		final ValueFactory valueFactory = session.getValueFactory();

		Collection<?> valueAsCollection = convertObjectValueToCollection(pd, entity.getValue(pd.getName()));
		for (ConfigurationItemPropertyDescriptor eachNestedPd : pd.getPropertyDescriptors()) {
			List<Value> jcrValueList = Lists.newArrayList();

			for (Object each : valueAsCollection) {
				if (!(each instanceof Map)) {
					throw new IllegalArgumentException("Element in property " + pd.getName() + " of repository entity " + entity.getId() + " is not a Map: "
					        + each);
				}
				Object objectValue = ((Map) each).get(eachNestedPd.getName());
				String stringValue;
				if (objectValue == null) {
					stringValue = "";
				} else if (objectValue instanceof String) {
					stringValue = (String) objectValue;
				} else {
					throw new IllegalArgumentException("Property " + eachNestedPd.getName() + " in element in property " + pd.getName()
					        + " of repository entity " + entity.getId() + " is not a String: " + objectValue);
				}
				jcrValueList.add(valueFactory.createValue(stringValue));
			}
			Value[] jcrValues = jcrValueList.toArray(new Value[jcrValueList.size()]);
			node.setProperty(JcrRepositoryService.generateJcrPropertyNameForNestedProperty(eachNestedPd), jcrValues);
		}
		node.setProperty(pd.getName(), "property-set");
	}

	private void copySetOfStringsPropertyIntoNode(ConfigurationItemPropertyDescriptor pd) throws RepositoryException {
		final ValueFactory valueFactory = session.getValueFactory();

		Collection<?> valueAsCollection = convertObjectValueToCollection(pd, entity.getValue(pd.getName()));
		List<Value> jcrValueList = newArrayList();
		for (Object each : valueAsCollection) {
			if (!(each instanceof String)) {
				throw new IllegalArgumentException("Element in property " + pd.getName() + " of repository entity " + entity.getId() + " is not a String: "
				        + each);
			}
			String stringValue = (String) each;
			jcrValueList.add(valueFactory.createValue(stringValue));
		}
		Value[] jcrValues = jcrValueList.toArray(new Value[jcrValueList.size()]);
		node.setProperty(pd.getName(), jcrValues);
	}

	private void copyConfigurationItemPropertyIntoNode(ConfigurationItemPropertyDescriptor pd) throws RepositoryException {
        if (pd.asContainment()) {
            return;
        }

		String referencedCiId = convertObjectValueToString(pd, entity.getValue(pd.getName()));
		Node referencedCi = session.getNode(getAbsolutePathFromId(referencedCiId));
		node.setProperty(pd.getName(), referencedCi);
	}

	private void copySetOfConfigurationItemsPropertyIntoNode(ConfigurationItemPropertyDescriptor pd) throws RepositoryException {
        if (pd.asContainment()) {
            return;
        }

		Collection<?> valueAsCollection = convertObjectValueToCollection(pd, entity.getValue(pd.getName()));
		List<Value> jcrReferenceList = newArrayList();
		for (Object each : valueAsCollection) {
			if (!(each instanceof String)) {
				throw new IllegalArgumentException("Element in property " + pd.getName() + " of repository entity " + entity.getId() + " is not a String: "
				        + each);
			}
			String referencedCiId = (String) each;
			Node referencedCi = session.getNode(getAbsolutePathFromId(referencedCiId));
			jcrReferenceList.add(new ReferenceValue(referencedCi));
		}
		Value[] jcrReferenceValues = jcrReferenceList.toArray(new Value[jcrReferenceList.size()]);
		node.setProperty(pd.getName(), jcrReferenceValues);
	}

	private String convertObjectValueToString(ConfigurationItemPropertyDescriptor pd, Object value) {
		if (!(value instanceof String)) {
			throw new IllegalArgumentException("Property " + pd.getName() + " of repository entity " + entity.getId() + " is not a String: " + value);
		}
		return (String) value;
	}

	private Collection<?> convertObjectValueToCollection(ConfigurationItemPropertyDescriptor pd, Object value) {
		if (!(value instanceof Collection)) {
			throw new IllegalArgumentException("Property " + pd.getName() + " of repository entity " + entity.getId() + " is not a Collection: " + value);
		}
		return (Collection<?>) value;
	}
}
