package com.xebialabs.deployit.repository;

import static com.google.common.collect.Lists.newArrayList;
import static com.google.common.collect.Sets.newHashSet;
import static com.xebialabs.deployit.jcr.JcrConstants.ARCHETYPE_NODETYPE_NAME;
import static com.xebialabs.deployit.jcr.JcrConstants.ARCHETYPE_PROPERTY_NAME;
import static com.xebialabs.deployit.jcr.JcrConstants.ARTIFACT_NODETYPE_NAME;
import static com.xebialabs.deployit.jcr.JcrConstants.CONFIGURATION_ITEM_NODETYPE_NAME;
import static com.xebialabs.deployit.jcr.JcrConstants.CONFIGURATION_ITEM_TYPE_PROPERTY_NAME;
import static com.xebialabs.deployit.jcr.JcrConstants.CREATING_TASK_ID_PROPERTY_NAME;
import static com.xebialabs.deployit.jcr.JcrConstants.DATA_PROPERTY_NAME;
import static com.xebialabs.deployit.jcr.JcrConstants.ID_PROPERTY_NAME;
import static com.xebialabs.deployit.jcr.JcrConstants.LAST_MODIFIED_DATE_PROPERTY_NAME;
import static com.xebialabs.deployit.repository.JcrPathHelper.getIdFromAbsolutePath;
import static javax.jcr.nodetype.NodeType.NT_FROZEN_NODE;
import static javax.jcr.query.Query.JCR_SQL2;
import static org.apache.jackrabbit.JcrConstants.JCR_FROZENMIXINTYPES;
import static org.apache.jackrabbit.JcrConstants.JCR_FROZENUUID;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.jcr.ItemNotFoundException;
import javax.jcr.Node;
import javax.jcr.NodeIterator;
import javax.jcr.Property;
import javax.jcr.RepositoryException;
import javax.jcr.Session;
import javax.jcr.Value;
import javax.jcr.ValueFactory;
import javax.jcr.nodetype.NodeType;
import javax.jcr.query.Query;
import javax.jcr.query.QueryManager;
import javax.jcr.query.QueryResult;

import org.apache.commons.io.IOUtils;
import org.apache.jackrabbit.spi.commons.value.QValueValue;

import com.google.common.base.Function;
import com.google.common.collect.Collections2;
import com.google.common.collect.Sets;
import com.xebialabs.deployit.jcr.RuntimeRepositoryException;
import com.xebialabs.deployit.reflect.ConfigurationItemDescriptor;
import com.xebialabs.deployit.reflect.ConfigurationItemPropertyDescriptor;
import com.xebialabs.deployit.typedescriptor.ConfigurationItemDescriptorRepositoryHolder;

/**
 * Reads an {@link RepositoryObjectEntity} from a JCR {@link Node}.
 */
class EntityNodeReader {

	private Session session;

	private Node node;

	private boolean readFully;

	public EntityNodeReader(final Session session, final Node node, final boolean readFully) {
		this.session = session;
		this.node = node;
		this.readFully = readFully;
	}

	@SuppressWarnings("unchecked")
	public <T extends RepositoryObjectEntity> T getEntity() throws RepositoryException, IOException {
		RepositoryObjectEntity entity = instantiateEntity();
		copyEntityMetadata(entity);
		if (readFully) {
			copyData(entity);
		}
		copyValuesIntoEntity(entity);
		return (T) entity;
	}

	private void copyEntityMetadata(RepositoryObjectEntity entity) throws RepositoryException {
		entity.setId(node.getProperty(ID_PROPERTY_NAME).getString());
		entity.setLastModified(node.getProperty(LAST_MODIFIED_DATE_PROPERTY_NAME).getDate());
		if (node.hasProperty(CREATING_TASK_ID_PROPERTY_NAME)) {
			entity.setCreatingTaskId(node.getProperty(CREATING_TASK_ID_PROPERTY_NAME).getString());
		}
	}

	private void copyData(RepositoryObjectEntity entity) throws RepositoryException, IOException {
		if (entity instanceof ArtifactEntity) {
			ArtifactEntity artifact = (ArtifactEntity) entity;
			InputStream binaryStream = node.getProperty(DATA_PROPERTY_NAME).getBinary().getStream();
			byte[] binaryData = IOUtils.toByteArray(binaryStream);
			ByteArrayInputStream copiedBinaryStream = new ByteArrayInputStream(binaryData);
			artifact.setData(copiedBinaryStream);
		}

	}

	private void copyValuesIntoEntity(RepositoryObjectEntity entity) throws RepositoryException {
		ConfigurationItemDescriptor ciDescriptor = ConfigurationItemDescriptorRepositoryHolder.getDescriptor(entity);
		for (ConfigurationItemPropertyDescriptor pd : ciDescriptor.getPropertyDescriptors()) {
			if (!node.hasProperty(pd.getName()) && !pd.asContainment()) {
				continue;
			}

			switch (pd.getType()) {
			case BOOLEAN:
			case INTEGER:
			case STRING:
			case ENUM:
				copyPrimitivePropertyFromNode(entity, pd);
				break;
			case LIST_OF_OBJECTS:
				copyListOfObjectsPropertyFromNode(entity, pd);
				break;
			case SET_OF_STRINGS:
				copySetOfStringsPropertyFromNode(entity, pd);
				break;
			case CI:
				copyConfigurationItemPropertyFromNode(entity, pd);
				break;
			case SET_OF_CIS:
				copySetOfConfigurationItemsPropertyFromNode(entity, pd);
				break;
			default:
				throw new IllegalArgumentException("Cannot convert property " + pd.getName() + " because it is of unsupported type " + pd.getType());
			}
		}
	}

	private RepositoryObjectEntity instantiateEntity() throws RepositoryException, IOException {
		final String configurationItemTypeName = node.getProperty(CONFIGURATION_ITEM_TYPE_PROPERTY_NAME).getString();
		final Collection<String> mixinNodeTypeNames = getMixinNodeTypeNames();

		if (mixinNodeTypeNames.contains(ARCHETYPE_NODETYPE_NAME)) {
			if (node.hasProperty(ARCHETYPE_PROPERTY_NAME)) {
				return new ArchetypeEntity(readArchetype());
			}
			return new ArchetypeEntity(configurationItemTypeName);
		}

		// check whether this entity is an artifact first, because an artifact is also a configuration-item
		if (mixinNodeTypeNames.contains(ARTIFACT_NODETYPE_NAME)) {
			if (node.hasProperty(ARCHETYPE_PROPERTY_NAME)) {
				return new ArtifactEntity(readArchetype());
			}
			return new ArtifactEntity(configurationItemTypeName);
		}

		if (mixinNodeTypeNames.contains(CONFIGURATION_ITEM_NODETYPE_NAME)) {
			if (node.hasProperty(ARCHETYPE_PROPERTY_NAME)) {
				return new ConfigurationItemEntity(readArchetype());
			}
			return new ConfigurationItemEntity(configurationItemTypeName);
		}

		throw new IllegalArgumentException("Cannot determine whether JCR node " + JcrPathHelper.getIdFromAbsolutePath(node.getPath())
		        + " is a configuration item or an archetype");
	}

	protected Collection<String> getMixinNodeTypeNames() throws RepositoryException {
		if (nodeIsFrozen()) {
			return getMixinNodeTypeNamesFromFrozenNode();
		} else {
			return getMixinNodeTypeNamesFromRegularNode();
		}
	}

	protected Collection<String> getMixinNodeTypeNamesFromFrozenNode() throws RepositoryException {
		final Property property = node.getProperty(JCR_FROZENMIXINTYPES);
		final ArrayList<Value> frozenMixinTypes = newArrayList(property.getValues());
		return Collections2.transform(frozenMixinTypes, new Function<Value, String>() {
			@Override
			public String apply(Value from) {
				try {
					return from.getString();
				} catch (RepositoryException e) {
					throw new RuntimeRepositoryException("Cannot convert Value into String", e);
				}
			}
		});
	}

	protected Collection<String> getMixinNodeTypeNamesFromRegularNode() throws RepositoryException {
		final ArrayList<NodeType> mixinNodeTypes = newArrayList(node.getMixinNodeTypes());
		return Collections2.transform(mixinNodeTypes, new Function<NodeType, String>() {
			@Override
			public String apply(NodeType from) {
				return from.getName();
			}
		});
	}

	private ArchetypeEntity readArchetype() throws RepositoryException, IOException {
		Node archetypeNode = node.getProperty(ARCHETYPE_PROPERTY_NAME).getNode();
		return new EntityNodeReader(session, archetypeNode, false).getEntity();
	}

	private void copyPrimitivePropertyFromNode(RepositoryObjectEntity entity, ConfigurationItemPropertyDescriptor pd) throws RepositoryException {
		entity.addValue(pd.getName(), node.getProperty(pd.getName()).getString());
	}

	private void copyListOfObjectsPropertyFromNode(RepositoryObjectEntity entity, ConfigurationItemPropertyDescriptor pd) throws RepositoryException {
		List<Map<String, String>> listOfObjects = null;
		for (ConfigurationItemPropertyDescriptor eachNestedPd : pd.getPropertyDescriptors()) {
			Value[] values = node.getProperty(JcrRepositoryService.generateJcrPropertyNameForNestedProperty(eachNestedPd)).getValues();
			if (listOfObjects == null) {
				listOfObjects = new ArrayList<Map<String, String>>();
				for (int i = 0; i < values.length; i++) {
					listOfObjects.add(new HashMap<String, String>());
				}
			}
			for (int i = 0; i < values.length; i++) {
				listOfObjects.get(i).put(eachNestedPd.getName(), values[i].getString());
			}
		}
		entity.addValue(pd.getName(), listOfObjects);
	}

	private void copySetOfStringsPropertyFromNode(RepositoryObjectEntity entity, ConfigurationItemPropertyDescriptor pd) throws RepositoryException {
		Set<String> setOfStrings = newHashSet();
		for (Value each : node.getProperty(pd.getName()).getValues()) {
			setOfStrings.add(each.getString());
		}
		entity.addValue(pd.getName(), setOfStrings);
	}

	private void copyConfigurationItemPropertyFromNode(RepositoryObjectEntity entity, ConfigurationItemPropertyDescriptor pd) throws RepositoryException {
        if (pd.asContainment()) {
            entity.addValue(pd.getName(), getIdFromAbsolutePath(node.getParent().getPath()));
        } else {
            Value value = node.getProperty(pd.getName()).getValue();
            entity.addValue(pd.getName(), getReferencedCiId(value));
        }
	}

	private void copySetOfConfigurationItemsPropertyFromNode(RepositoryObjectEntity entity, ConfigurationItemPropertyDescriptor pd) throws RepositoryException {
		Set<String> referencedCiIds = Sets.newHashSet();
        if (pd.asContainment()) {
            SearchParameters params = new SearchParameters().setParent(entity.getId()).setConfigurationItemType(pd.getCollectionMemberClassname());
            SearchQueryBuilder builder = new SearchQueryBuilder(ConfigurationItemDescriptorRepositoryHolder.getDescriptorRepository(), params);
            final Query query = builder.build(session);
            final QueryResult queryResult = query.execute();
            NodeIterator iterator = queryResult.getNodes();
            while (iterator.hasNext()) {
                referencedCiIds.add(getIdFromAbsolutePath(iterator.nextNode().getPath()));
            }
        } else {
            for (Value each : node.getProperty(pd.getName()).getValues()) {
                referencedCiIds.add(getReferencedCiId(each));
            }
        }
		entity.addValue(pd.getName(), referencedCiIds);
	}

	private String getReferencedCiId(Value value) throws RepositoryException {
		String referencedCiUuid = ((QValueValue) value).getQValue().getString();
		Node referencedCi = getNodeByUuid(referencedCiUuid);
		return getIdFromAbsolutePath(referencedCi.getPath());
	}

	protected Node getNodeByUuid(String referencedCiUuid) throws RepositoryException {
		try {
			return session.getNodeByIdentifier(referencedCiUuid);
		} catch (ItemNotFoundException exc) {
			if (nodeIsFrozen()) {
				Node frozenNode = getFrozenNodeByUuid(referencedCiUuid);
				if (frozenNode != null) {
					return frozenNode;
				}
			}
			throw exc;
		}
	}

	private Node getFrozenNodeByUuid(String referencedCiUuid) throws RepositoryException {
		QueryManager queryManager = session.getWorkspace().getQueryManager();
		ValueFactory valueFactory = session.getValueFactory();

		Query query = queryManager.createQuery("SELECT * FROM [" + NT_FROZEN_NODE + "] WHERE [" + JCR_FROZENUUID + "] = $uuid", JCR_SQL2);
		query.bindValue("uuid", valueFactory.createValue(referencedCiUuid));
		QueryResult result = query.execute();
		NodeIterator resultNodes = result.getNodes();
		while (resultNodes.hasNext()) {
			// Assuming that a query by UUID only return one node
			return resultNodes.nextNode();
		}
		return null;
	}

	private boolean nodeIsFrozen() throws RepositoryException {
		return node.isNodeType(NodeType.NT_FROZEN_NODE);
	}

}
