package com.xebialabs.deployit.plugin;

import static com.google.common.collect.Lists.newArrayList;
import static com.google.common.collect.Maps.newHashMap;
import static com.google.common.collect.Sets.newHashSet;
import static com.xebialabs.deployit.service.importer.Exploder.explode;
import static java.io.File.createTempFile;

import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.lang.reflect.Field;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.commons.io.FileUtils;
import org.apache.commons.io.IOUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.beans.factory.annotation.Autowired;

import com.google.common.base.Function;
import com.google.common.collect.Collections2;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import com.xebialabs.deployit.checks.Checks;
import com.xebialabs.deployit.ci.artifact.DeployableArtifact;
import com.xebialabs.deployit.ci.artifact.Folder;
import com.xebialabs.deployit.ci.mapping.KeyValuePair;
import com.xebialabs.deployit.exception.RuntimeIOException;
import com.xebialabs.deployit.reflect.ConfigurationItemDescriptor;
import com.xebialabs.deployit.reflect.ConfigurationItemPropertyDescriptor;
import com.xebialabs.deployit.repository.ArchetypeEntity;
import com.xebialabs.deployit.repository.ArtifactEntity;
import com.xebialabs.deployit.repository.ConfigurationItemEntity;
import com.xebialabs.deployit.repository.RepositoryObjectEntity;
import com.xebialabs.deployit.repository.RepositoryService;
import com.xebialabs.deployit.service.importer.Imploder;
import com.xebialabs.deployit.typedescriptor.ConfigurationItemDescriptorRepositoryHolder;
import com.xebialabs.deployit.typedescriptor.ConfigurationItemTypeDescriptorRepository;

public class PojoConverter implements InitializingBean {

	public static final String IGNORE_PLACEHOLDER = "<ignore>";
	public static final String EMPTY_PLACEHOLDER = "<empty>";
	public static final String NO_ARTIFACT_DATA_LOCATION = "no-artifact-data";

	private final ConfigurationItemTypeDescriptorRepository descriptorRepository;

	private final RepositoryService repositoryService;

	private File workDir;

	@Autowired
	public PojoConverter(final ConfigurationItemTypeDescriptorRepository descriptorRepository, final RepositoryService repositoryService) {
		this.descriptorRepository = descriptorRepository;
		this.repositoryService = repositoryService;
	}

	@Override
	public void afterPropertiesSet() throws Exception {
		if (!workDir.mkdirs() && !workDir.exists()) {
			throw new RuntimeIOException("Cannot create work dir " + workDir);
		}
	}

	public Context getContext() {
		return new Context();
	}

	public class Context {
		private Map<String, Object> convertedPojos = newHashMap();

		private Context() {
		}

		@SuppressWarnings("unchecked")
		public <T> T toPojo(ConfigurationItemEntity item) {
			if (convertedPojos.containsKey(item.getId())) {
				logger.debug("Returning previously generated POJO for configuration item {}", item.getId());
				return (T) convertedPojos.get(item.getId());
			}

			logger.debug("Reading {}", item.getId());
			final ConfigurationItemDescriptor descriptor = descriptorRepository.getDescriptorByType(item.getConfigurationItemTypeName());
			final Object pojo = descriptor.newInstance();
			storeReferenceToPojo(item, pojo);
			fillLabel(pojo, item, descriptor);
			fillArchetypeFields(pojo, item.getConfigurationItemArchetype(), descriptor);
			fillFields(pojo, item, descriptor);
			if (item instanceof ArtifactEntity) {
				fillArtifactData(pojo, (ArtifactEntity) item, descriptor);
			}
			return (T) pojo;
		}

		private void storeReferenceToPojo(ConfigurationItemEntity item, final Object pojo) {
			// The created object is stored in the convertedPojos map right away to make sure that no StackOverflowError occurs when a CI refers to itself
			// either directly or indirectly
			convertedPojos.put(item.getId(), pojo);
		}

		public <F extends ConfigurationItemEntity, G> Collection<G> toPojo(Collection<F> entities) {
			return Collections2.transform(entities, new Function<F, G>() {
				@SuppressWarnings("unchecked")
				@Override
				public G apply(F from) {
					return (G) toPojo(from);
				}
			});
		}

		private void fillLabel(Object o, ConfigurationItemEntity item, ConfigurationItemDescriptor descriptor) {
			descriptor.setLabelValueInConfigurationItem(o, item.getId());
		}

		private void fillArchetypeFields(final Object o, final ArchetypeEntity configurationItemArchetype, final ConfigurationItemDescriptor descriptor) {
			if (configurationItemArchetype != null) {
				// Recursive call because archetypes can be chained.
				fillArchetypeFields(o, configurationItemArchetype.getConfigurationItemArchetype(), descriptor);
				fillFields(o, configurationItemArchetype, descriptor);
			}
		}

		private void fillFields(final Object o, final RepositoryObjectEntity item, final ConfigurationItemDescriptor descriptor) {
			final ConfigurationItemPropertyDescriptor[] descriptors = descriptor.getPropertyDescriptors();
			for (ConfigurationItemPropertyDescriptor propertyDescriptor : descriptors) {
				if (!propertyDescriptor.getName().equals(descriptor.getPlaceholdersName())) {
					fillProperty(o, item, propertyDescriptor);
				} else {
					fillPlaceholders(o, item, propertyDescriptor);
				}
			}
		}

		@SuppressWarnings("unchecked")
		private void fillPlaceholders(Object o, RepositoryObjectEntity item, ConfigurationItemPropertyDescriptor propertyDescriptor) {
			final Collection<Map<String, String>> placeholders = (Collection<Map<String, String>>) item.getValue(propertyDescriptor.getName());
			final ArrayList<Object> pojoPlaceholders = newArrayList();
			if (placeholders != null) {
				for (Map<String, String> placeholder : placeholders) {
					String key = placeholder.get("key");
					String value = placeholder.get("value");
					if (!value.equals(IGNORE_PLACEHOLDER)) {
						final KeyValuePair pair = new KeyValuePair(key, value.equals(EMPTY_PLACEHOLDER) ? "" : value);
						pojoPlaceholders.add(pair);
					}
				}
			}
			propertyDescriptor.setPropertyValueInConfigurationItem(o, pojoPlaceholders);
		}

		private void fillProperty(final Object o, final RepositoryObjectEntity item, final ConfigurationItemPropertyDescriptor propertyDescriptor) {
			final String propertyName = propertyDescriptor.getName();
			final Object itemValue = item.getValue(propertyName);
			fillPropertyValue(o, itemValue, propertyDescriptor);
		}

		@SuppressWarnings("unchecked")
		private void fillPropertyValue(final Object o, final Object itemValue, final ConfigurationItemPropertyDescriptor propertyDescriptor) {
			Object value = null;
			if (itemValue == null)
				return;

			switch (propertyDescriptor.getType()) {
			case STRING:
				value = itemValue;
				break;
			case BOOLEAN:
				value = Boolean.valueOf((String) itemValue);
				break;
			case INTEGER:
				value = Integer.valueOf((String) itemValue);
				break;
			case ENUM:
				value = handleEnumValue(propertyDescriptor, (String) itemValue);
				break;
			case LIST_OF_OBJECTS:
				value = handleListOfObjects(propertyDescriptor, itemValue);
				break;
			case SET_OF_STRINGS:
				value = handleSetOfString(itemValue);
				break;
			case CI:
				logger.debug("Loading {} of {} with value {}", new Object[] { propertyDescriptor.getName(), o, itemValue });
				value = handleConfigurationItem((String) itemValue);
				break;
			case SET_OF_CIS:
				logger.debug("Loading {} of {} with value {}", new Object[] { propertyDescriptor.getName(), o, itemValue });
				value = handleSetofConfigurationItems((Collection<String>) itemValue);
				break;
			case UNSUPPORTED:
				throw new IllegalStateException("An UNSUPPORTED PropertyDescriptor should not be present! Call in the bug-hunters!");
			}
			propertyDescriptor.setPropertyValueInConfigurationItem(o, value);
		}

		@SuppressWarnings("unchecked")
		private Set<String> handleSetOfString(Object itemValue) {
			if (itemValue instanceof Set) {
				return (Set<String>) itemValue;
			} else if (itemValue instanceof Collection) {
				return newHashSet((Collection<String>) itemValue);
			} else {
				throw new IllegalStateException("Didn't get a Set of String or other Collection type.");
			}
		}

		private Object handleSetofConfigurationItems(final Collection<String> ids) {
			Set<Object> result = Sets.newHashSet();
			for (String id : ids) {
				result.add(handleConfigurationItem(id));
			}
			return result;
		}

		private Object handleConfigurationItem(final String id) {
			// Even though toPojo will check the convertedPojos map again, we save a call to readFully by checking the map early
			if (convertedPojos.containsKey(id)) {
				return convertedPojos.get(id);
			}
			final ConfigurationItemEntity ci = repositoryService.readFully(id);
			return toPojo(ci);
		}

		@SuppressWarnings("unchecked")
		private List<Object> handleListOfObjects(final ConfigurationItemPropertyDescriptor propertyDescriptor, final Object itemValue) {
			List<Object> result = newArrayList();
			for (Map<String, String> each : (List<Map<String, String>>) itemValue) {
				Object o;
				try {
					o = propertyDescriptor.getCollectionMemberClass().newInstance();
				} catch (Exception exc) {
					throw new IllegalStateException("Unable to instantiate type " + propertyDescriptor.getCollectionMemberClassname() + ": " + exc.getMessage());
				}
				for (ConfigurationItemPropertyDescriptor eachProp : propertyDescriptor.getListObjectPropertyDescriptors()) {
					fillPropertyValue(o, each.get(eachProp.getName()), eachProp);
				}
				result.add(o);
			}
			return result;
		}

		@SuppressWarnings({ "unchecked", "rawtypes" })
		private Object handleEnumValue(final ConfigurationItemPropertyDescriptor propertyDescriptor, final String value) {
			final Class<Enum> propertyClass = (Class<Enum>) propertyDescriptor.getPropertyClass();
			return Enum.valueOf(propertyClass, value);
		}

		private void fillArtifactData(final Object o, final ArtifactEntity item, final ConfigurationItemDescriptor descriptor) {
			if (item.getData() == null) {
				logger.warn("Not filling in the artifact data, because it wasn't set.");
				return;
			}
			if (o instanceof Folder) {
				DeployableArtifact da = (DeployableArtifact) o;
				da.setLocation(writeArtifactDataToTemporaryFolder(item).getPath());
			} else if (o instanceof DeployableArtifact) {
				DeployableArtifact da = (DeployableArtifact) o;
				da.setLocation(writeArtifactDataToTemporaryFile(item).getPath());
			}
		}

		private File writeArtifactDataToTemporaryFile(final ArtifactEntity item) {
			try {
				logger.debug("Writing data for artifact {} to temporary file in work directory {}", item.getId(), workDir);
				File artifactDataFile = createTempFile(item.getId().replace('/', '_'), ".tmp", workDir);
				FileOutputStream artifactDataOut = new FileOutputStream(artifactDataFile);
				try {
					InputStream artifactDataIn = item.getData();
					try {
						IOUtils.copy(artifactDataIn, artifactDataOut);
					} finally {
						artifactDataIn.close();
					}
				} finally {
					artifactDataOut.close();
				}
				logger.debug("Wrote data for artifact {} to file {} in work directory", item.getId(), artifactDataFile.getPath());
				return artifactDataFile;
			} catch (IOException exc) {
				throw new RuntimeIOException("Cannot write artifact data to temporary file for artifact " + item.getId(), exc);
			}
		}

		private File writeArtifactDataToTemporaryFolder(final ArtifactEntity item) {
			try {
				InputStream artifactDataIn = item.getData();
				try {
					logger.debug("Exploding folder artifact {} to temporary directory work directory {}", item.getId(), workDir);
					File artifactDataFolder = explode(artifactDataIn);
					logger.debug("Exploded folder artifact {} to directory {} in work directory", item.getId(), artifactDataFolder);
					return artifactDataFolder;
				} finally {
					artifactDataIn.close();
				}
			} catch (IOException exc) {
				throw new RuntimeIOException("Cannot write artifact data to temporary folder for artifact " + item.getId(), exc);
			}
		}

		public void destroy() {
			for (Object each : convertedPojos.values()) {
				if (each instanceof DeployableArtifact) {
					cleanupArtifactData((DeployableArtifact) each);
				}
			}
			convertedPojos.clear();
		}

		private void cleanupArtifactData(DeployableArtifact artifact) {
			Checks.checkNotNull(artifact.getLocation(), "Location of artifact is null");
			if (NO_ARTIFACT_DATA_LOCATION.equals(artifact.getLocation())) {
				logger.debug("Not cleaning up artifact data for {} because it has already been cleaned up", artifact);
				return;
			}

			File artifactData = new File(artifact.getLocation());
			artifact.setLocation(NO_ARTIFACT_DATA_LOCATION);
			if (artifactData.exists()) {
				if (artifactData.isDirectory()) {
					try {
						logger.debug("Deleting artifact data folder {} for artifact {}", artifactData, artifact);
						FileUtils.deleteDirectory(artifactData);
						logger.debug("Deleted artifact data folder {} for artifact {}", artifactData, artifact);
					} catch (IOException exc) {
						logger.warn("Cannot delete artifact data folder {} for artifact {} temporarily exported for {}", artifactData, artifact);
					}
				} else {
					logger.debug("Deleting artifact data file {} for artifact {}", artifactData, artifact);
					if (!artifactData.delete()) {
						logger.warn("Cannot delete artifact data file {} for artifact {} temporarily exported for {}", artifactData, artifact);
					}
					logger.debug("Deleted artifact data file {} for artifact {}", artifactData, artifact);
				}
			} else {
				logger.debug("Not cleaning up artifact data for artifact {} because the location {} no longer exists", artifactData);
			}
		}

		public <T> ConfigurationItemEntity toEntity(final T item) {
			ConfigurationItemEntity entity;
			if (item instanceof DeployableArtifact) {
				entity = new ArtifactEntity(item.getClass().getName());
			} else {
				entity = new ConfigurationItemEntity(item.getClass().getName());
			}

			setLabel(item, entity);
			putProperties(item, entity);
			putPlaceholders(item, entity);
			if (item instanceof DeployableArtifact) {
				setData((DeployableArtifact) item, (ArtifactEntity) entity);
			}
			return entity;
		}

		@SuppressWarnings("unchecked")
		public <F, G extends ConfigurationItemEntity> List<G> toEntity(final List<F> items) {
			List<G> entities = Lists.newArrayList();
			for (F each : items) {
				entities.add((G) toEntity(each));
			}
			return entities;
		}

		private <T> void setLabel(final T item, final ConfigurationItemEntity entity) {
			String id = ConfigurationItemDescriptorRepositoryHolder.getDescriptor(entity).getLabelValueFromConfigurationItem(item);
			entity.setId(id);
		}

		private <T> void putProperties(final T item, final ConfigurationItemEntity entity) {
			for (ConfigurationItemPropertyDescriptor each : ConfigurationItemDescriptorRepositoryHolder.getDescriptor(entity).getPropertyDescriptors()) {
				putProperty(item, entity, each);
			}
		}

		@SuppressWarnings("unchecked")
		private <T> void putPlaceholders(final T item, final ConfigurationItemEntity entity) {
			final ConfigurationItemDescriptor descriptor = ConfigurationItemDescriptorRepositoryHolder.getDescriptor(entity);
			final Field placeholderField = descriptor.getPlaceholdersField();
			if (placeholderField != null) {
				try {
					final List<KeyValuePair> p = (List<KeyValuePair>) placeholderField.get(item);
					List<Map<String, String>> placeholders = newArrayList();
					for (KeyValuePair keyValuePair : p) {
						Map<String, String> placeholder = newHashMap();
						placeholder.put("key", keyValuePair.getKey());
						placeholder.put("value", keyValuePair.getValue());
						placeholders.add(placeholder);
					}
					entity.addValue(descriptor.getPlaceholdersName(), placeholders);

				} catch (IllegalAccessException e) {
					throw new IllegalStateException("Should be able to access the placeholderField", e);
				}
			}
		}

		private <T> void putProperty(final T item, ConfigurationItemEntity entity, final ConfigurationItemPropertyDescriptor propertyDescriptor) {
			Object valueForEntity = getValue(item, propertyDescriptor);
			if (valueForEntity != null) {
				entity.addValue(propertyDescriptor.getName(), valueForEntity);
			}
		}

		@SuppressWarnings("unchecked")
		private <T> Object getValue(final T item, final ConfigurationItemPropertyDescriptor propertyDescriptor) {
			Object valueForEntity = null;
			switch (propertyDescriptor.getType()) {
			case STRING:
			case BOOLEAN:
			case INTEGER:
			case ENUM:
				Object valueFromItem = propertyDescriptor.getPropertyValueFromConfigurationItem(item);
				if (valueFromItem != null) {
					valueForEntity = valueFromItem.toString();
				}
				break;
			case SET_OF_STRINGS:
				valueForEntity = propertyDescriptor.getPropertyValueFromConfigurationItem(item);
				break;
			case CI:
				Object referencedCi = propertyDescriptor.getPropertyValueFromConfigurationItem(item);
				if (referencedCi != null) {
					valueForEntity = getLabelFromConfigurationItem(referencedCi);
				}
				break;
			case SET_OF_CIS:
				Set<Object> referencedCis = (Set<Object>) propertyDescriptor.getPropertyValueFromConfigurationItem(item);
				Set<String> result = Sets.newHashSet();
				if (referencedCis != null) {
					for (Object each : referencedCis) {
						result.add(getLabelFromConfigurationItem(each));
					}
				}
				valueForEntity = result;
				break;
			case LIST_OF_OBJECTS:
				List<Object> listOfObjects = (List<Object>) propertyDescriptor.getPropertyValueFromConfigurationItem(item);
				List<Map<String, String>> listofObjects = Lists.newArrayList();
				if (listOfObjects != null) {
					ConfigurationItemPropertyDescriptor[] nestedPropertyDescriptors = propertyDescriptor.getListObjectPropertyDescriptors();
					for (Object each : listOfObjects) {
						Map<String, String> objectValues = Maps.newHashMap();
						for (ConfigurationItemPropertyDescriptor eachProp : nestedPropertyDescriptors) {
							Object propValue = getValue(each, eachProp);
							if (propValue != null) {
								objectValues.put(eachProp.getName(), propValue.toString());
							}
						}
						listofObjects.add(objectValues);
					}
				}
				valueForEntity = listofObjects;
				break;
			}
			return valueForEntity;
		}

		private String getLabelFromConfigurationItem(final Object ci) {
			ConfigurationItemDescriptor referencedCiDescriptor = ConfigurationItemDescriptorRepositoryHolder.getDescriptorRepository().getDescriptorByClass(
			        ci.getClass());
			return referencedCiDescriptor.getLabelValueFromConfigurationItem(ci);
		}

		private <T extends DeployableArtifact> void setData(final T item, final ConfigurationItemEntity entity) {
			if (item.getLocation() == null) {
				return;
			}

			if (item instanceof Folder) {
				setFolderData((Folder) item, (ArtifactEntity) entity);
			} else {
				setFileData((DeployableArtifact) item, (ArtifactEntity) entity);
			}
		}

		private <T extends Folder> void setFolderData(final T item, final ArtifactEntity entity) {
			try {
				byte[] implodedFolderData = Imploder.implode(new File(item.getLocation()));
				entity.setData(new ByteArrayInputStream(implodedFolderData));
			} catch (IOException exc) {
				throw new RuntimeIOException("Cannot read artifact folder data from " + item, exc);
			}
		}

		private <T extends DeployableArtifact> void setFileData(final T item, final ArtifactEntity entity) {
			try {
				FileInputStream dataIn = new FileInputStream(item.getLocation());
				try {
					byte[] artifactData = IOUtils.toByteArray(dataIn);
					entity.setData(new ByteArrayInputStream(artifactData));
				} finally {
					dataIn.close();
				}
			} catch (IOException exc) {
				throw new RuntimeIOException("Cannot read artifact file data from " + item, exc);
			}
		}
	}

	public <T> ConfigurationItemEntity toEntity(final T item) {
		return new Context().toEntity(item);
	}

	public <F, G extends ConfigurationItemEntity> List<G> toEntity(final List<F> items) {
		return new Context().toEntity(items);
	}

	public void setWorkDir(File workDir) {
		this.workDir = workDir;
	}

	private static final Logger logger = LoggerFactory.getLogger(PojoConverter.class);

}
