/*
 * Copyright (c) 2008-2010 XebiaLabs B.V. All rights reserved.
 *
 * Your use of XebiaLabs Software and Documentation is subject to the Personal
 * License Agreement.
 *
 * http://www.xebialabs.com/deployit-personal-edition-license-agreement
 *
 * You are granted a personal license (i) to use the Software for your own
 * personal purposes which may be used in a production environment and/or (ii)
 * to use the Documentation to develop your own plugins to the Software.
 * "Documentation" means the how to's and instructions (instruction videos)
 * provided with the Software and/or available on the XebiaLabs website or other
 * websites as well as the provided API documentation, tutorial and access to
 * the source code of the XebiaLabs plugins. You agree not to (i) lease, rent
 * or sublicense the Software or Documentation to any third party, or otherwise
 * use it except as permitted in this agreement; (ii) reverse engineer,
 * decompile, disassemble, or otherwise attempt to determine source code or
 * protocols from the Software, and/or to (iii) copy the Software or
 * Documentation (which includes the source code of the XebiaLabs plugins). You
 * shall not create or attempt to create any derivative works from the Software
 * except and only to the extent permitted by law. You will preserve XebiaLabs'
 * copyright and legal notices on the Software and Documentation. XebiaLabs
 * retains all rights not expressly granted to You in the Personal License
 * Agreement.
 */

package com.xebialabs.deployit.plugin;

import com.xebialabs.deployit.ConfigurationItem;
import com.xebialabs.deployit.RunBook;
import com.xebialabs.deployit.exception.RuntimeIOException;
import com.xebialabs.deployit.typedescriptor.ConfigurationItemTypeDescriptorRepository;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import javax.annotation.PostConstruct;
import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.util.*;
import java.util.jar.Attributes;
import java.util.jar.Manifest;

@Component
public class PluginLoader {
	/**
	 * The name of the Manifest attribute that specifies whether a manifest
	 * entry is a {@link ConfigurationItem}.
	 */
	public static final String CONFIGURATION_ITEM_ATTRIBUTE_NAME = "ConfigurationItem";

	/**
	 * The name of the Manifest attribute that specifies whether a manifest
	 * entry is a {@link RunBook}.
	 */
	public static final String RUN_BOOK_ATTRIBUTE_NAME = "RunBook";

	/**
	 * The name of the Manifest attribute that specifies which {@link RunBook
	 * runbooks} to disable.
	 */
	public static final String DISABLED_RUN_BOOKS_ATTRIBUTE_NAME = "Disabled-RunBooks";

	/**
	 * The name of the Manifest attribute that specifies which
	 * {@link ConfigurationItem configurations}to disable.
	 */
	public static final String DISABLED_CONFIGURATION_ITEMS_ATTRIBUTE_NAME = "Disabled-ConfigurationItems";

	@Autowired
	public ConfigurationItemTypeDescriptorRepository repository;

	private final ClassLoader pluginClassLoader;

	private final Set<String> runbookEntryNames = new HashSet<String>();
	private final Set<String> configurationItemEntryNames = new HashSet<String>();
	private final Set<String> runbookEntryNamesToDisable = new HashSet<String>();
	private final Set<String> configurationItemsEntryNamesToDisable = new HashSet<String>();

	private final Collection<RunBook> runbooks = new ArrayList<RunBook>();
	private final Collection<Class<?>> configurationItemTypes = new ArrayList<Class<?>>();

	private final List<String> loadingErrors = new ArrayList<String>();

	/**
	 * Creates a {@code PluginLoader} that will look for plugins on the class
	 * path of the given class loader.
	 * 
	 * @param classLoader
	 *            the class loader to use to scan for plugins
	 * @see #PluginLoader()
	 */
	public PluginLoader(ClassLoader classLoader) {
		this.pluginClassLoader = classLoader;
	}

	/**
	 * Creates a {@code PluginLoader} that will look for plugins on the class
	 * path of the class loader that declares this class.
	 * 
	 * @see #PluginLoader(ClassLoader)
	 */
	public PluginLoader() {
		this(PluginLoader.class.getClassLoader());
	}

	@PostConstruct
	public void loadPlugins() {
		scanManifests();
		disableEntries();
		loadEntries();
	}

	public void loadPluginsWithoutDisabling() {
		scanManifests();
		loadEntries();
	}

	private void scanManifests() {
		try {
			Enumeration<URL> resources = pluginClassLoader.getResources("META-INF/MANIFEST.MF");
			while (resources.hasMoreElements()) {
				URL manifestURL = (URL) resources.nextElement();
				if (logger.isDebugEnabled()) {
					logger.debug("Loading manifest file " + manifestURL);
				}
				readManifest(manifestURL);
			}
		} catch (IOException exc) {
			throw new RuntimeIOException(exc);
		}
	}

	private void readManifest(URL manifestURL) throws IOException {
		InputStream manifestInput = manifestURL.openStream();
		try {
			Manifest manifest = new Manifest(manifestInput);
			readManifestMainAttributes(manifest);
			Map<String, Attributes> entries = manifest.getEntries();
			for (Map.Entry<String, Attributes> entry : entries.entrySet()) {
				readManifestEntry(entry);
			}
		} finally {
			manifestInput.close();
		}
	}

	private void readManifestMainAttributes(Manifest manifest) {
		String disabledRunbookEntryNamesAttrValue = StringUtils.trim(manifest.getMainAttributes().getValue(DISABLED_RUN_BOOKS_ATTRIBUTE_NAME));
		if (disabledRunbookEntryNamesAttrValue != null) {
			String[] disabledRunbookEntryNames = StringUtils.split(disabledRunbookEntryNamesAttrValue);
			runbookEntryNamesToDisable.addAll(Arrays.asList(disabledRunbookEntryNames));
		}
		String disabledConfigurationItemkEntryNamesAttrValue = StringUtils.trim(manifest.getMainAttributes().getValue(
				DISABLED_CONFIGURATION_ITEMS_ATTRIBUTE_NAME));
		if (disabledConfigurationItemkEntryNamesAttrValue != null) {
			String[] disabledCIEntryNames = StringUtils.split(disabledConfigurationItemkEntryNamesAttrValue);
			configurationItemsEntryNamesToDisable.addAll(Arrays.asList(disabledCIEntryNames));
		}

	}

	private void readManifestEntry(Map.Entry<String, Attributes> entry) {
		String entryName = StringUtils.trim(entry.getKey());
		Attributes attributes = entry.getValue();
		if ("true".equalsIgnoreCase(StringUtils.trim(attributes.getValue(RUN_BOOK_ATTRIBUTE_NAME)))) {
			runbookEntryNames.add(entryName);
		} else if ("true".equalsIgnoreCase(StringUtils.trim(attributes.getValue(CONFIGURATION_ITEM_ATTRIBUTE_NAME)))	) {
			configurationItemEntryNames.add(entryName);
		} else {
			logger.debug("Ignoring entry " + entryName + " because it is not a RunBook or a ConfigurationItem");
		}
	}

	private void disableEntries() {
		for (String each : runbookEntryNamesToDisable) {
			if (logger.isDebugEnabled())
				logger.debug("Disabling run book " + each);
			runbookEntryNames.remove(each);
		}

		for (String each : configurationItemsEntryNamesToDisable) {
			if (logger.isDebugEnabled())
				logger.debug("Disabling configuration item " + each);
			configurationItemEntryNames.remove(each);
		}
	}

	private void loadEntries() {
		loadRunBooks();
		loadConfigurationItems();
		repository.loadConfigurationItems(getConfigurationItemTypes(), loadingErrors);
	}

	private void loadRunBooks() {
		for (String each : runbookEntryNames) {
			if (each.endsWith(".class")) {
				loadJavaRunBook(each);
			} else {
				reportError(each + " is declared as a run book factory but is not a Java class or a Python file");
			}
		}
	}

	private void loadJavaRunBook(String entryName) {
		String javaClassName = convertEntryNameToJavaClassName(entryName);
		try {
			RunBook runbook = (RunBook) Class.forName(javaClassName).newInstance();
			runbooks.add(runbook);
			logger.info("Loaded Java run book " + javaClassName);
		} catch (Exception exc) {
			reportError("Cannot load Java run book " + javaClassName, exc);
		}
	}

	private void loadConfigurationItems() {
		for (String each : configurationItemEntryNames) {
			if (each.endsWith(".class")) {
				loadJavaConfigurationItem(each);
			} else {
				reportError(each + " is declared as a configuration item but is not a Java class");
			}
		}
	}

	private void loadJavaConfigurationItem(String entryName) {
		String javaClassName = convertEntryNameToJavaClassName(entryName);
		try {
			configurationItemTypes.add(Class.forName(javaClassName));
			logger.info("Loaded Java configuration item " + javaClassName);
		} catch (Exception exc) {
			reportError("Cannot load Java configuration item " + javaClassName, exc);
		}
	}

	private static String convertEntryNameToJavaClassName(String each) {
		return StringUtils.removeEnd(each, ".class").replace('/', '.');
	}

	private void reportError(String errorMessage, Exception exc) {
		logger.error(errorMessage, exc);
		loadingErrors.add(errorMessage + ": " + exc.toString());
	}

	private void reportError(String errorMessage) {
		logger.error(errorMessage);
		loadingErrors.add(errorMessage);
	}

	public Collection<RunBook> getRunBooks() {
		return runbooks;
	}

	public Collection<Class<?>> getConfigurationItemTypes() {
		return configurationItemTypes;
	}

	public List<String> getLoadingErrors() {
		return Collections.unmodifiableList(loadingErrors);
	}

	private static final Logger logger = LoggerFactory.getLogger(PluginLoader.class);
}
