package com.xebialabs.deployit.jcr;

import com.xebialabs.deployit.exception.RuntimeIOException;
import com.xebialabs.deployit.security.AuthenticationFailureException;
import com.xebialabs.deployit.security.PermissionDeniedException;
import com.xebialabs.deployit.security.UsernameAndPasswordCredentials;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.security.core.context.SecurityContext;
import org.springframework.security.core.context.SecurityContextHolder;
import org.springframework.stereotype.Component;

import javax.jcr.*;
import java.io.IOException;

import static com.xebialabs.deployit.jcr.JcrConstants.ADMIN_PASSWORD;
import static com.xebialabs.deployit.jcr.JcrConstants.ADMIN_USERNAME;


@Component
public class JcrTemplate {

	private final Repository repository;

	@Autowired
	public JcrTemplate(final Repository repository) {
		this.repository = repository;
	}

	public final <T> T executeAsAdmin(final JcrCallback<T> callback) {
		return execute(ADMIN_USERNAME, ADMIN_PASSWORD, callback);
	}

	public final <T> T execute(final JcrCallback<T> callback) {
		SecurityContext context = SecurityContextHolder.getContext();
		UsernameAndPasswordCredentials creds = (UsernameAndPasswordCredentials) context.getAuthentication();
		if (creds == null) {
			throw new AuthenticationFailureException();
		}
		return execute(creds.getUsername(), creds.getPassword(), callback);
	}

	public final <T> T execute(final String username, final String password, final JcrCallback<T> callback) throws RuntimeIOException, RuntimeRepositoryException {
		try {
			Credentials creds = new SimpleCredentials(username, password.toCharArray());
			Session session = repository.login(creds);
			try {
				return callback.doInJcr(session);
			} finally {
				session.logout();
			}
		} catch (IOException e) {
			throw new RuntimeIOException(e);
		} catch (AccessDeniedException ade) {
			throw PermissionDeniedException.withMessage("Failed to complete your request.", ade);
		} catch (LoginException le) {
			throw new AuthenticationFailureException(le, "Incorrect credentials for user %s", username);
		} catch (RepositoryException e) {
			throw new RuntimeRepositoryException(e.toString(), e);
		}
	}

}
