package com.xebialabs.deployit.jcr;

import static com.xebialabs.deployit.ConfigurationItemRoot.APPLICATIONS;
import static com.xebialabs.deployit.ConfigurationItemRoot.ENVIRONMENTS;
import static com.xebialabs.deployit.ConfigurationItemRoot.INFRASTRUCTURE;
import static com.xebialabs.deployit.jcr.JcrConstants.ARCHETYPE_NODETYPE_NAME;
import static com.xebialabs.deployit.jcr.JcrConstants.ARCHETYPE_ROOT_NODE_NAME;
import static com.xebialabs.deployit.jcr.JcrConstants.ARTIFACT_NODETYPE_NAME;
import static com.xebialabs.deployit.jcr.JcrConstants.CONFIGURATION_ITEM_NODETYPE_NAME;
import static com.xebialabs.deployit.jcr.JcrConstants.CONFIGURATION_ITEM_TYPE_PROPERTY_NAME;
import static com.xebialabs.deployit.jcr.JcrConstants.CONFIGURATION_NODETYPE_NAME;
import static com.xebialabs.deployit.jcr.JcrConstants.CONFIGURATION_NODE_ID;
import static com.xebialabs.deployit.jcr.JcrConstants.CONFIGURATION_NODE_NAME;
import static com.xebialabs.deployit.jcr.JcrConstants.DEPLOYIT_NAMESPACE_PREFIX;
import static com.xebialabs.deployit.jcr.JcrConstants.DEPLOYIT_NAMESPACE_URI;
import static com.xebialabs.deployit.jcr.JcrConstants.ID_PROPERTY_NAME;
import static com.xebialabs.deployit.jcr.JcrConstants.LAST_MODIFIED_DATE_PROPERTY_NAME;
import static com.xebialabs.deployit.jcr.JcrConstants.STEP_NODETYPE_NAME;
import static com.xebialabs.deployit.jcr.JcrConstants.TASKS_NODE_NAME;
import static com.xebialabs.deployit.jcr.JcrConstants.TASK_NODETYPE_NAME;

import java.io.File;
import java.io.IOException;
import java.util.Calendar;

import javax.jcr.*;
import javax.jcr.nodetype.NodeTypeManager;
import javax.jcr.nodetype.NodeTypeTemplate;
import javax.jcr.security.AccessControlEntry;
import javax.jcr.security.AccessControlManager;
import javax.jcr.security.AccessControlPolicy;
import javax.jcr.security.AccessControlPolicyIterator;
import javax.jcr.security.Privilege;

import org.apache.jackrabbit.api.JackrabbitRepository;
import org.apache.jackrabbit.api.security.JackrabbitAccessControlList;
import org.apache.jackrabbit.core.RepositoryImpl;
import org.apache.jackrabbit.core.config.RepositoryConfig;
import org.apache.jackrabbit.core.security.principal.EveryonePrincipal;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.DisposableBean;
import org.springframework.beans.factory.FactoryBean;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.beans.factory.annotation.Required;
import org.springframework.core.io.Resource;
import org.xml.sax.InputSource;

import com.xebialabs.deployit.ci.Root;

public class JackrabbitRepositoryFactoryBean implements InitializingBean, FactoryBean<Repository>, DisposableBean {

	private Resource homeDir;

	private Resource configuration;

	private JackrabbitRepository repository;

	private boolean createHomeDirIfNotExists;

	@Override
	public void afterPropertiesSet() throws IOException, RepositoryException {
		InputSource configurationInputSource = new InputSource(configuration.getInputStream());
		File homeDirFile = homeDir.getFile();
		if (!homeDirFile.exists() && !createHomeDirIfNotExists) {
			throw new RepositoryException("Jackrabbit home dir " + homeDirFile + " does not exist");
		}

		RepositoryConfig repositoryConfig = RepositoryConfig.create(configurationInputSource, homeDirFile.getAbsolutePath());
		repository = RepositoryImpl.create(repositoryConfig);
		if (logger.isDebugEnabled())
			logger.debug("Instantiated JackRabbit repository");
	}

	public void configureJcrRepositoryForDeployit() {
		new JcrTemplate(repository).executeAsAdmin(new JcrCallback<Object>() {
			@Override
			public Object doInJcr(final Session session) throws IOException, RepositoryException {
				final NamespaceRegistry namespaceRegistry = session.getWorkspace().getNamespaceRegistry();
				namespaceRegistry.registerNamespace(DEPLOYIT_NAMESPACE_PREFIX, DEPLOYIT_NAMESPACE_URI);

				final NodeTypeManager typeManager = session.getWorkspace().getNodeTypeManager();
				createMixinNodeType(typeManager, CONFIGURATION_ITEM_NODETYPE_NAME);
				createMixinNodeType(typeManager, ARTIFACT_NODETYPE_NAME);
				createMixinNodeType(typeManager, ARCHETYPE_NODETYPE_NAME);

				createRoot(session, APPLICATIONS.getRootNodeName());
				createRoot(session, INFRASTRUCTURE.getRootNodeName());
				createRoot(session, ENVIRONMENTS.getRootNodeName());
				createRoot(session, ARCHETYPE_ROOT_NODE_NAME);

				createMixinNodeType(typeManager, TASK_NODETYPE_NAME);
				createMixinNodeType(typeManager, STEP_NODETYPE_NAME);
				createNode(session, TASKS_NODE_NAME);

				createMixinNodeType(typeManager, CONFIGURATION_NODETYPE_NAME);
				createNode(session, CONFIGURATION_NODE_NAME, CONFIGURATION_NODETYPE_NAME);
				setAccessControlListOnConfigurationNode(session);
				setAccessControlOnRootNode(session);
				return null;
			}

		});
	}

	private void createMixinNodeType(final NodeTypeManager typeManager, final String nodetypeName) throws RepositoryException {
		final NodeTypeTemplate nodeTypeTemplate = typeManager.createNodeTypeTemplate();
		nodeTypeTemplate.setName(nodetypeName);
		nodeTypeTemplate.setQueryable(true);
		nodeTypeTemplate.setAbstract(false);
		nodeTypeTemplate.setMixin(true);

		typeManager.registerNodeType(nodeTypeTemplate, false);
	}

	private void createRoot(final Session session, final String rootNodeName) throws RepositoryException {
		Node node = session.getRootNode().addNode(rootNodeName);
		node.addMixin(JcrConstants.CONFIGURATION_ITEM_NODETYPE_NAME);
		node.setProperty(CONFIGURATION_ITEM_TYPE_PROPERTY_NAME, Root.class.getName());
		node.setProperty(ID_PROPERTY_NAME, rootNodeName);
		node.setProperty(LAST_MODIFIED_DATE_PROPERTY_NAME, Calendar.getInstance());
	}

	private void createNode(final Session session, final String nodeName, final String... mixinNodeTypeNames) throws RepositoryException {
		Node rootNode = session.getRootNode();
		Node addedNode = rootNode.addNode(nodeName);
		for (String each : mixinNodeTypeNames) {
			addedNode.addMixin(each);
		}

		session.save();
	}

	protected void setAccessControlOnRootNode(final Session session) throws RepositoryException {
		AccessControlManager acm = session.getAccessControlManager();
		AccessControlPolicyIterator applicablePolicies = acm.getApplicablePolicies("/");

		boolean policySet = false;
		while (!policySet && applicablePolicies.hasNext()) {
			AccessControlPolicy each = applicablePolicies.nextAccessControlPolicy();
			if (!(each instanceof JackrabbitAccessControlList))
				continue;

			JackrabbitAccessControlList acl = (JackrabbitAccessControlList) each;

			Privilege readPrivilege = acm.privilegeFromName(Privilege.JCR_READ);

			final EveryonePrincipal everyone = EveryonePrincipal.getInstance();
			for (AccessControlEntry entry : acl.getAccessControlEntries()) {
				if (entry.getPrincipal().equals(everyone)) {
					acl.removeAccessControlEntry(entry);
				}
			}

			acl.addEntry(everyone, new Privilege[]{readPrivilege}, false);

			logger.debug("Setting {} on root", acl);
			acm.setPolicy("/", acl);
		}
	}

	protected void setAccessControlListOnConfigurationNode(final Session session) throws RepositoryException {
		AccessControlManager acm = session.getAccessControlManager();
		AccessControlPolicyIterator applicablePolicies = acm.getApplicablePolicies(CONFIGURATION_NODE_ID);

		boolean policySet = false;
		while (!policySet && applicablePolicies.hasNext()) {
			AccessControlPolicy each = applicablePolicies.nextAccessControlPolicy();
			if (!(each instanceof JackrabbitAccessControlList))
				continue;

			JackrabbitAccessControlList acl = (JackrabbitAccessControlList) each;
			for (AccessControlEntry eachEntry : acl.getAccessControlEntries()) {
				acl.removeAccessControlEntry(eachEntry);
			}

			Privilege allPrivilege = acm.privilegeFromName(Privilege.JCR_ALL);
			Privilege readPrivilege = acm.privilegeFromName(Privilege.JCR_READ);

			acl.addEntry(EveryonePrincipal.getInstance(), new Privilege[]{allPrivilege}, false);
			acl.addEntry(EveryonePrincipal.getInstance(), new Privilege[]{readPrivilege}, true);
			acm.setPolicy(CONFIGURATION_NODE_ID, acl);
			policySet = true;
		}

		if (!policySet) {
			throw new IllegalStateException("Could not set permission on preferences node " + CONFIGURATION_NODE_ID);
		}
		session.save();
	}

	@Override
	public Repository getObject() {
		return repository;
	}

	@Override
	public void destroy() {
		repository.shutdown();
	}

	@Override
	public Class<Repository> getObjectType() {
		return Repository.class;
	}

	@Override
	public boolean isSingleton() {
		return true;
	}

	public Resource getHomeDir() {
		return homeDir;
	}

	@Required
	public void setHomeDir(final Resource homeDir) {
		this.homeDir = homeDir;
	}

	public Resource getConfiguration() {
		return configuration;
	}

	@Required
	public void setConfiguration(final Resource configuration) {
		this.configuration = configuration;
	}

	public boolean isCreateHomeDirIfNotExists() {
		return createHomeDirIfNotExists;
	}

	public void setCreateHomeDirIfNotExists(boolean autoCreateRepositoryDir) {
		this.createHomeDirIfNotExists = autoCreateRepositoryDir;
	}

	private static final Logger logger = LoggerFactory.getLogger(JackrabbitRepositoryFactoryBean.class);

}
