package com.xebialabs.deployit.security.principaldata;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.ldap.core.DirContextOperations;
import org.springframework.security.core.userdetails.UsernameNotFoundException;
import org.springframework.security.ldap.authentication.AbstractLdapAuthenticator;
import org.springframework.security.ldap.authentication.BindAuthenticator;
import org.springframework.security.ldap.authentication.LdapAuthenticationProvider;
import org.springframework.security.ldap.search.LdapUserSearch;

import java.lang.reflect.Method;
import java.util.Optional;

/**
 * Provides user data from LDAP by performing user searches.
 * This class interacts with LDAP to fetch user details like email and display name.
 */
class LdapUserDataProvider implements UserDataProvider {
    private static final Logger logger = LoggerFactory.getLogger(LdapUserDataProvider.class);

    private final String mailAttribute = "mail";

    private final String displayNameAttribute = "displayName";

    private LdapUserSearch userSearch;

    /**
     * Constructs an instance using an LdapAuthenticationProvider.
     * This constructor extracts the user search component from the authentication provider.
     *
     * @param authProvider The LdapAuthenticationProvider used for authentication.
     */
    public LdapUserDataProvider(LdapAuthenticationProvider authProvider) {
        try {
            BindAuthenticator authenticator = invokePrivateMethod(authProvider, LdapAuthenticationProvider.class, "getAuthenticator", BindAuthenticator.class);
            this.userSearch = invokePrivateMethod(authenticator, AbstractLdapAuthenticator.class, "getUserSearch", LdapUserSearch.class);
        } catch (Exception ex) {
            logger.warn("Error constructing LdapDataProviderBase", ex);
        }
    }

    /**
     * Constructs an instance directly with an LdapUserSearch.
     * This constructor is used when an LdapUserSearch component is directly available.
     *
     * @param userSearch The LdapUserSearch component for searching users.
     */
    public LdapUserDataProvider(LdapUserSearch userSearch) {
        this.userSearch = userSearch;
    }

    /**
     * Invokes a private method using reflection.
     * This method is a utility for accessing private methods of a given object.
     *
     * @param target      The object containing the method.
     * @param targetClass The class of the target object.
     * @param methodName  The name of the method to invoke.
     * @param returnType  The return type of the method.
     * @param <T>         The type of the target object.
     * @param <R>         The return type of the method.
     * @return The result of invoking the method.
     * @throws Exception If reflection fails or the method is not accessible.
     */
    private static <T, R> R invokePrivateMethod(Object target, Class<T> targetClass, String methodName, Class<R> returnType) throws Exception {
        Method method = targetClass.getDeclaredMethod(methodName);
        method.setAccessible(true);
        return returnType.cast(method.invoke(target));
    }

    /**
     * Fetches user data from LDAP based on the username.
     * This method searches LDAP for the user and extracts their email and display name.
     *
     * @param username The username of the user to search for.
     * @return UserData object containing the user's email and display name, or UserData.NOT_FOUND if not found.
     */
    @Override
    public UserData getUserData(String username) {
        return searchForUserData(username).map(data -> {
            String email = LdapDataHelper.readStringAttribute(username, data, mailAttribute);
            String fullName = LdapDataHelper.readStringAttribute(username, data, displayNameAttribute);
            return new UserData(email, fullName);
        }).orElse(UserData.NOT_FOUND);
    }

    /**
     * Searches for a user in LDAP and returns their directory context.
     * This method uses the configured LdapUserSearch to find the user's LDAP entry.
     *
     * @param username The username of the user to search for.
     * @return An Optional containing the DirContextOperations if the user is found, or empty if not found.
     */
    private Optional<DirContextOperations> searchForUserData(String username) {
        try {
            return Optional.ofNullable(userSearch.searchForUser(username));
        } catch (UsernameNotFoundException e) {
            logger.info("User with username: {} not found", username);
        } catch (Exception e) {
            logger.warn("Error accessing LDAP server", e);
        }
        return Optional.empty();
    }
}
