package com.xebialabs.deployit.service.deployment

import com.xebialabs.deployit.core.{EncryptedStringValue, StringValue}
import com.xebialabs.deployit.plugin.api.udm._
import com.xebialabs.deployit.plugin.api.udm.base.BaseDeployable
import com.xebialabs.deployit.plugin.api.validation.ValidationMessage._
import com.xebialabs.deployit.provision.resolver.placeholder.MissingTemplatePlaceholderCollector
import com.xebialabs.deployit.service.replacement.{ConsolidatedDictionary, DictionaryValueException}

import scala.jdk.CollectionConverters._
import scala.collection.mutable
import scala.util.{Failure, Success, Try}


class ProvisioningPlaceholderPropertyResolverDeployedGenerator(nextGenerator: DeployedGenerator) extends DeployedGenerator {
  private val cardinalityFieldName = "cardinality"

  def findMissingPlaceholdersInTemplates(deployedApplication: DeployedApplication, templates: mutable.Set[Template], dictionary: ConsolidatedDictionary): Unit = {
    val missingTemplatePlaceholderCollector = new MissingTemplatePlaceholderCollector(dictionary)
    templates.foreach(t => missingTemplatePlaceholderCollector.missingPlaceholders(deployedApplication, t))
    val children = templates.flatMap(_.childTemplates.asScala.toSet)
    if (children.nonEmpty) {
      findMissingPlaceholdersInTemplates(deployedApplication, children, dictionary)
    }
  }

  override def generateDeployed(deploymentContext: DeploymentContext, deployable: Deployable, container: Container): GeneratedDeployeds = {
    val deployeds = new GeneratedDeployeds()
    val deployedApplication = deploymentContext.getDeployedApplication
    deployable match {
      case provisionable: BaseProvisionable =>
        val templates: mutable.Set[Template] = provisionable.getBoundTemplates.asScala ++ provisionable.getProvisioners.asScala.map(_.getHostTemplate)
        findMissingPlaceholdersInTemplates(deployedApplication, templates, deploymentContext.dictionaryFilteredByContainer(container))
        resolveCardinality(provisionable, deploymentContext.dictionaryFilteredByContainer(container)).recover {
          case _: DictionaryValueException =>
            deployedApplication.get$validationMessages.add(warn(provisionable.getId, cardinalityFieldName, "Could not resolve the deployable value \"" + provisionable.getCardinality + "\" using the dictionary."))
          case nfe: NumberFormatException => throw new IllegalArgumentException("cardinality must be resolved to integer", nfe)
        }
        deployeds.merge(nextGenerator.generateDeployed(deploymentContext, provisionable, container))
        deployeds
      case _ =>
        findMissingPlaceholdersInTemplates(deployedApplication, deployable.asInstanceOf[BaseDeployable].getBoundTemplates.asScala, deploymentContext.dictionaryFilteredByContainer(container))
        deployeds.merge(nextGenerator.generateDeployed(deploymentContext, deployable, container))
        deployeds
    }
  }

  private def resolveCardinality(provisionable: BaseProvisionable, consolidatedDictionary: ConsolidatedDictionary): Try[Int] = {
    Try(consolidatedDictionary.resolve(Option(provisionable.getCardinality).getOrElse(1), provisionable.getType.getDescriptor.getPropertyDescriptor(cardinalityFieldName))) match {
      case Success(_: EncryptedStringValue) =>
        Failure(new scala.IllegalArgumentException("Currently, we do not support encrypted values for cardinality"))
      case Success(value: StringValue) =>
        Try(value.toPublicFacingValue.toInt)
      case Failure(e) => Failure(e)
    }
  }
}
