package com.xebialabs.deployit.security.principaldata;

import org.springframework.security.core.GrantedAuthority;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;

import static com.xebialabs.deployit.security.principaldata.UserData.NOT_FOUND;

/**
 * Implementation of the PrincipalDataProvider interface.
 * This class is responsible for aggregating user data from multiple sources.
 */
class PrincipalDataProviderImpl implements PrincipalDataProvider {

    private final List<UserDataProvider> userProviders;

    public PrincipalDataProviderImpl() {
        userProviders = new ArrayList<>();
    }

    public void addUserProvider(UserDataProvider provider) {
        this.userProviders.add(provider);
    }

    public void addUserProvider(List<UserDataProvider> providers) {
        userProviders.addAll(providers);
    }

    /**
     * Retrieves user data for a given username from the available providers.
     * Iterates through each provider until user data is found or returns NOT_FOUND if no data is available.
     *
     * @param username The username to search for across all providers.
     * @return UserData instance with user details or NOT_FOUND if no data is available.
     */
    @Override
    public UserData getUserData(final String username) {
        return userProviders.stream()
                .filter(provider -> provider.getUserData(username) != NOT_FOUND)
                .map(provider -> provider.getUserData(username))
                .findFirst()
                .orElse(NOT_FOUND);
    }
    public Collection<? extends GrantedAuthority> getAuthorities(final String username) {
        return this.userProviders.stream()
                .map(provider -> provider.getAuthorities(username))
                .filter(authorities -> !authorities.isEmpty())
                .findFirst()
                .orElse(Collections.emptySet());
    }

}
