package com.xebialabs.deployit.service.dependency

import com.xebialabs.deployit.plugin.api.deployment.specification.Operation
import com.xebialabs.deployit.plugin.api.deployment.specification.Operation.{DESTROY, MODIFY}
import com.xebialabs.deployit.plugin.api.semver.VersionRange
import com.xebialabs.deployit.plugin.api.udm.DeploymentPackage.DependencyResolution
import com.xebialabs.deployit.plugin.api.udm.{DeployedApplication, DeploymentPackage => UdmPackage, Version => UdmVersion}
import com.xebialabs.deployit.repository.RepositoryService
import com.xebialabs.deployit.service.dependency.DependencyConstraints.DependencyException
import com.xebialabs.deployit.service.dependency.DeployedApplicationsFinder.DeployedApplications
import com.xebialabs.deployit.service.dependency.Implicits._
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Component

import java.util
import scala.collection.mutable
import scala.jdk.CollectionConverters._
import scala.util.Try

@Component
class DependencyService @Autowired()(dependencyFinder: DependencyFinder,
                                     val repositoryService: RepositoryService,
                                     deployedApplicationsFinder: DeployedApplicationsFinder) extends DeployedApplicationInfoLoader {

  def resolveMultiDeployment(envId: String, udmVersion: UdmVersion, dependencyResolution: String): List[List[DeploymentResult]] = {
    resolveMultiDeployment(envId, udmVersion, findDependencies(udmVersion, envId, dependencyResolution))
  }

  def findDependencyGroupsToDeleteAndValidate(deployedApplication: DeployedApplication): List[List[DeployedApplication]] = {
    val deployedApplications: DeployedApplications = deployedApplicationsFinder.allDeployedApplicationsWithDependencies(deployedApplication.getEnvironment.getId)
    validateCanBeUndeployed(deployedApplication, deployedApplications)
    loadDependencyGroupsToUndeploy(deployedApplication, deployedApplications)
  }

  private def loadDependencyGroupsToUndeploy(deployedApplication: DeployedApplication,
                                             deployedApplications: DeployedApplications): List[List[DeployedApplication]] = {
    val deployedApps: List[List[DeployedApplicationInfo]] = if (deployedApplication.isUndeployDependencies) {
      deployedApplications.findRemovableDeployedApplicationsGroups(deployedApplication)
    } else {
      Nil
    }
    deployedApps.map(group => group.map(deployedInfo => repositoryService.read[DeployedApplication](deployedInfo.deployedApplicationId)))
  }

  private def validateCanBeUndeployed(deployedApplication: DeployedApplication, deployedApplications: DeployedApplications): Unit = {
    deployedApplications.findDependentApplications(deployedApplication).headOption.foreach { dependant =>
      val name: String = dependant.applicationId.name
      val version: String = dependant.applicationDependencies(deployedApplication.getVersion.getApplication.getName)
      val appName: String = deployedApplication.getVersion.getApplication.getName
      throw new DependencyException(
        s"""Application "$appName" cannot be undeployed, because the deployed application "$name" depends on its current version. It requires a version compatible with '$version'.""")
    }
  }

  private[dependency] def resolveMultiDeployment(envId: String, main: UdmVersion, dependencies: List[List[UdmVersion]]): List[List[DeploymentResult]] = {
    val allDeployedApplications = loadDeployedApplications(envId)
    val deps: mutable.Map[String, Map[String, String]] = mutable.Map(allDeployedApplications.map(i => i.applicationId -> i.applicationDependencies): _*)

    def validate(udmVersion: UdmVersion): Unit = validateDependenciesNotBroken(udmVersion, deps.toMap, MODIFY) { v =>
      Try(v.map(r => new VersionRange(r)))
        // Found an application range. The version should be in the range.
        .map(_.forall(_.includes(parseVersion(udmVersion.getVersion))))
        .recover {
          // Not a valid range. The version should be the same
          case _: IllegalArgumentException => v.getOrElse(throw new IllegalStateException(s"Cannot validate version $v")).equals(udmVersion.getVersion)
        }.getOrElse(throw new IllegalStateException(s"Cannot validate version $v"))
    }

    def isMain(udmVersion: UdmVersion) = udmVersion == main

    val results: List[List[DeploymentResult]] = (List(main) +: dependencies).map { versions =>
      versions.map { v =>
        validate(v)
        if (deps.contains(v.getApplication.getId)) {
          deps.put(v.getApplication.getId, extractDeps(v))
        }
        DeploymentResult(v.getId, allDeployedApplications.find(fto => fto.applicationId == v.getApplication.getId), isMain(v))
      }
    }

    def isMainApplication(group: util.List[DeploymentResult]): Boolean = {
      if (group.size() == 1) {
        val headMaybe: Option[DeploymentResult] = group.asScala.headOption
        return headMaybe.exists(head => !head.isMain && head.deployedDependency.exists(deployedDependency => deployedDependency.versionId == head.udmVersionId))
      }
      false
    }

    results.collect { case group if !isMainApplication(group.asJava) => group }.reverse
  }

  private[this] def extractDeps(v: UdmVersion): Map[String, String] = v match {
    case udmVer: UdmPackage => udmVer.getApplicationDependencies.asScala.toMap
    case _ => Map()
  }

  private[this] def validateDependenciesNotBroken(udmVersion: UdmVersion, dependencies: Map[String, Map[String, String]], op: Operation)(isValid: Option[String] => Boolean): Unit = {
    dependencies.foreach { case (appId, deps) =>
      if (!isValid(deps.get(udmVersion.getApplication.getName))) {
        val version = deps(udmVersion.getApplication.getName)
        throw new DependencyException(s"""Application "${udmVersion.getApplication.getId}" cannot be ${opToVerb(op)}, because the deployed application "$appId" depends on its current version. It requires a version compatible with '$version'.""")
      }
    }
  }

  private[this] def opToVerb(op: Operation): String = op match {
    case MODIFY => "upgraded"
    case DESTROY => "undeployed"
    case _ => "deployed"
  }

  private[dependency] def findDependencies(udmVersion: UdmVersion, envId: String, dependencyResolution: String): List[List[UdmVersion]] = udmVersion match {
    case pck: UdmPackage =>
      val dependencies = dependencyResolution.toUpperCase match {
        case DependencyResolution.EXISTING =>
          val deployedApplications: DeployedApplications = deployedApplicationsFinder.allDeployedApplicationsWithDependencies(envId)
          dependencyFinder.find(pck, findExistingDeployedApplication(deployedApplications))
        case DependencyResolution.LATEST => dependencyFinder.find(pck)
        case s => throw new IllegalArgumentException(s"'$s' is not a valid value for DependencyResolution. Please check the value of the DependencyResolution property in ${udmVersion.getId}.")
      }
      DependencyConstraints.validateApplicationGraph(dependencies)
      dependencies.allRequiredVersionsInTopologicalOrder
    case _ => Nil
  }

  private[dependency] def findExistingDeployedApplication(deployedApplications: DeployedApplications)(appId: String, versions: List[UdmPackage], versionRange: VersionRange): Option[UdmPackage] = {
    deployedApplications.findApplicationInfo(appId)
      .map(_.versionId)
      .map(strAfterLast('/'))
      .flatMap(s => Try(parseVersion(s)).toOption)
      .flatMap(v => v in versionRange)
      .flatMap(vers => versions.find(pck => pck.toOsgi == vers))
  }

  private[this] def strAfterLast(c: Char)(s: String): String = {
    s.lastIndexOf(c) match {
      case -1 => s
      case i => s.substring(i + 1)
    }
  }

}
