package com.xebialabs.deployit.service.deployment.setter;

import com.google.common.base.Splitter;
import com.google.common.base.Strings;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import com.xebialabs.deployit.core.EncryptedStringValue;
import com.xebialabs.deployit.core.StringValue;
import com.xebialabs.deployit.plugin.api.reflect.PropertyDescriptor;
import com.xebialabs.deployit.plugin.api.reflect.PropertyKind;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import com.xebialabs.deployit.repository.RepositoryService;
import com.xebialabs.deployit.service.replacement.ConsolidatedDictionary;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.Collection;
import java.util.Map;

import static com.google.common.collect.Lists.newArrayList;
import static com.xebialabs.deployit.checks.Checks.checkArgument;
import static com.xebialabs.deployit.plugin.api.validation.ValidationMessage.warn;

public abstract class AbstractPropertySetter {
    private static final Logger logger = LoggerFactory.getLogger(AbstractPropertySetter.class);

    private RepositoryService repository;

    AbstractPropertySetter(RepositoryService repository) {
        this.repository = repository;
    }

    public abstract void setProperty(ConfigurationItem currentTarget, ConfigurationItem previousTarget, ConfigurationItem currentSource,
                              ConfigurationItem previousSource, ConsolidatedDictionary dictionary, PropertyDescriptor targetPropertyDescriptor, PropertyDescriptor sourcePropertyDescriptor);

    Object getValueIfExists(ConfigurationItem configurationItem, String name) {
        if (configurationItem == null || !configurationItem.hasProperty(name)) {
            return null;
        }

        return configurationItem.getProperty(name);
    }

    private boolean optionalKind(PropertyDescriptor deployablePropDesc, PropertyKind kind) {
        return deployablePropDesc == null || deployablePropDesc.getKind() == kind;
    }

    @SuppressWarnings("unchecked")
    private Collection<ConfigurationItem> getCollectionOfCis(Object valueToSet) {
        Collection<ConfigurationItem> cis = newArrayList();
        for (String s : (Collection<String>) valueToSet) {
            cis.add(repository.read(s));
        }
        return cis;
    }

    private Collection<String> splitValue(Object valueToSet) {
        checkArgument((valueToSet instanceof String) || (valueToSet instanceof StringValue), "Should be String or StringValue to split.");
        Splitter splitter = Splitter.on(",").omitEmptyStrings();
        return newArrayList(splitter.split(valueToSet.toString()));
    }

    Object convertIfNeeded(Object valueToSet, PropertyDescriptor deployablePropDesc, PropertyDescriptor deployedPropDesc) {
        if (valueToSet == null) return null;
        if (optionalKind(deployablePropDesc, PropertyKind.STRING) && deployedPropDesc.getKind() == PropertyKind.CI) {
            return repository.read(valueToSet.toString());
        } else if (optionalKind(deployablePropDesc, PropertyKind.SET_OF_STRING) && deployedPropDesc.getKind() == PropertyKind.SET_OF_CI) {
            return Sets.newHashSet(getCollectionOfCis(valueToSet));
        } else if (optionalKind(deployablePropDesc, PropertyKind.STRING) && deployedPropDesc.getKind() == PropertyKind.SET_OF_CI) {
            return Sets.newHashSet(getCollectionOfCis(splitValue(valueToSet)));
        } else if (optionalKind(deployablePropDesc, PropertyKind.LIST_OF_STRING) && deployedPropDesc.getKind() == PropertyKind.LIST_OF_CI) {
            return Lists.newArrayList(getCollectionOfCis(valueToSet));
        } else if (optionalKind(deployablePropDesc, PropertyKind.STRING) && deployedPropDesc.getKind() == PropertyKind.LIST_OF_CI) {
            return Lists.newArrayList(getCollectionOfCis(splitValue(valueToSet)));
        } else if (deployedPropDesc.getKind() == PropertyKind.STRING) {
            if (valueToSet instanceof EncryptedStringValue && !deployedPropDesc.isPassword()) {
                logger.warn("Going to write a Encrypted value as plaintext in \"{}\", as it is not a password field.", deployedPropDesc);
            }
            return valueToSet.toString();
        } else if (valueToSet instanceof StringValue) {
            return valueToSet.toString();
        }
        return valueToSet;
    }

    boolean isNullOrEmpty(final Object value, final PropertyDescriptor pd) {
        if (value == null) return true;
        switch (pd.getKind()) {
            case BOOLEAN:
            case INTEGER:
            case STRING:
            case ENUM:
            case DATE:
            case CI:
                return Strings.isNullOrEmpty(value.toString());
            case SET_OF_STRING:
            case SET_OF_CI:
            case LIST_OF_STRING:
            case LIST_OF_CI:
                return ((Collection<?>) value).isEmpty();
            case MAP_STRING_STRING:
                return ((Map<?, ?>) value).isEmpty();
            default:
                throw new IllegalStateException("Unsupported property kind: " + pd.getKind() + " for property: " + pd.getFqn());
        }
    }

    void setDeployedFromDictionary(ConfigurationItem deployed, ConsolidatedDictionary dictionary,
                                   PropertyDescriptor deployedPropDesc, PropertyDescriptor deployablePropDesc) {
        if (dictionary.containsKey(deployedPropDesc.getFqn())) {
            Object value = dictionary.get(deployedPropDesc.getFqn());
            value = convertIfNeeded(value, deployablePropDesc, deployedPropDesc);
            deployedPropDesc.set(deployed, value);
        }
    }

    void setSilently(ConfigurationItem ci, PropertyDescriptor pd, Object value) {
        try {
            pd.set(ci, value);
        } catch (RuntimeException exc) {
            logger.error("Could not convert (resolved) value to correct type for property {} of {}. Error message was: {}", pd, ci, exc.getMessage());
            ci.get$validationMessages().add(warn(ci.getId(), pd.getName(), "Could not correctly convert resolved value, property empty"));
        }
    }
}
