package com.xebialabs.deployit.service.deployment

import java.util.{Set => JSet}

import com.xebialabs.deployit.engine.api.dto.Deployment
import com.xebialabs.deployit.plugin.api.deployment.ResolvedPlaceholder
import com.xebialabs.deployit.plugin.api.reflect.Type
import com.xebialabs.deployit.plugin.api.udm._
import com.xebialabs.deployit.plugin.api.udm.base.BaseDeployedArtifact
import com.xebialabs.deployit.service.replacement.ConsolidatedDictionary
import com.xebialabs.deployit.service.replacement.Dictionaries.of
import grizzled.slf4j.Logging

import scala.jdk.CollectionConverters._

object ResolvedPlaceholderGenerator extends Logging {

  def generateAndAdd(deployment: Deployment, dictionary: ConsolidatedDictionary): Unit = {
    val deployedApplication = deployment.getDeployedApplication.asInstanceOf[DeployedApplication]
    deployment.addResolvedPlaceholders(generate(deployedApplication, dictionary))
  }

  def generateAndAdd(deployment: Deployment): Unit = {
    val deployedApplication: DeployedApplication = deployment.getDeployedApplication.asInstanceOf[DeployedApplication]
    //main deployment artifacts placeholders
    val resolvedPlaceholders = generate(deployedApplication, getBaseDeployedArtifacts(deployment))
    deployment.addResolvedPlaceholders(resolvedPlaceholders)
    deployedApplication.set$resolvedPlaceholders(deployment.getResolvedPlaceholders)
    //application dependencies artifacts placeholders
    deployment.getRequiredDeployments.forEach(generateAndAdd)
  }

  def generate(deployedApplication: DeployedApplication, dictionary: ConsolidatedDictionary,
               containerId: String): JSet[ResolvedPlaceholder] =
    dictionary.getResolvedPlaceholders.asScala.foldLeft(Set[ResolvedPlaceholder]()){ case (acc, (key, value)) =>
      if (dictionary.getWrapped.get(key) != null) {
        acc + newResolvedPlaceholder(key, value, deployedApplication, dictionary, containerId)
      } else {
        logger.info(s"Placeholder for key `$key` has not been saved")
        acc
      }
    }.asJava

  def generate(deployedApplication: DeployedApplication,
               deployeds: List[BaseDeployedArtifact[DeployableArtifact, Container]]): JSet[ResolvedPlaceholder] = {
    val version = deployedApplication.getVersion
    val environment = deployedApplication.getEnvironment
    val dictionaries = of(environment).filterBy(version)
    var knownDictionaries = Map[Container, ConsolidatedDictionary]()

    deployeds.flatMap { artifact =>
      artifact.getPlaceholders.asScala.map { case (key, value) =>
        val container = artifact.getContainer
        val dictionary: ConsolidatedDictionary = knownDictionaries.getOrElse(container, {
          val newDictionary = dictionaries.filterBy(container).consolidate()
          knownDictionaries += (container -> newDictionary)
          newDictionary
        })
        Option(dictionary.getWrapped.get(key)).map(_ =>
          newResolvedPlaceholder(key, value, deployedApplication, dictionary, container.getId))
      }
    }.toSet.flatten.asJava
  }

  private def getBaseDeployedArtifacts(deployment: Deployment): List[BaseDeployedArtifact[DeployableArtifact, Container]] = {
    val bda = classOf[BaseDeployedArtifact[DeployableArtifact, Container]]
    deployment.getDeployeds.asScala
      .filter(_.getType.instanceOf(Type.valueOf(bda)))
      .map(_.asInstanceOf[BaseDeployedArtifact[DeployableArtifact, Container]]).toList
  }

  private[deployment] def generate(deployedApplication: DeployedApplication,
                                   dictionary: ConsolidatedDictionary): JSet[ResolvedPlaceholder] =
    generate(deployedApplication, dictionary, "")

  private def newResolvedPlaceholder(key: String, _value: String, deployedApplication: DeployedApplication,
                                     dictionary: ConsolidatedDictionary, containerId: String): ResolvedPlaceholder = {
    val isEncrypted = dictionary.getWrapped.get(key).isEncrypted
    val value = if (isEncrypted) "" else _value
    val dictionaryId = dictionary.getDictionaryId(key)
    val environmentId = deployedApplication.getEnvironment.getId
    val versionId = deployedApplication.getVersion.getId

    ResolvedPlaceholder(isEncrypted, key, value, containerId, containerDeleted = false,
      deployedApplication.getId, dictionaryId, dictionaryDeleted = false,
      environmentId, environmentDeleted = false, versionId)()
  }
}
