package com.xebialabs.deployit.security.policy

import com.xebialabs.deployit.security.principaldata.PrincipalDataProviderHolder
import com.xebialabs.deployit.service.profile.XldUserProfileServiceHolder
import grizzled.slf4j.Logging
import org.springframework.security.authentication.{RememberMeAuthenticationToken}
import org.springframework.security.core.Authentication
import org.springframework.stereotype.Component

/**
 * Default implementation of the UserProfileCreationPolicy interface.
 * This policy acts as a catch-all for user profile creation, ensuring that a profile is created or updated
 * for every successful authentication, regardless of specific conditions.
 */
@Component
class DefaultUserProfileCreationPolicy
  extends UserProfileCreationPolicy with Logging {

  /**
   * Specifies the order in which this policy is evaluated.
   * Being a catch-all, it has a lower priority compared to other policies.
   *
   * @return The order of this policy.
   */
  override def order(): Int = 1000 // Is the default catch-all policy, should be last to try out

  /**
   * Determines if this policy applies to the given authentication.
   * This implementation always returns true, indicating it applies to all authentications.
   *
   * @param authentication The authentication object resulting from a successful authentication attempt.
   * @return True, indicating the policy always applies.
   */
  override def policyApplies(authentication: Authentication): Boolean = true // catch-all

  /**
   * Creates or updates a user profile based on the given authentication object.
   * If the principal is a String (internal user), it logs the login event.
   * Otherwise, it retrieves user data from the PrincipalDataProvider and creates or updates the user profile.
   *
   * @param authentication The authentication object for which to create or update the user profile.
   */
  override def createProfile(authentication: Authentication): Unit = {
    if (authentication == null || authentication.getPrincipal == null) {
      logger.info("Authentication or principal is null")
      return
    }
    authentication match {
      case rememberMe: RememberMeAuthenticationToken =>
        logger.info(s"Handling Remember Me User ${rememberMe.getName} LoginEvent")
        XldUserProfileServiceHolder.getXldUserProfileService.loginAllowedUserStatus(rememberMe.getName)
        logger.info(s"Login allowed for ${rememberMe.getName}")
        XldUserProfileServiceHolder.getXldUserProfileService.registerLoginTime(rememberMe.getName)
      case _ =>
        authentication.getPrincipal match {
          case internalUser: String =>
            logger.info(s"Handling Internal User $internalUser LoginEvent")
            XldUserProfileServiceHolder.getXldUserProfileService.loginAllowedUserStatus(internalUser)
            logger.info(s"Login allowed for $internalUser")
            XldUserProfileServiceHolder.getXldUserProfileService.registerLoginTime(internalUser)
          case _ =>
                val userData = PrincipalDataProviderHolder.getPrincipalDataProvider.getUserData(authentication.getName)
              if (userData == null) {
                logger.info(s"Invalid user data for ${authentication.getName}")
              } else {
                XldUserProfileServiceHolder.getXldUserProfileService.createOrUpdateUserProfile(userData.getFullName, userData.getEmail, authentication.getName)
              }
        }
    }
  }
}
