package com.xebialabs.deployit.listener.authentication

import com.xebialabs.deployit.booter.local.utils.Strings
import com.xebialabs.deployit.security.policy.UserProfileCreationPolicy
import com.xebialabs.deployit.security.SecurityServiceLocator
import com.xebialabs.deployit.security.permission.PlatformPermissions
import org.slf4j.LoggerFactory
import org.springframework.context.ApplicationListener
import org.springframework.security.authentication.DisabledException
import org.springframework.security.authentication.event.AuthenticationSuccessEvent
import org.springframework.security.core.Authentication
import org.springframework.security.core.context.SecurityContextHolder
import org.springframework.stereotype.Component

import java.util.{List => JList}
import scala.jdk.CollectionConverters._

/**
 * Listens for successful authentication events and triggers user profile creation or update.
 * This class is a component that listens to authentication success events within the application.
 * Upon receiving such an event, it performs additional checks and potentially triggers the creation
 * or update of a user profile based on the configured policies.
 */
@Component
class AuthenticationSuccessEventListener(userProfileCreationPolicies: JList[UserProfileCreationPolicy]) extends ApplicationListener[AuthenticationSuccessEvent] {

  private val logger = LoggerFactory.getLogger(classOf[AuthenticationSuccessEventListener])

  /**
   * Handles the AuthenticationSuccessEvent by checking permissions and applying user profile creation policies.
   * This method is called when an authentication success event is published. It checks if the authenticated
   * user has login permission. If so, it finds and applies the appropriate user profile creation policy.
   *
   * @param event The authentication success event.
   */
  override def onApplicationEvent(event: AuthenticationSuccessEvent): Unit = {
    val authentication: Authentication = event.getAuthentication
    if (authentication != null && Strings.isNotEmpty(authentication.getName)) {
      val authFromContext: Authentication = SecurityContextHolder.getContext.getAuthentication
      try {
        if (authFromContext == null) {
          SecurityContextHolder.getContext.setAuthentication(authentication)
        }
        val hasLoginPermission: Boolean = SecurityServiceLocator.getPermissionEnforcer.hasPermission(authentication, PlatformPermissions.LOGIN)
        if (!hasLoginPermission) {
          throw new DisabledException("The user " + authentication.getName + " or its group(s) do not have login permission")
        }
        val creationPolicyOption = findApplicableUserCreationPolicy(authentication)
        creationPolicyOption match {
          case Some(creationPolicy) => creationPolicy.createProfile(authentication)
          case None => logger.info("No applicable user profile creation policy found for authentication: {}", authentication.getName)
        }
      } catch {
        case e: Exception =>
          logger.warn("An error occurred during authentication success event handling for user: {}, {}", authentication.getName, e.getMessage)
      } finally {
        SecurityContextHolder.getContext.setAuthentication(authFromContext)
      }
    }
  }

  /**
   * Finds the applicable user profile creation policy for the given authentication.
   * This method iterates through the available user profile creation policies, sorted by their order,
   * and returns the first one that applies to the given authentication.
   *
   * @param authentication The authentication for which to find the applicable policy.
   * @return The applicable UserProfileCreationPolicy.
   * @throws IllegalStateException If no applicable policy is found.
   */
  private def findApplicableUserCreationPolicy(authentication: Authentication): Option[UserProfileCreationPolicy] =
    userProfileCreationPolicies.asScala
      .sortBy(_.order())
      .find(_.policyApplies(authentication))
}
