package com.xebialabs.deployit.service.profile

import com.xebialabs.deployit.exception.NotFoundException
import com.xebialabs.deployit.security.repository.XldUserProfileRepository
import com.xebialabs.deployit.security.model.XldUserProfile
import com.xebialabs.deployit.security.sql.SqlUserService
import com.xebialabs.license.LicenseProperty
import com.xebialabs.license.service.LicenseService
import org.slf4j.LoggerFactory.getLogger
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.security.authentication.BadCredentialsException
import org.springframework.stereotype.Service

import java.util.Date

/**
 * The `XldUserProfileService` class is responsible for managing user profiles within the application.
 * It handles operations such as registering login times and creating or updating user profiles,
 * ensuring these actions comply with the application's licensing constraints. This class interacts
 * with both internal and external user data, providing a centralized point for user profile management.
 *
 * @param xldUserProfileRepository Repository for accessing and manipulating user profile data.
 * @param userService Service for accessing user details, particularly to check for the existence of internal users.
 * @param licenseService Service for checking licensing constraints related to user creation and management.
 */
@Service
@Autowired
class XldUserProfileService(xldUserProfileRepository: XldUserProfileRepository, userService: SqlUserService,
                            licenseService: LicenseService) {

  private val logger = getLogger(classOf[XldUserProfileService])

  /**
   * Registers the last login time for a user by updating their profile with the current timestamp.
   * This method is synchronized to ensure thread safety when updating the login time concurrently for multiple users.
   *
   * @param username The username of the user whose login time is being registered.
   */
  def registerLoginTime(username: String): Unit = {
    xldUserProfileRepository.updateLastActive(username, new Date)
  }

  /**
   * Creates a new user profile or updates an existing one based on the provided full name and email.
   * It performs checks to ensure no internal user exists with the same username and adheres to licensing constraints
   * for creating new external users. This method handles both creation of new profiles and updates to existing ones,
   * ensuring data consistency and compliance with application policies.
   *
   * @param fullName The full name of the user for the profile.
   * @param email The email address of the user for the profile.
   * @param userName The username associated with the profile.
   * @throws BadCredentialsException If an internal user exists with the same username or if license limits are reached.
   */
  def createOrUpdateUserProfile(fullName: String, email: String, userName: String): Unit = {
    try {
      // if authentication is of an external user (e.g. LDAP, Crowd), then we need to make sure, that
      // internal user with same username does not exist
      userService.read(userName)
      logger.info("Unable to create profile for user [{}]. A local account exists.", userName)
      throw new BadCredentialsException(String.format("A local account exists for user %s", userName))
    } catch {
      case _: NotFoundException =>
        val profile = xldUserProfileRepository.findOne(userName)
        if (profile.isEmpty) {
          if (!hasLicenseToCreateNewExternalUser) {
            logger.info("Maximum number of users allowed by license has been reached. Cannot create external user profile for user [{}]", userName)
            throw new LicenseLimitReachedException(String.format("Maximum number of users allowed by license has been reached. Cannot create external user profile for user %s", userName))
          }
          val userProfile = new XldUserProfile(userName.toLowerCase, true, Set.empty, fullName, email, true, new Date, false)
          xldUserProfileRepository.createUserProfile(userProfile, updateLastActive = true)
        } else {
          val existingProfile = profile.get
          if (!equals(existingProfile.email, email) || !equals(existingProfile.fullName, fullName)) {
            xldUserProfileRepository.updateExternalUserProfile(fullName, email, new Date, userName)
          } else {
            xldUserProfileRepository.updateLastActive(userName, new Date)
          }
        }
    }
  }

  /**
   * Compares two strings for equality, considering nulls to be equal to each other.
   * This method is used to determine if profile information has changed and needs updating.
   *
   * @param str1 the first string to compare
   * @param str2 the second string to compare
   * @return true if the strings are equal or both null, false otherwise
   */
  private def equals(str1: String, str2: String): Boolean = {
    Option(str1) == Option(str2)
  }

  /**
   * Checks if the current license allows for the creation of a new external user.
   * This method reads the maximum number of users allowed from the license and compares it
   * against the current number of users with login allowed to determine if a new user can be added.
   *
   * @return true if a new external user can be created, false if the license limit has been reached
   */
  private def hasLicenseToCreateNewExternalUser: Boolean = {
    val maxNumberOfUsers = licenseService.getLicense.getStringValue(LicenseProperty.MAX_NUMBER_OF_USERS)
    if (maxNumberOfUsers != null) {
      val countUserWithLoginAllowed = xldUserProfileRepository.countUserWithLoginAllowed()
      Integer.parseInt(maxNumberOfUsers) > countUserWithLoginAllowed
    } else {
      true
    }
  }
}

class LicenseLimitReachedException(message: String) extends Exception(message)
