package com.xebialabs.deployit.spring

import com.xebialabs.deployit.engine.api.distribution.TaskExecutionWorkerRepository
import com.xebialabs.deployit.engine.spi.services.RepositoryFactory
import com.xebialabs.deployit.engine.tasker._
import com.xebialabs.deployit.engine.tasker.distribution.WorkerManager
import com.xebialabs.deployit.engine.tasker.distribution.versioning.ConfigurationHashProvider
import com.xebialabs.deployit.engine.tasker.log.{StepLogFactory, StepLogRetriever}
import com.xebialabs.deployit.engine.tasker.query.QueryActor
import com.xebialabs.deployit.engine.tasker.repository.{ActiveTaskRepository, PendingTaskRepository}
import com.xebialabs.deployit.engine.tasker.websockets.RemoteChangeSetEventListener
import com.xebialabs.deployit.inspection.service.discovery.{DiscoveryService, DiscoveryWorker}
import com.xebialabs.xlplatform.settings.shared.TaskerSettings
import org.apache.pekko.actor._
import org.slf4j.LoggerFactory
import org.springframework.beans.factory.annotation._
import org.springframework.context.annotation._
import org.springframework.jms.config.JmsListenerContainerFactory
import org.springframework.jms.listener.DefaultMessageListenerContainer

import scala.util._

@Configuration
@Lazy
@DependsOn(Array("repositoryServiceHolder", "placeholderRepositoryHolder", "artifactResolver", "commandWhitelistValidator"))
class EngineBeanBuilder(@Autowired val archive: Archive,
                        @Autowired val repositoryAdapter: RepositoryFactory) {

  private val logger = LoggerFactory.getLogger(classOf[EngineBeanBuilder])

  override def toString: TaskId = super.toString
  @Value("${deploy.task.in-process-worker:true}") var inProcessTaskEngine: Boolean = _
  @Value("${deploy.task.resilient:false}") var taskResilientEnabled: Boolean = _

  val wrappedWorkerManager: BeanWrapper[ActorRef] = BeanWrapper[ActorRef]()
  val wrappedEngine: BeanWrapper[TaskExecutionEngine] = BeanWrapper[TaskExecutionEngine]()
  val wrappedDiscoveryService: BeanWrapper[DiscoveryService] = BeanWrapper[DiscoveryService]()

  @Bean
  def workerManager(): BeanWrapper[ActorRef] = {
    wrappedWorkerManager
  }

  @Bean
  @DependsOn(Array("upgrader"))
  def engine(): BeanWrapper[TaskExecutionEngine] = {
    wrappedEngine
  }

  @Bean
  def discoveryService(): BeanWrapper[DiscoveryService] = {
    wrappedDiscoveryService
  }

  def initWorkerManager(system: ActorSystem, taskExecutionWorkerRepository: TaskExecutionWorkerRepository,
                    configurationHashProvider: ConfigurationHashProvider): Unit = {

    val workerManger: ActorRef = EngineBeanBuilder.createWorkerManager(system, taskExecutionWorkerRepository,
      configurationHashProvider.configurationHash, inProcessTaskEngine)
    wrappedWorkerManager.set(workerManger)
  }

  def initEngine(implicit system: ActorSystem,
              activeTaskRepository: ActiveTaskRepository,
              pendingTaskRepository: PendingTaskRepository,
              containerFactory: JmsListenerContainerFactory[DefaultMessageListenerContainer],
              taskExecutionWorkerRepository: TaskExecutionWorkerRepository,
              stepLogFactory: StepLogFactory,
              stepLogRetriever: StepLogRetriever,
              configurationHashProvider: ConfigurationHashProvider,
              taskerSettings: TaskerSettings,
              taskQueueService: TaskQueueService): Unit = {
    val taskFinalizer = new TaskFinalizerImpl(system, activeTaskRepository, pendingTaskRepository, taskerSettings.askTimeout)
    val engine = new TaskExecutionEngine(activeTaskRepository, pendingTaskRepository, taskExecutionWorkerRepository,
      taskQueueService, system, workerManager().get(), taskFinalizer, taskResilientEnabled)
    if (inProcessTaskEngine) {
      logger.info("An in-process task execution worker will be registered.")
      TaskExecutionWorker
        .local(repositoryAdapter, archive, configurationHashProvider, activeTaskRepository, pendingTaskRepository, taskExecutionWorkerRepository,
          containerFactory, taskerSettings, stepLogFactory, stepLogRetriever)
        .recoverTasks()
      DiscoveryWorker.initialize(system)
    } else {
      logger.debug("Registering for remote change set events.")
      system.actorOf(RemoteChangeSetEventListener.props(), RemoteChangeSetEventListener.name)
      logger.debug("Registering for remote deployment state events.")
      system.actorOf(RemoteDeploymentStateEventListener.props(), RemoteDeploymentStateEventListener.name)
    }
    system.actorOf(QueryActor.props(), QueryActor.name)
    rescheduleTasks(pendingTaskRepository, engine)
    wrappedEngine.set(engine)
  }

  def initDiscoveryService(system: ActorSystem): Unit =
    wrappedDiscoveryService.set(new DiscoveryService(system, workerManager().get(), inProcessTaskEngine))

  private def rescheduleTasks(pendingTaskRepository: PendingTaskRepository, engine: TaskExecutionEngine): Unit = {
    pendingTaskRepository.scheduledTasks().forEach { task =>
      val scheduledDate = task.getScheduledDate
      val date = scheduledDate.toDateTimeISO.toString("yyyy-MM-dd'T'HH:mm:ss")
      if (scheduledDate.isBeforeNow) {
        logger.info(s"Scheduled task [${task.getId}] with date [$date] has already passed so executing now.")
        engine.execute(task.getId)
      } else {
        Try {
          engine.schedule(task.getId, scheduledDate)
        } match {
          case Failure(ex: TaskerException) =>
            logger.warn(s"Scheduled task [${task.getId}] with start date [$date] cannot be rescheduled automatically.")
            logger.warn(ex.getMessage)
          case Failure(ex: Exception) =>
            logger.warn(s"Scheduled task [${task.getId}] with start date [$date] cannot be rescheduled automatically. Unexpected error occurred.")
            throw ex
          case Success(_) =>
            logger.info(s"Scheduled task [${task.getId}] with start date [$date] was rescheduled")
        }
      }
    }
  }
}

object EngineBeanBuilder {

  def createWorkerManager(system: ActorSystem, taskExecutionWorkerRepository: TaskExecutionWorkerRepository,
                          configurationHash: String, shouldNotChunk: Boolean): ActorRef =
    system.actorOf(
      WorkerManager.props(taskExecutionWorkerRepository, configurationHash, shouldNotChunk),
      WorkerManager.name
    )
}
