package com.xebialabs.deployit.service.importer

import java.io.{File, IOException}
import java.lang.reflect.Modifier
import java.util

import ai.digital.configuration.central.deploy.ClientProperties
import ai.digital.deploy.core.common.security.permission.DeployitPermissions
import com.google.common.collect.Lists
import com.google.common.collect.Lists.newArrayList
import com.google.common.collect.Sets.newHashSet
import com.google.common.io.Files
import com.xebialabs.deployit.checksum.ChecksumAlgorithmProvider
import com.xebialabs.deployit.core.sql.RepositoryHelper
import com.xebialabs.deployit.engine.spi.command.{CreateCisCommand, RepositoryBaseCommand}
import com.xebialabs.deployit.io.ArtifactFileUtils
import com.xebialabs.deployit.packager.DarPackager
import com.xebialabs.deployit.plugin.api.reflect.PropertyKind.{LIST_OF_CI, SET_OF_CI}
import com.xebialabs.deployit.plugin.api.reflect.{PropertyDescriptor, PropertyKind, Type}
import com.xebialabs.deployit.plugin.api.udm._
import com.xebialabs.deployit.plugin.api.udm.artifact.SourceArtifact
import com.xebialabs.deployit.plugin.api.validation.ValidationMessage
import com.xebialabs.deployit.repository.{RepositoryService, SearchParameters}
import com.xebialabs.deployit.security.permission.Permission
import com.xebialabs.deployit.security.{PermissionDeniedException, RoleService}
import com.xebialabs.deployit.server.api.importer._
import com.xebialabs.deployit.server.api.util.IdGenerator
import com.xebialabs.deployit.service.validation.Validator
import com.xebialabs.overthere.local.LocalFile
import com.xebialabs.xldeploy.packager.placeholders.PlaceholdersUtil.SourceArtifactUtil
import com.xebialabs.xldeploy.packager.placeholders.SourceArtifactScanner
import com.xebialabs.xlplatform.artifact.resolution.ArtifactResolverRegistry
import de.schlichtherle.truezip.file.TFile
import javax.annotation.PostConstruct
import nl.javadude.scannit.Scannit
import org.slf4j.LoggerFactory
import org.springframework.beans.factory.annotation.Autowired

import scala.jdk.CollectionConverters._

class ImporterServiceImpl @Autowired()(var repositoryService: RepositoryService,
                                       var roleService: RoleService,
                                       var validator: Validator,
                                       var scanner: SourceArtifactScanner,
                                       var clientConfiguration: ClientProperties,
                                       var repositoryHelper: RepositoryHelper,
                                       var checksumAlgorithmProvider: ChecksumAlgorithmProvider) extends ImporterService {
  private val importers: util.List[Importer] = newArrayList()
  private val logger = LoggerFactory.getLogger(classOf[ImporterServiceImpl])
  private val isCollectionKind = (kind: PropertyKind) => (kind eq SET_OF_CI) || (kind eq LIST_OF_CI)
  private var importablePackageDirectory: File = new File(clientConfiguration.getImportablePackageDirectory)

  @PostConstruct
  def initImporters(): Unit = {
    val importerClasses = Scannit.getInstance.getSubTypesOf(classOf[Importer])
    logger.debug("Found importers: {}", importerClasses)
    importerClasses.asScala.filter(cls =>
      !(cls.isInterface || Modifier.isAbstract(cls.getModifiers) || cls == classOf[XmlManifestDarImporter])
    ).foreach { importerClass =>
      try {
        logger.debug("Importer {} registered.", importerClass)
        importers.add(importerClass.newInstance)
      } catch {
        case e: Exception =>
          throw new IllegalStateException("Could not instantiate importer: " + importerClass, e)
      }
    }
    importers.sort((o1: Any, o2: Any) => o1.getClass.getSimpleName.compareTo(o2.getClass.getSimpleName))

    importers.add(new XmlManifestDarImporter(repositoryService))
    logger.debug("Importer {} registered.", classOf[XmlManifestDarImporter])
    logger.info("Importers configured in XL Deploy: {}", importers)
  }

  override def getImportablePackageDirectory: File = importablePackageDirectory

  def setImportablePackageDirectory(importablePackageDirectory: File): Unit =
    this.importablePackageDirectory = importablePackageDirectory

  override def listPackages: util.List[String] = {
    Lists.newArrayList((for (
      importer <- importers.asScala
      if importer.isInstanceOf[ListableImporter];
      pkgFound <- importer.asInstanceOf[ListableImporter].list(importablePackageDirectory).asScala
    ) yield pkgFound).sorted.asJava)
  }

  override def importPackage(source: ImportSource): String = {
    try {
      importers.asScala.find(_.canHandle(source)) match {
        case Some(importer) =>
          doImport(source, importer)
        case None =>
          throw new ImporterException("The selected file does not have the expected format for an importable package")
      }
    } finally {
      source.cleanUp()
    }
  }

  protected def checkPermission(permission: Permission, onConfigurationItems: String): Unit = {
    if (!permission.getPermissionHandler.hasPermission(onConfigurationItems)) {
      throw PermissionDeniedException.forPermission(permission, onConfigurationItems)
    }
  }

  private def doImport(source: ImportSource, importer: Importer): String = {
    val ctx: ImportingContext = new DefaultImportingContext
    val packageInfo: PackageInfo = importer.preparePackage(source, ctx)
    try {
      val upgrade: Boolean = isUpgrade(packageInfo)
      checkPermission(upgrade, packageInfo)
      checkImported(packageInfo)
      val importedPackage: ImportedPackage = importer.importEntities(packageInfo, ctx)
      scanPlaceholders(importedPackage, ctx)
      val toCreate: util.Set[ConfigurationItem] = newHashSet()
      createEntities(importedPackage, upgrade, toCreate)
      validate(toCreate)
      if (upgrade) {
        importedPackage.getVersion.setApplication(repositoryService.read(packageInfo.getApplicationId))
      }
      val event: RepositoryBaseCommand = new CreateCisCommand(Lists.newArrayList(toCreate))
      repositoryHelper.publishCommand(event)
      importedPackage.getVersion.getId
    } finally {
      importer.cleanUp(packageInfo, ctx)
    }
  }

  private def scanPlaceholders(importedPackage: ImportedPackage, ctx: ImportingContext): Unit = {
    if (importedPackage.getVersion.isInstanceOf[DeploymentPackage]) {
      ctx.getAttribute[util.List[TFile]]("temporaryFiles").add(new TFile(Files.createTempDir()))
      getAllArtifacts(importedPackage.getDeployables).asScala.foreach { sourceArtifact =>
        if (ArtifactFileUtils.hasRealOrResolvedFile(sourceArtifact)) {
          logger.info("Artifact {} is a resolved artifact", sourceArtifact)
          if (!isArtifactPreEnriched(sourceArtifact)) {
            sourceArtifact.addChecksumAndScan(scanner, () => checksumAlgorithmProvider.getMessageDigest)
          }
        }
        else {
          logger.info(s"Resolving artifact $sourceArtifact from url ${sourceArtifact.getFileUri}")
          resolveAndEnrichArtifact(sourceArtifact)
        }
      }
    }
  }

  private def resolveAndEnrichArtifact(sourceArtifact: SourceArtifact): Unit = {
    try {
      import com.xebialabs.xlplatform.utils.ResourceManagement.using

      using(ArtifactResolverRegistry.resolve(sourceArtifact)) { resolve =>
        using(resolve.openStream) { is =>
          if (!isArtifactPreEnriched(sourceArtifact)) {
            sourceArtifact.addChecksumAndScan(scanner, resolve.getFileName, is, () => checksumAlgorithmProvider.getMessageDigest)
          }
          sourceArtifact.setFile(LocalFile.valueOf(new File(resolve.getFileName)))
        }
      }
    } catch {
      case _: IOException =>
        throw new ImporterException(s"Could not open stream of uri ${sourceArtifact.getFileUri} of unresolved artifact $sourceArtifact")
    }
  }

  private def isArtifactPreEnriched(deployable: SourceArtifact): Boolean = {
    val isPreEnrichedArtifact =
      deployable.hasCheckSum && deployable.hasProperty(DarPackager.PRESCANNED_PLACEHOLDERS_PROPERTYNAME) && deployable.getProperty[Boolean](DarPackager.PRESCANNED_PLACEHOLDERS_PROPERTYNAME)
    logger.info(s"Artifact $deployable is ${if (isPreEnrichedArtifact) "" else "not"} pre-enriched")
    isPreEnrichedArtifact
  }


  private def checkImported(packageInfo: PackageInfo): Unit = {
    val id = IdGenerator.generateId(packageInfo.getApplicationId, packageInfo.getApplicationVersion)
    if (repositoryService.exists(id)) {
      throw new ImporterException(
        "Already imported version %s of application %s",
        packageInfo.getApplicationVersion, packageInfo.getApplicationName)
    }

    val dirs = subStringBeforeLast(packageInfo.getApplicationId, "/")
    if (!repositoryService.exists(dirs)) {
      throw new ImporterException(
        "The directory structure [%s] specified for the import of application [%s] does not exist.",
        dirs, appName(packageInfo))
    }
  }

  private def createEntities(importedPackage: ImportedPackage, isUpgrade: Boolean,
                             toCreateCollector: util.Set[ConfigurationItem]): Unit = {
    if (!isUpgrade) {
      toCreateCollector.add(importedPackage.getApplication)
    }
    val version = importedPackage.getVersion
    toCreateCollector.add(version)
    for (pd <- filterExistingPackages(version) if isCollectionKind(pd.getKind)) {
      val cis = pd.get(version).asInstanceOf[util.Collection[ConfigurationItem]]
      toCreateCollector.addAll(newHashSet(cis))
      cis.forEach(ci => createNestedConfigurationItems(ci, toCreateCollector))
    }
  }

  private def filterExistingPackages(version: Version): Set[PropertyDescriptor] = {
    val propertyDescriptors = version.getType.getDescriptor.getPropertyDescriptors.asScala.toSet
    if (!version.isInstanceOf[CompositePackage]) {
      propertyDescriptors
    } else {
      propertyDescriptors.filterNot(_.getName == "packages")
    }
  }

  private def createNestedConfigurationItems(ci: ConfigurationItem, toCreateCollector: util.Set[ConfigurationItem]): Unit = {
    val propertyDescriptors = ci.getType.getDescriptor.getPropertyDescriptors
    for (propertyDescriptor <- propertyDescriptors.asScala
         if isCollectionKind(propertyDescriptor.getKind) && !propertyDescriptor.getName.toLowerCase.contains("dependencies");
         embedded <- propertyDescriptor.get(ci).asInstanceOf[util.Collection[ConfigurationItem]].asScala
         if !toCreateCollector.contains(embedded)) {
      toCreateCollector.add(embedded)
      createNestedConfigurationItems(embedded, toCreateCollector)
    }
  }

  private def validate(toCreate: util.Set[ConfigurationItem]): Unit = {
    val msgs = toCreate.asScala.foldLeft[Seq[ValidationMessage]](Seq.empty) { (acc: Seq[ValidationMessage], tc: ConfigurationItem) =>
      val validated: Seq[ValidationMessage] = validator.validate(tc, newArrayList(toCreate)).asScala.toSeq
      if (validated.nonEmpty) {
        acc ++ validated
      } else {
        acc
      }
    }

    if (msgs.nonEmpty) {
      throw new ImporterException("Import failed with the following validation errors %s", msgs.mkString(", "))
    }
  }

  private def isUpgrade(packageInfo: PackageInfo): Boolean = {
    val locationSpecified: Boolean = packageInfo.getApplicationName.contains("/")

    val applications = repositoryService.list(new SearchParameters()
      .setType(Type.valueOf(classOf[Application]))
      .setName(appName(packageInfo))
      .setAncestor(packageInfo.getApplicationRoot)).asScala.toList

    applications match {
      case Nil =>
        false
      case app :: Nil if locationSpecified && !(app.getId == packageInfo.getApplicationId) =>
        throw new ImporterException(
          "The manifest contains the path [%s] to import the application into, but the application exists at the path [%s]",
          packageInfo.getApplicationId, app.getId)
      case app :: Nil =>
        val appId = app.getId
        packageInfo.setDirectories(
          appId.substring(packageInfo.getApplicationRoot.length, appId.length - packageInfo.getApplicationName.length)
        )
        true
      case app :: _ =>
        throw new IllegalStateException(
          s"Found more than 1 [${app.getType}] with the same name: [${applications.mkString(", ")}]")
    }
  }

  private def appName(packageInfo: PackageInfo): String = subStringAfterLast(packageInfo.getApplicationId, "/")

  private def subStringAfterLast(s: String, sep: String): String =
    if (!s.contains(sep)) s else s.substring(s.lastIndexOf(sep) + sep.length)

  private def subStringBeforeLast(s: String, sep: String): String =
    if (!s.contains(sep)) s else s.substring(0, s.lastIndexOf(sep))

  private def checkPermission(isUpgrade: Boolean, packageInfo: PackageInfo): Unit = {
    val requiredPermission = if (isUpgrade) DeployitPermissions.IMPORT_UPGRADE else DeployitPermissions.IMPORT_INITIAL
    checkPermission(requiredPermission, packageInfo.getApplicationId)
  }

}
