package com.xebialabs.deployit.service.deployment;

import com.google.common.base.Predicate;
import com.google.common.collect.Sets;
import com.xebialabs.deployit.plugin.api.udm.Container;
import com.xebialabs.deployit.plugin.api.udm.Deployable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;

import java.util.Set;

import static com.google.common.base.Strings.isNullOrEmpty;
import static com.google.common.collect.Sets.filter;
import static com.google.common.collect.Sets.newHashSet;

@Component
public class TagMatcher {

    public boolean areMatching(Deployable deployable, Container container) {

        Predicate<String> NOT_EMPTY_STRING = input -> !isNullOrEmpty(input);

        Set<String> deployableTags = emptySetIfNull(filter(deployable.getTags(), NOT_EMPTY_STRING));
        Set<String> containerTags = emptySetIfNull(filter(container.getTags(), NOT_EMPTY_STRING));

        logger.debug("Using tags: deployable {} and container {}", deployableTags, containerTags);

        boolean bothEmpty = bothEmpty(deployableTags, containerTags);
        boolean tagMatching = haveAMatchingTag(deployableTags, containerTags);
        boolean containsStar = hasStarWildcard(deployableTags) || hasStarWildcard(containerTags);
        boolean containsPlus = hasPlusWildcard(deployableTags, containerTags) || hasPlusWildcard(containerTags, deployableTags);
        return bothEmpty || tagMatching || containsStar || containsPlus;
    }

    private boolean hasPlusWildcard(Set<String> tags, Set<String> otherTags) {
        return tags.contains("+") && !otherTags.isEmpty();
    }

    private boolean hasStarWildcard(Set<String> tags) {
        return tags.contains("*");
    }

    private boolean haveAMatchingTag(Set<String> deployableTags, Set<String> containerTags) {
        return !Sets.intersection(deployableTags, containerTags).isEmpty();
    }

    private boolean bothEmpty(Set<String> deployableTags, Set<String> containerTags) {
        return (deployableTags.isEmpty() && containerTags.isEmpty());
    }

    private Set<String> emptySetIfNull(Set<String> tags) {
        if (tags == null) {
            return newHashSet();
        }
        return tags;
    }

    private static final Logger logger = LoggerFactory.getLogger(TagMatcher.class);
}
