package com.xebialabs.deployit.service.deployment;

import ai.digital.deploy.task.status.queue.TaskPathStatusListener;
import ai.digital.deploy.tasker.common.TaskType;
import com.google.common.collect.Collections2;
import com.google.common.collect.FluentIterable;
import com.google.common.collect.Sets;
import com.xebialabs.deployit.deployment.planner.*;
import com.xebialabs.deployit.deployment.rules.PlanCreationContextFactory;
import com.xebialabs.deployit.deployment.service.ArtifactTransformerFactory;
import com.xebialabs.deployit.engine.api.dto.Deployment;
import com.xebialabs.deployit.engine.api.execution.TaskPackageDependency;
import com.xebialabs.deployit.engine.spi.execution.ExecutionStateListener;
import com.xebialabs.deployit.engine.tasker.PhaseContainer;
import com.xebialabs.deployit.engine.tasker.TaskSpecification;
import com.xebialabs.deployit.plugin.api.deployment.specification.Delta;
import com.xebialabs.deployit.plugin.api.deployment.specification.DeltaSpecification;
import com.xebialabs.deployit.plugin.api.deployment.specification.Operation;
import com.xebialabs.deployit.plugin.api.reflect.Descriptor;
import com.xebialabs.deployit.plugin.api.reflect.PropertyDescriptor;
import com.xebialabs.deployit.plugin.api.reflect.PropertyKind;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.plugin.api.services.Repository;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import com.xebialabs.deployit.plugin.api.udm.Deployed;
import com.xebialabs.deployit.plugin.api.udm.DeployedApplication;
import com.xebialabs.deployit.plugin.api.udm.EmbeddedDeployed;
import com.xebialabs.deployit.plugin.api.udm.base.BaseConfigurationItem;
import com.xebialabs.deployit.plugin.overthere.Host;
import com.xebialabs.deployit.repository.RepositoryAdapterFactory;
import com.xebialabs.deployit.repository.WorkDir;
import com.xebialabs.deployit.security.Permissions;
import com.xebialabs.deployit.task.WorkdirCleanerTrigger;
import com.xebialabs.xlplatform.config.ConfigurationHolder;
import com.xebialabs.xlplatform.satellite.Satellite;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Component;

import java.io.StringWriter;
import java.util.*;
import java.util.stream.Stream;

import static ai.digital.deploy.tasker.common.TaskMetadata.*;
import static ai.digital.deploy.tasker.common.TaskType.*;
import static com.google.common.base.Predicates.instanceOf;
import static com.google.common.base.Predicates.or;
import static com.google.common.collect.FluentIterable.from;
import static com.google.common.collect.Lists.newArrayList;
import static com.xebialabs.deployit.checks.Checks.checkArgument;
import static com.xebialabs.deployit.deployment.planner.MultiDeltaSpecification.forUndeploy;
import static com.xebialabs.deployit.deployment.planner.MultiDeltaSpecificationHelper.applicationNames;
import static com.xebialabs.deployit.deployment.planner.MultiDeltaSpecificationHelper.commonOperation;
import static com.xebialabs.deployit.task.TaskMetadataModifier.putMetadata;
import static java.util.stream.Collectors.toList;

@Component("deploymentService")
public class DeploymentService {

    @Value("${deploy.task.step.on-copy-artifact.enable-retry:false}")
    private String enableRetry;

    private final Planner planner;
    private final RepositoryAdapterFactory repositoryFactory;
    private final PlanCreationContextFactory planCreationContextFactory;
    private final ArtifactTransformerFactory artifactTransformerFactory;

    @Autowired
    public DeploymentService(Planner planner,
                             RepositoryAdapterFactory repositoryFactory,
                             PlanCreationContextFactory planCreationContextFactory,
                             ArtifactTransformerFactory artifactTransformerFactory) {
        this.planner = planner;
        this.repositoryFactory = repositoryFactory;
        this.planCreationContextFactory = planCreationContextFactory;
        this.artifactTransformerFactory = artifactTransformerFactory;
    }

    public MultiDeltaSpecification prepareInitialSpecification(final Deployment deployment, final Map<String, DeployedApplication> existingDepoyedApplications) {
        return MultiDeltaSpecification.withDependencies(
                prepareInitialSpecification(deployment),
                prepareDependencyDeployments(deployment.getGroupedRequiredDeployments(), existingDepoyedApplications));
    }

    private DeltaSpecification prepareInitialSpecification(final Deployment deployment) {
        return prepareInitialSpecification(extractDeployedApplicationWithDeployeds(deployment));
    }

    DeltaSpecification prepareInitialSpecification(final DeployedApplication deployedApplication) {
        DeltaSpecificationBuilder builder = DeltaSpecificationBuilder.newSpecification().initial(deployedApplication);
        DeploymentOperationCalculator.calculate(builder, Sets.newHashSet(), deployedApplication.getDeployeds());
        return builder.build();
    }

    public MultiDeltaSpecification prepareUpgradeSpecification(final Deployment deployment, final Map<String, DeployedApplication> existingDepoyedApplications) {
        return MultiDeltaSpecification.withDependencies(
                prepareUpgradeSpecification(deployment, existingDepoyedApplications.get(deployment.getDeployedApplication().getId())),
                prepareDependencyDeployments(deployment.getGroupedRequiredDeployments(), existingDepoyedApplications));
    }

    private DeltaSpecification prepareUpgradeSpecification(final Deployment deployment, final DeployedApplication existingDeployedApplication) {
        return prepareUpgradeSpecification(extractDeployedApplicationWithDeployeds(deployment), existingDeployedApplication);
    }

    DeltaSpecification prepareUpgradeSpecification(final DeployedApplication newDeployedApplication, final DeployedApplication existingDeployedApplication) {
        // The existingDeployedApplication does not have any 'transient' properties set. These properties are used for setting
        // deployment behaviour, and in case of rollback are taken from the now 'existing' deployed application.
        copyTransientProperties(newDeployedApplication, existingDeployedApplication);
        final DeltaSpecificationBuilder builder = DeltaSpecificationBuilder.newSpecification().upgrade(existingDeployedApplication, newDeployedApplication);

        logger.trace("Incoming Deployeds {}", newDeployedApplication.getDeployeds());
        logger.trace("Existing Deployeds {}", existingDeployedApplication.getDeployeds());


        if (newDeployedApplication.isForceRedeploy()) {
            DeploymentOperationCalculator.deltaToRedeploy(builder, existingDeployedApplication.getDeployeds(), newDeployedApplication.getDeployeds());
        } else {
            DeploymentOperationCalculator.calculate(builder, existingDeployedApplication.getDeployeds(), newDeployedApplication.getDeployeds());
        }
        return builder.build();
    }

    private List<List<DeltaSpecification>> prepareDependencyDeployments(List<List<Deployment>> groupedRequiredDeployments, final Map<String, DeployedApplication> existingDepoyedApplications) {
        List<List<DeltaSpecification>> groupedDeltaSpecs = new ArrayList<>();
        for (List<Deployment> requiredDeploymentGroup : groupedRequiredDeployments) {
            groupedDeltaSpecs.add(requiredDeploymentGroup.stream().map(requiredDeployment -> prepareDependencyDeployment(requiredDeployment, existingDepoyedApplications)).collect(toList()));
        }
        return groupedDeltaSpecs;
    }

    private DeltaSpecification prepareDependencyDeployment(Deployment requiredDeployment, final Map<String, DeployedApplication> existingDepoyedApplications) {
        switch (requiredDeployment.getDeploymentType()) {
            case INITIAL:
                return prepareInitialSpecification(requiredDeployment);
            case UPDATE:
                return prepareUpgradeSpecification(requiredDeployment, existingDepoyedApplications.get(requiredDeployment.getDeployedApplication().getId()));
            default:
                throw new IllegalArgumentException("Deployment " + requiredDeployment.getId() + " has invalid type " + requiredDeployment.getDeploymentType());
        }
    }

    private static DeployedApplication extractDeployedApplicationWithDeployeds(Deployment deployment) {
        DeployedApplication deployedApp = (DeployedApplication) deployment.getDeployedApplication();
        deployedApp.addDeployeds(getDeployeds(deployment));
        return deployedApp;
    }

    @SuppressWarnings("rawtypes")
    private static HashSet<Deployed> getDeployeds(Deployment deployment) {
        Collection<ConfigurationItem> deployedEntities = deployment.getDeployeds();
        FluentIterable<ConfigurationItem> from = from(deployedEntities);
        checkArgument(from.allMatch(or(instanceOf(Deployed.class), instanceOf(EmbeddedDeployed.class))), "The Deployment can only contain Deployed or EmbeddedDeployed configuration items");
        return Sets.newHashSet(from.filter(Deployed.class));
    }

    private void copyTransientProperties(DeployedApplication newDeployment, DeployedApplication existingDeployment) {
        Descriptor descriptor = newDeployment.getType().getDescriptor();
        logger.debug("Copying transient properties to previous deployed application for {}", newDeployment.getId());
        for (PropertyDescriptor pd : descriptor.getPropertyDescriptors()) {
            if (!pd.isTransient()) {
                logger.trace("Skipping copy of non-transient property {}", pd.getFqn());
                continue;
            }

            if (pd.get(newDeployment) == null || pd.get(existingDeployment) != null) {
                logger.trace("Skipping copy of property {} as it is either not set on 'new', or already set on 'existing'", pd.getFqn());
                continue;
            }

            logger.debug("Copying transient property {} to existing deployment", pd.getFqn());
            pd.set(existingDeployment, pd.get(newDeployment));
        }
    }

    @SuppressWarnings("rawtypes")
    public MultiDeltaSpecification prepareUndeployment(final DeployedApplication deployedApplication) {
        return new MultiDeltaSpecification(prepareSingleUndeployment(deployedApplication));
    }

    public MultiDeltaSpecification prepareUndeploymentWithDependencies(final Deployment deployment) {
        //TODO refactor this to more elegant code
        DeployedApplication deployedApplication = (DeployedApplication) deployment.getDeployedApplication();
        List<DeltaSpecification> deltaSpecifications = new ArrayList<>();
        DeltaSpecification specification = prepareSingleUndeployment(deployedApplication);
        deltaSpecifications.add(specification);
        List<List<DeltaSpecification>> dependencies = dependencyStream(deployment).map(x -> x.stream().map(rd -> prepareSingleUndeployment((DeployedApplication) rd.getDeployedApplication())).collect(toList())).collect(toList());
        dependencies.add(deltaSpecifications);
        return forUndeploy(dependencies, deployedApplication);
    }

    private Stream<List<Deployment>> dependencyStream(Deployment deployment) {
        return deployment.getGroupedRequiredDeployments().stream();
    }

    DeltaSpecification prepareSingleUndeployment(final DeployedApplication deployedApplication) {
        DeltaSpecificationBuilder builder = DeltaSpecificationBuilder.newSpecification().undeploy(deployedApplication);
        DeploymentOperationCalculator.calculate(builder, deployedApplication.getDeployeds(), Sets.newHashSet());
        return builder.build();
    }

    TaskSpecification getTaskSpecification(String deploymentId, DeltaSpecification deltaSpecification, WorkDir currentWorkDir, WorkDir... workdirsToCleanup) {
        return getTaskFullSpecification(deploymentId, new MultiDeltaSpecification(deltaSpecification), currentWorkDir, workdirsToCleanup);
    }

    public TaskSpecification getTaskFullSpecification(String deploymentId, MultiDeltaSpecification spec, WorkDir currentWorkDir, WorkDir... workdirsToCleanup) {
        TaskSpecification taskSpec = null;
        Operation operation = commonOperation(spec.getAllDeltaSpecifications()).orElse(Operation.NOOP);
        List<DeployedApplication> allDeployedApplications = spec.getAllDeployedApplications();
        DeployedApplication mainDeployedApplication = allDeployedApplications.get(allDeployedApplications.size() - 1);
        switch (operation) {
            case CREATE:
                taskSpec = getTaskSpecification(deploymentId, "Initial deployment of " + applicationNames(spec), spec, currentWorkDir,
                        mainDeployedApplication, workdirsToCleanup);
                addMetadata(taskSpec, mainDeployedApplication, INITIAL);
                break;
            case DESTROY:
                taskSpec = getTaskSpecification(deploymentId, "Undeployment of " + applicationNames(spec), spec, currentWorkDir,
                        mainDeployedApplication, workdirsToCleanup);
                addMetadata(taskSpec, mainDeployedApplication, UNDEPLOY);
                break;
            case MODIFY:
                String description = "Update deployment of " + applicationNames(spec);
                taskSpec = getTaskSpecification(deploymentId, description, spec, currentWorkDir, mainDeployedApplication, workdirsToCleanup);
                addMetadata(taskSpec, mainDeployedApplication, UPGRADE);
                break;
            case NOOP:
                break;
        }

        return taskSpec;
    }

    private TaskSpecification getTaskSpecification(final String taskId, final String description, final MultiDeltaSpecification specification, WorkDir currentWorkDir, DeployedApplication mainDeployedApplication, WorkDir... workdirsToCleanup) {
        unsetTokensForRealityPush(specification);
        Repository repository = repositoryFactory.create(currentWorkDir);
        PhasedPlan plan = planner.plan(specification, planCreationContextFactory.createPlanCreationContext(repository, artifactTransformerFactory));

        logger.info("Generated plan for task {}:\n{}", taskId, plan.writePlan(new StringWriter()));

        PhaseContainer executionBlock = Plans.toBlockBuilder(plan).build();

        boolean isRollback = specification.getAllDeltaSpecifications().stream().allMatch(DeltaSpecification::isRollback);
        TaskSpecification spec = new TaskSpecification(taskId, description, Permissions.getAuthentication(), currentWorkDir, executionBlock, null, true, true, isRollback, mainDeployedApplication.getOnSuccessPolicy(), mainDeployedApplication.getOnFailurePolicy());
        String autoDetectSetting = "deploy.task.artifact-copy-strategy.autodetect";
        HashMap<String, String> configMap = new HashMap<>();
        if (ConfigurationHolder.get() != null) {
            String hasAutoDetect = ConfigurationHolder.get().hasPath(autoDetectSetting) ?
                    Boolean.toString(ConfigurationHolder.get().getBoolean(autoDetectSetting)) : "false";
            configMap.put(autoDetectSetting, hasAutoDetect);
        }
        spec.setConfig(configMap);
        spec.getMetadata().put("enableCopyArtifactRetry", enableRetry);
        spec.getListeners().addAll(plan.getListeners());
        ArrayList<WorkDir> workDirs = newArrayList(workdirsToCleanup);
        workDirs.add(currentWorkDir);
        spec.getListeners().add(new WorkdirCleanerTrigger(workDirs));
        spec.getListeners().add(new ReferentialIntegrityTrigger(specification));
        spec.getListeners().add(createCheckpointManagerListener(specification, plan));
        spec.getListeners().add(new TaskPathStatusListener());

        addTaskRefs(specification, spec, mainDeployedApplication);

        return spec;
    }

    private void addTaskRefs(final MultiDeltaSpecification specification, final TaskSpecification spec, DeployedApplication mainDeployedApplication) {
        Stream<DeployedApplication> dependencies = specification.getAllDeltaSpecifications().stream().
                filter(spec1 -> !isSameApplication(spec1.getDeployedApplication(), mainDeployedApplication))
                .map(spec1 -> {
                    if (spec1.getDeployedApplication() != null) {
                        return spec1.getDeployedApplication();
                    } else {
                        return spec1.getPreviousDeployedApplication();
                    }
                });
        dependencies
                .map(d -> new TaskPackageDependency(d.getName(), d.getVersion().getVersion()))
                .forEach(td -> spec.getPackageDependencies().add(td));
    }

    protected ExecutionStateListener createCheckpointManagerListener(MultiDeltaSpecification specification, Plan plan) {
        return new CheckPointManagerListener(specification, plan.findCheckpoints());
    }

    private static void unsetTokensForRealityPush(final MultiDeltaSpecification fullSpec) {
        for (DeployedApplication deployedApplication : fullSpec.getAllDeployedApplications()) {
            deployedApplication.set$token(null);
        }

        for (Delta delta : fullSpec.getAllDeltas()) {
            ConfigurationItem ci = delta.getDeployed();
            unsetTokenOnCi(ci);
        }
    }

    private boolean isSameApplication(DeployedApplication application1, DeployedApplication application2) {
        return applicationId(application1).equals(applicationId(application2));
    }

    private String applicationId(DeployedApplication application) {
        return application.getVersion().getApplication().getId();
    }

    private static void unsetTokenOnCi(ConfigurationItem ci) {
        if (ci instanceof BaseConfigurationItem) {
            BaseConfigurationItem bci = (BaseConfigurationItem) ci;
            bci.set$token(null);
            unsetTokensOnEmbeddeds(bci);
        }
    }

    private static void unsetTokensOnEmbeddeds(BaseConfigurationItem deployed) {
        Collection<PropertyDescriptor> propertyDescriptors = deployed.getType().getDescriptor().getPropertyDescriptors();
        Collection<PropertyDescriptor> embeddedProperties = Collections2.filter(propertyDescriptors, input -> input.isAsContainment()
                && EnumSet.of(PropertyKind.LIST_OF_CI, PropertyKind.SET_OF_CI).contains(input.getKind())
                && input.getReferencedType().instanceOf(Type.valueOf(EmbeddedDeployed.class)));
        for (PropertyDescriptor embeddedProperty : embeddedProperties) {
            Collection<ConfigurationItem> c = (Collection<ConfigurationItem>) embeddedProperty.get(deployed);
            for (ConfigurationItem configurationItem : c) {
                unsetTokenOnCi(configurationItem);
            }
        }

    }

    private static Set<String> getSatelliteIds(DeployedApplication deployedApp) {
        Set<String> satelliteIds = new HashSet<>();
        for (Deployed deployed : deployedApp.getDeployeds()) {
            if (deployed.getContainer() instanceof Host) {
                Satellite satellite = ((Host) deployed.getContainer()).getSatellite();
                if (satellite != null) {
                    satelliteIds.add(satellite.getId());
                }
            }
        }
        return satelliteIds;
    }

    private static String extractSecureId(Integer ciId) {
        return ciId == null ? null : String.valueOf(ciId);
    }

    private static void addMetadata(TaskSpecification spec, DeployedApplication deployedApp, TaskType taskType) {
        putMetadata(spec, SATELLITE_IDS, StringUtils.join(getSatelliteIds(deployedApp), ','));
        putMetadata(spec, ENVIRONMENT_ID, deployedApp.getEnvironment().getId());
        putMetadata(spec, ENVIRONMENT_INTERNAL_ID, String.valueOf(deployedApp.getEnvironment().get$internalId()));
        putMetadata(spec, ENVIRONMENT_REFERENCE_ID, String.valueOf(deployedApp.getEnvironment().get$referenceId()));
        putMetadata(spec, ENVIRONMENT_SECURED_CI, extractSecureId(deployedApp.getEnvironment().get$securedCi()));
        putMetadata(spec, ENVIRONMENT_DIRECTORY_REFERENCE, deployedApp.getEnvironment().get$directoryReference());
        putMetadata(spec, ENVIRONMENT, deployedApp.getEnvironment().getName());
        putMetadata(spec, VERSION, deployedApp.getVersion().getName());
        putMetadata(spec, APPLICATION, deployedApp.getVersion().getApplication().getName());
        putMetadata(spec, APPLICATION_INTERNAL_ID, String.valueOf(deployedApp.getVersion().getApplication().get$internalId()));
        putMetadata(spec, APPLICATION_REFERENCE_ID, String.valueOf(deployedApp.getVersion().getApplication().get$referenceId()));
        putMetadata(spec, APPLICATION_SECURED_CI, extractSecureId(deployedApp.getVersion().getApplication().get$securedCi()));
        putMetadata(spec, APPLICATION_DIRECTORY_REFERENCE, deployedApp.getVersion().getApplication().get$directoryReference());
        putMetadata(spec, TASK_TYPE, taskType.name());
    }

    private static final Logger logger = LoggerFactory.getLogger(DeploymentService.class);
}
