package com.xebialabs.deployit.listener.version

import ai.digital.deploy.core.common.ordering.VersionOrdering
import com.xebialabs.deployit.engine.spi.event._
import com.xebialabs.deployit.plugin.api.reflect.Type
import com.xebialabs.deployit.plugin.api.udm.{Application, Version}
import com.xebialabs.deployit.repository.RepositoryServiceHolder.getRepositoryService
import com.xebialabs.deployit.repository.SearchParameters
import com.xebialabs.xlplatform.coc.dto.SCMTraceabilityData
import grizzled.slf4j.Logging
import nl.javadude.t2bus.Subscribe

import scala.jdk.CollectionConverters._

@DeployitEventListener class LastVersionListener extends Logging {

  import com.xebialabs.deployit.core.util.IdExtensions._

  @Subscribe def receiveCisCreated(event: CisCreatedEvent): Unit = {
    event.getCis.asScala.foreach {
      case version: Version =>
        val application = readApplication(version)
        if (VersionOrdering.gt(version.getVersion, application.getLastVersion)) {
          updateVersion(version.getVersion, application, Option(event.getSCMTraceabilityData))
        }
      case _ =>
    }
  }

  @Subscribe def receiveCiCopied(event: CiCopiedEvent): Unit = {
    event.getCi match {
      case version: Version =>
        val application = readApplication(version)
        val newVersion = event.getNewId.getName
        if (VersionOrdering.gt(newVersion, application.getLastVersion)) {
          updateVersion(newVersion, application)
        }
      case _ =>
    }
  }

  @Subscribe def receiveCisDeleted(event: CisDeletedEvent): Unit = {
    event.getCis.asScala.foreach {
      case version: Version =>
        val application = readApplication(version)
        if (VersionOrdering.equiv(application.getLastVersion, version.getId.getName)) {
          updateVersion(calculateLastVersion(application), application)
        }
      case _ =>
    }
  }

  @Subscribe def receiveCiRenamed(event: CiRenamedEvent): Unit = {
    event.getCi match {
      case version: Version =>
        val application = readApplication(version)
        val newVersion = event.getNewName
        if (VersionOrdering.gt(newVersion, application.getLastVersion)) {
          updateVersion(newVersion, application)
        } else if (VersionOrdering.equiv(application.getLastVersion, event.getId.getName)) {
          updateVersion(calculateLastVersion(application, Map(event.getId.getName -> newVersion)), application)
        }
      case _ =>
    }
  }

  protected def updateVersion(lastVersion: String, application: Application, scmTraceabilityData: Option[SCMTraceabilityData] = None): Unit = {
    logger.debug(s"Update last version [${application.getId}] - $lastVersion")
    application.setLastVersion(lastVersion)
    getRepositoryService.updateCis((application :: Nil).asJava, scmTraceabilityData.orNull)
  }

  protected def calculateLastVersion(application: Application, versionOverrides: Map[String, String] = Map()): String = {
    getRepositoryService
      .list(new SearchParameters().setParent(application.getId).setType(Type.valueOf(classOf[Version])))
      .asScala
      .map(ci => versionOverrides.getOrElse(ci.getId.getName, ci.getId.getName))
      .sorted(VersionOrdering.reverse)
      .headOption
      .getOrElse(VersionOrdering.EMPTY_LAST_VERSION)
  }

  protected def readApplication(version: Version): Application = {
    getRepositoryService.read[Application](version.getId.getParent, 1, null, false)
  }
}
