package com.xebialabs.deployit.service

import com.github.zafarkhaja.semver.{Version => OsgiVersion}
import com.xebialabs.deployit.plugin.api.semver.VersionRange
import com.xebialabs.deployit.plugin.api.udm.DeploymentPackage
import scalax.collection.GraphEdge.DiEdge
import scalax.collection.GraphPredef.Param
import scalax.collection.edge.LDiEdge
import scalax.collection.{mutable => xmutable}

import scala.language.{higherKinds, implicitConversions}
import scala.util.Try

package object dependency {
  private[dependency] type ApplicationGraph = xmutable.Graph[Application, LDiEdge]
  private[dependency] type DeployedApplicationGraph = xmutable.Graph[DeployedApplicationInfo, DiEdge]

  private[dependency] def graph(elems: Param[Application, LDiEdge]*): ApplicationGraph = {
    xmutable.Graph[Application, LDiEdge](elems: _*)
  }

  private[dependency] def deployedGraph(elems: Param[DeployedApplicationInfo, DiEdge]*): DeployedApplicationGraph = {
    xmutable.Graph[DeployedApplicationInfo, DiEdge](elems: _*)
  }

  object Implicits {

    implicit class RangeUtils(val osgiVersion: OsgiVersion) extends AnyVal {
      def in(versionRange: VersionRange) = if (versionRange.includes(osgiVersion)) Option(osgiVersion) else None
    }

    implicit class VersionUtils(val v1: Option[OsgiVersion]) extends AnyVal {
      def max(v2: Option[OsgiVersion]) = Try(Seq(v1, v2).flatten.max).toOption
    }

    implicit class UdmVersionUtils(val v1: DeploymentPackage) extends AnyVal {

      def toOsgi = parseVersion(v1.getName)
    }

    implicit class ApplicationGraphUtils(val graph: ApplicationGraph) extends AnyVal {
      def allRequiredVersionsInTopologicalOrder: List[List[DeploymentPackage]] = graph.topologicalSort match  {
        case Right(nodes) =>
          nodes.toLayered.toOuter.tail.map(_._2.toList.sorted.map(version)).toList
        case Left(cycle) =>
          throw new RuntimeException(s"Cycle dependency identified on $cycle")
      }

      def version(app: Application): DeploymentPackage = graph.get(app).incoming
        .headOption
        .flatMap(incoming => incoming.label.asInstanceOf[Label].udmVersion)
        .getOrElse(throw new NoSuchElementException("No deployment package"))
    }

    implicit class DeployedApplicationGraphUtils(val deployedApplicationGraph: DeployedApplicationGraph) extends AnyVal {
      def dependencyApplicationGroups: List[List[DeployedApplicationInfo]] = deployedApplicationGraph.topologicalSort match {
        case Right(nodes) =>
          nodes.toLayered.toOuter.tail.map(_._2.toList.sorted).toList
        case Left(cycle) =>
          throw new RuntimeException(s"Cycle dependency identified on $cycle")
      }
    }

    implicit val applicationOrdering: Ordering[Application] = Ordering.by(_.id)

    implicit val deployedInfoOrdering: Ordering[DeployedApplicationInfo] = Ordering.by(_.applicationId)

    implicit val osgiOrdering: Ordering[OsgiVersion] = Ordering.fromLessThan((v1, v2) => v1.compareTo(v2) < 0)

    implicit class IdUtils(val id: String) extends AnyVal {
      def name = {
        if (id.nonEmpty)
          id.split("/").lastOption.getOrElse("")
        else
          id
      }
    }
  }

  def parseVersion(version: String) = {
    val parts = version.split("\\.")
    if (parts.length == 1) {
      OsgiVersion.forIntegers(parts(0).toInt)
    } else if (parts.length == 2) {
      OsgiVersion.forIntegers(parts(0).toInt, parts(1).toInt)
    } else {
      OsgiVersion.valueOf(version)
    }
  }

}
