package com.xebialabs.deployit.service.deployment;

import ai.digital.deploy.core.common.TypeCalculator;
import com.google.common.collect.Collections2;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import com.xebialabs.deployit.checks.Checks;
import com.xebialabs.deployit.plugin.api.reflect.PropertyDescriptor;
import com.xebialabs.deployit.plugin.api.reflect.PropertyKind;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.plugin.api.udm.*;
import com.xebialabs.deployit.server.api.util.IdGenerator;
import com.xebialabs.deployit.service.deployment.setter.AbstractPropertySetter;
import com.xebialabs.deployit.service.replacement.ConsolidatedDictionary;
import com.xebialabs.deployit.service.replacement.DictionaryValueException;
import com.xebialabs.deployit.service.replacement.MustachePlaceholderScanner;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.Collection;
import java.util.List;
import java.util.stream.Stream;

import static com.xebialabs.deployit.checks.Checks.checkArgument;
import static java.util.stream.Collectors.toSet;

public class DeployedPropertySetter {

    private final TypeCalculator calculator = new TypeCalculator();

    private final AbstractPropertySetter propertySetter;

    public DeployedPropertySetter(AbstractPropertySetter propertySetter) {
        this.propertySetter = propertySetter;
    }

    void setProperties(EmbeddedDeployedContainer<?, ?> deployed, EmbeddedDeployedContainer<?, ?> previousDeployed, ConsolidatedDictionary dictionary) {
        checkArgument(previousDeployed == null || deployed.getType().equals(previousDeployed.getType()), "New and existing target types should be equal when upgrading (%s)", deployed.getType());
        for (PropertyDescriptor deployedPropDesc : deployed.getType().getDescriptor().getPropertyDescriptors()) {

            if (deployedPropDesc.isDeployedSpecific()) {
                continue;
            }

            if (isEmbeddedProperty(deployedPropDesc)) {
                generateEmbeddeds(deployed, deployedPropDesc, dictionary);
                continue;
            }

            final ConfigurationItem currentDeployable = deployed.getDeployable();
            ConfigurationItem previousDeployable = previousDeployed == null ? null : previousDeployed.getDeployable();
            setProperty(deployed, previousDeployed, currentDeployable, previousDeployable, dictionary, deployedPropDesc);
        }
    }

    void setProperties(DeployedApplication currentDeployedApp, DeployedApplication existingDeployedApp, ConsolidatedDictionary dictionary) {
        final ConfigurationItem currentPackage = currentDeployedApp.getVersion();

        Stream<PropertyDescriptor> deployedAppPd = currentDeployedApp.getType().getDescriptor().getPropertyDescriptors().stream()
                .filter(pd -> !pd.isDeployedSpecific());

        Stream<PropertyDescriptor> packagePd = currentPackage.getType().getDescriptor().getPropertyDescriptors().stream()
                .filter(pd -> currentDeployedApp.hasProperty(pd.getName()))
                .map(pd -> currentDeployedApp.getType().getDescriptor().getPropertyDescriptor(pd.getName()));

        final ConfigurationItem previousPackage = existingDeployedApp == null ? null : existingDeployedApp.getVersion();
        Stream.concat(deployedAppPd, packagePd).
                collect(toSet()).
                forEach(pd -> setProperty(currentDeployedApp, existingDeployedApp, currentPackage, previousPackage, dictionary, pd));
    }


    private void setProperty(ConfigurationItem currentTarget, ConfigurationItem previousTarget, ConfigurationItem currentSource,
                             ConfigurationItem previousSource, ConsolidatedDictionary dictionary, PropertyDescriptor targetPropertyDescriptor) {

        logger.trace("Going to set property {} of currentTarget {}", targetPropertyDescriptor.getFqn(), currentTarget);
        PropertyDescriptor sourcePropertyDescriptor = null;
        if (currentTarget instanceof Container &&
                targetPropertyDescriptor.getName().equals("tags") &&
                currentSource instanceof DeployableContainer) {
            sourcePropertyDescriptor = currentSource.getType().getDescriptor().getPropertyDescriptor("containerTags");
        } else if (currentSource.hasProperty(targetPropertyDescriptor.getName())) {
            sourcePropertyDescriptor = currentSource.getType().getDescriptor().getPropertyDescriptor(targetPropertyDescriptor.getName());
        }

        propertySetter.setProperty(currentTarget, previousTarget, currentSource, previousSource, dictionary, targetPropertyDescriptor, sourcePropertyDescriptor);
    }

    private void generateEmbeddeds(EmbeddedDeployedContainer<?, ?> deployed, final PropertyDescriptor deployedPropDesc, ConsolidatedDictionary dictionary) {
        final ConfigurationItem deployable = deployed.getDeployable();
        String name = deployedPropDesc.getName();
        Collection<EmbeddedDeployed<?, ?>> embeddedDeployeds = deployedPropDesc.getKind() == PropertyKind.LIST_OF_CI ? Lists.<EmbeddedDeployed<?, ?>>newArrayList() : Sets.<EmbeddedDeployed<?, ?>>newHashSet();
        PropertyDescriptor deployableProperty = deployable.getType().getDescriptor().getPropertyDescriptor(name);
        if (deployableProperty == null) {
            logger.info("Deployed [{}] has embedded deployeds in property [{}] but source [{}] has no such property.", deployed.getId(), deployedPropDesc.getFqn(), deployable.getId());
            return;
        }
        @SuppressWarnings("unchecked")
        Collection<EmbeddedDeployable> embeddedDeployables = (Collection<EmbeddedDeployable>) deployableProperty.get(deployable);

        for (EmbeddedDeployable embeddedDeployable : embeddedDeployables) {
            final List<Type> embeddedDeployedTypes = calculator.findAllMostSpecificDeployedTypesForDeployableAndContainerTypes(embeddedDeployable.getType(), deployed.getType());
            Collection<Type> filtered = Collections2.filter(embeddedDeployedTypes, input -> input.instanceOf(deployedPropDesc.getReferencedType()));
            if (filtered.isEmpty()) {
                logger.info("Not found a matching EmbeddedDeployed type for [{}] and [{}]", embeddedDeployable.getType(), deployed.getType());
            } else if (filtered.size() > 1) {
                logger.error("Found more than 1 ({}) compatible EmbeddedDeployed type for [{}] and [{}]", filtered, embeddedDeployable.getType(), deployed.getType());
                throw new Checks.IncorrectArgumentException("Found more than 1 (%s) compatible EmbeddedDeployed type for [%s] and [%s]", filtered, embeddedDeployable.getType(), deployed.getType());
            } else {
                Type embeddedDeployedType = filtered.iterator().next();
                EmbeddedDeployed embeddedDeployed = embeddedDeployedType.getDescriptor().newInstance(IdGenerator.generateId(deployed, embeddedDeployable.getId()));
                //noinspection unchecked
                embeddedDeployed.setDeployable(embeddedDeployable);
                //noinspection unchecked
                embeddedDeployed.setContainer(deployed);

                if (MustachePlaceholderScanner.hasPlaceholders(embeddedDeployed.getName())) {
                    try {
                        dictionary.resolveDeployedName(embeddedDeployed);
                    } catch (DictionaryValueException e) {
                        throw new DeployedApplicationFactory.IncorrectDeployedException(e, "Couldn't generate name for target from [%s]", embeddedDeployable.getId());
                    }
                }
                embeddedDeployeds.add(embeddedDeployed);
                setProperties(embeddedDeployed, null, dictionary);
            }
        }

        deployedPropDesc.set(deployed, embeddedDeployeds);
    }

    private boolean isEmbeddedProperty(PropertyDescriptor propertyDescriptor) {
        return propertyDescriptor.isAsContainment() &&
                (propertyDescriptor.getKind() == PropertyKind.SET_OF_CI || propertyDescriptor.getKind() == PropertyKind.LIST_OF_CI) &&
                propertyDescriptor.getReferencedType().isSubTypeOf(Type.valueOf(EmbeddedDeployed.class));
    }

    private static final Logger logger = LoggerFactory.getLogger(DeployedPropertySetter.class);
}
