package com.xebialabs.deployit.service.deployment;

import com.google.common.base.Function;
import com.google.common.collect.Lists;
import com.xebialabs.deployit.checks.Checks;
import com.xebialabs.deployit.engine.api.dto.Deployment;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.plugin.api.udm.*;
import com.xebialabs.deployit.plugin.api.validation.ValidationMessage;
import com.xebialabs.deployit.service.replacement.ConsolidatedDictionary;
import com.xebialabs.deployit.service.replacement.Dictionaries;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.util.*;
import java.util.stream.Collectors;

import static com.google.common.collect.Lists.newArrayList;
import static com.xebialabs.deployit.checks.Checks.checkArgument;
import static com.xebialabs.deployit.service.replacement.Dictionaries.of;
import static java.util.stream.Collectors.toList;

@Component
public class DeployedService {

    private final DeployedApplicationFactory deployedApplicationFactory;
    private final DeployedProcessorsFactory deployedProcessorsFactory;

    @Autowired
    public DeployedService(DeployedApplicationFactory deployedApplicationFactory, DeployedProcessorsFactory deployedProcessorsFactory) {
        this.deployedApplicationFactory = deployedApplicationFactory;
        this.deployedProcessorsFactory = deployedProcessorsFactory;
    }

    public DeployedApplication generateDeployedApplication(Type deployedApplicationType, Version version, Environment env, ConsolidatedDictionary consolidatedDictionary) {
        return deployedApplicationFactory.createDeployedApplication(deployedApplicationType, version, env, consolidatedDictionary);
    }

    public GeneratedDeployeds generateSelectedDeployeds(Deployment deployment, List<ConfigurationItem> deployableCis, Version version, Environment env) {
        return generateSelectedDeployeds(deployment, deployableCis, env.getMembers(), of(env).filterBy(version));
    }

    public GeneratedDeployeds generateSelectedDeployeds(Deployment deployment, List<ConfigurationItem> deployableCis, Set<? extends Container> containers, Dictionaries dictionaries) {
        List<Deployable> deployables = Lists.transform(deployableCis, from -> {
            checkArgument(from instanceof Deployable, "The entity %s is not a deployable", from.getId());
            return (Deployable) from;
        });

        return generateDeployedsOfType(deployment, deployables, containers, null, dictionaries);
    }

    public GeneratedDeployeds createSelectedDeployed(Deployment deployment, Deployable deployable, Container container, Type deployedType, Version version, Environment env) {
        logger.debug("Creating deployed for [{}] and [{}]", deployable, container);

        Dictionaries dictionaries = of(env).filterBy(version);
        return generateDeployedsOfType(deployment, newArrayList(deployable), newArrayList(container), deployedType, dictionaries, true);
    }

    public GeneratedDeployeds generateUpgradedDeployeds(DeployedApplication deployedApplication, Version newPackage) {
        final Environment environment = deployedApplication.getEnvironment();
        GeneratedDeployeds generatedDeployeds = new GeneratedDeployeds();
        Dictionaries dictionaries = of(environment).filterBy(newPackage);
        DeploymentContext deploymentContext = deployedProcessorsFactory.createContextForUpgrade(deployedApplication, dictionaries, deployedApplication.getDeployeds());
        DeployedGenerator deployedGenerator = deployedProcessorsFactory.createUpgradeDeployedGenerator();
        Set<Container> containers = deployedApplication.getDeployeds().stream().map(Deployed::getContainer).collect(Collectors.toSet());
        newPackage.getDeployables().stream().forEach(d -> {
            for (Container container : containers) {
                try {
                    generatedDeployeds.merge(deployedGenerator.generateDeployed(deploymentContext, d, container));

                } catch (Checks.IncorrectArgumentException exception) {
                    deployedApplication.get$validationMessages().add(ValidationMessage.error(d.getId(), null, exception.getMessage()));
                }
            }
        });

        return generatedDeployeds;
    }

    public DeployedApplication generateUpdateDeployedApplication(DeployedApplication deployedApplication, Version newVersion, ConsolidatedDictionary dictionary) {
        return deployedApplicationFactory.createUpgradeDeployedApplication(newVersion, deployedApplication, dictionary);
    }

    public List<Deployed> createDeployedsForNonExistingCombinations(Deployment deployment, Version version, Environment environment, final Set<Deployed> deployeds, Map<String, String> userProvidedPlaceholder) {
        Set<? extends Deployable> deployables = version.getDeployables();
        Set<? extends Container> members = environment.getMembers();

        final List<Deployed> validExistingDeployeds = deployeds.stream().filter(input -> input.getDeployable() != null).collect(toList());
        logger.debug("Valid existing deployeds: {}", validExistingDeployeds);

        final List<Deployed> invalidExistingDeployeds = deployeds.stream().filter(input -> input.getDeployable() == null).collect(toList());
        logger.debug("Invalid existing deployeds: {}", invalidExistingDeployeds);

        Dictionaries dictionaries = of(environment).filterBy(version).withAdditionalEntries(userProvidedPlaceholder);
        GeneratedDeployeds generatedDeployeds = generateDeployedsOfType(deployment, deployables, members, null, dictionaries);
        logger.debug("Going to filter previously present deployeds");

        FilterSet filterSet = new FilterSet(validExistingDeployeds);
        generatedDeployeds.getDeployeds().forEach(filterSet::add);
        List<Deployed> allValidAsList = filterSet.getAllValidAsList();
        allValidAsList.addAll(invalidExistingDeployeds.stream().filter(d -> !allValidAsList.contains(d)).collect(Collectors.toList()));
        return allValidAsList;
    }

    private GeneratedDeployeds generateDeployedsOfType(Deployment deployment, Collection<? extends Deployable> deployables, Collection<? extends Container> containers, Type deployedType, Dictionaries dictionaries) {
        return generateDeployedsOfType(deployment, deployables, containers, deployedType, dictionaries, false);
    }

    private GeneratedDeployeds generateDeployedsOfType
            (Deployment deployment, Collection<? extends Deployable> deployables, Collection<? extends Container> containers, Type deployedType, Dictionaries dictionaries, boolean ignoreTags) {
        GeneratedDeployeds generatedDeployeds = new GeneratedDeployeds();
        DeployedGenerator deployedGenerator = ignoreTags
                ? deployedProcessorsFactory.createCoreDeployedGenerator()
                : deployedProcessorsFactory.createTagDeployedGenerator();
        DeploymentContext deploymentContext = deployedType != null
                ? deployedProcessorsFactory.createContextForSpecificType(deployment, dictionaries, deployedType)
                : deployedProcessorsFactory.createContextWithCalculatedType(deployment, dictionaries);
        for (Deployable deployable : deployables) {
            for (Container container : containers) {
                generatedDeployeds.merge(deployedGenerator.generateDeployed(deploymentContext, deployable, container));
            }
        }
        return generatedDeployeds;
    }

    private static final Logger logger = LoggerFactory.getLogger(DeployedService.class);

    public void scanMissingPlaceholders(DeployedApplication deployedApplication, Environment environment) {
        deployedApplicationFactory.scanMissingPlaceholders(deployedApplication, environment);
    }

    private static class FilterSet {

        private Function<Deployed, Key> KEY_FUNC = input -> new Key(input.getDeployable().getId(), input.getContainer().getId());

        private Function<Value, String> VALUE_KEY_FUNC = input -> input.deployed.getId();

        private Map<Key, Map<String, Value>> map = new HashMap<>();

        /**
         * Create a set with initial data which is all invalid.
         *
         * @param initialDeployeds
         */
        public FilterSet(Collection<Deployed> initialDeployeds) {
            initialDeployeds.stream().forEach(d -> {
                if (isProvisionable(d.getDeployable())) {
                    addProvisioned(d);
                } else {
                    addDeployed(d);
                }
            });
        }

        private void addDeployed(Deployed d) {
            putInitial(KEY_FUNC.apply(d), new Value(d));
        }

        private void addProvisioned(Deployed d) {
            Key key = KEY_FUNC.apply(d);
            if (keyExists(key)) {
                putInExisting(key, new Value(d));
            } else {
                putInitial(key, new Value(d));
            }
        }

        private void putInExisting(Key key, Value v) {
            map.get(key).put(VALUE_KEY_FUNC.apply(v), v);
        }

        private boolean keyExists(Key key) {
            return map.containsKey(key);
        }

        private Map<String, Value> putInitial(Key key, Value v) {
            final Map m = new HashMap<String, Value>();
            m.put(VALUE_KEY_FUNC.apply(v), v);
            return map.put(key, m);
        }

        private boolean isProvisionable(Deployable deployable) {
            return deployable instanceof BaseProvisionable;
        }

        /**
         * Add a new element to the set, if the element previously existed, mark the previous occurrence valid, else add it as a valid element.
         *
         * @param ci
         */
        public void add(ConfigurationItem ci) {
            Deployed<?, ?> deployed = (Deployed<?, ?>) ci;
            Key k = KEY_FUNC.apply(deployed);
            Value v = new Value(deployed).valid();
            if (isProvisionable(deployed.getDeployable())) {
                addProvisioned(deployed, k, v);
            } else {
                addDeployed(deployed, k, v);
            }
        }

        private void addProvisioned(Deployed<?, ?> deployed, Key k, Value v) {
            if (keyExists(k)) {
                final Map<String, Value> previous = map.get(k);
                final String valueKey = VALUE_KEY_FUNC.apply(v);
                final Value previousValue = previous.put(valueKey, v);
                if (previousValue != null) {
                    logger.debug("Filtering out [{}] because it is already present.", deployed);
                    previous.put(valueKey, previousValue.valid());
                }
            } else {
                putInitial(k, v);
            }
        }

        private void addDeployed(Deployed<?, ?> deployed, Key k, Value v) {
            Map<String, Value> previous = putInitial(k, v);
            // Reset the previous value and validate it.
            if (previous != null) {
                logger.debug("Filtering out [{}] because it is already present.", deployed);
                previous = previous.entrySet().stream().collect(Collectors.toMap(e -> e.getKey(), e -> e.getValue().valid()));
                map.put(k, previous);
            }
        }

        public List<Deployed> getAllValidAsList() {
            return map.values().stream().map(m -> m.values()).flatMap(v -> v.stream()).filter(Value::isValid).map(value -> value.deployed).collect(toList());
        }
    }

    private static class Value {
        private Deployed deployed;
        private boolean valid;

        private Value(Deployed deployed) {
            this.deployed = deployed;
        }

        Value valid() {
            this.valid = true;
            return this;
        }

        private boolean isValid() {
            return valid;
        }

    }

    private static class Key {
        private String deployableId;
        private String containerId;

        private Key(String deployableId, String containerId) {
            this.deployableId = deployableId;
            this.containerId = containerId;
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) return true;
            if (o == null || getClass() != o.getClass()) return false;

            final Key key = (Key) o;

            return containerId.equals(key.containerId) && deployableId.equals(key.deployableId);

        }

        @Override
        public int hashCode() {
            int result = deployableId.hashCode();
            result = 31 * result + containerId.hashCode();
            return result;
        }
    }

}
