package com.xebialabs.deployit.core.api.dto;

import com.google.common.collect.Multimap;

import javax.xml.bind.annotation.*;
import java.util.List;

import com.google.common.base.*;
import com.google.common.collect.*;

import static com.google.common.collect.Lists.newArrayList;
import static com.google.common.collect.Collections2.filter;

import static java.lang.String.format;

@XmlAccessorType(XmlAccessType.FIELD)
@XmlRootElement(name="security-settings")
public class PrincipalPermissions extends AbstractDTO {

	@XmlAttribute
	private String username;

	@XmlElement
	private List<PrincipalPermission> permissions = newArrayList();

	public PrincipalPermissions() {
	}

	public PrincipalPermissions(final String userName, final Multimap<String, String> principalPermissions) {
		this.username = userName;
		if (principalPermissions != null) {
			convert(principalPermissions);
		}
	}

	private void convert(Multimap<String, String> principalPermissions) {
		for (String node : principalPermissions.keySet()) {
			permissions.add(new PrincipalPermission(node, newArrayList(principalPermissions.get(node))));
		}
	}

	public String getUsername() {
		return username;
	}

	public void setUsername(final String userName) {
		this.username = userName;
	}

	public boolean hasPermissions() {
		return !this.permissions.isEmpty();
	}

	public boolean hasPermission(final String permission) {
		return !filter(permissions, new Predicate<PrincipalPermission>() {
			public boolean apply(PrincipalPermission input) {
				return Strings.isNullOrEmpty(input.node) && input.permissions.contains(permission);
			}
		}).isEmpty();
	}

	public boolean hasPermission(final String node, final String permission) {
		return !filter(permissions, new Predicate<PrincipalPermission>() {
			public boolean apply(PrincipalPermission input) {
				return ((node == input.node) || (node != null && node.equals(input.node))) && input.permissions.contains(permission);
			}
		}).isEmpty();
	}

	@Override
	public String toString() {
		StringBuilder permissionString = new StringBuilder();
		if ((permissions == null) || permissions.isEmpty()) {
			permissionString.append("There are no permissions set for the requested user, or the requested user doesn't exist!\n");
		} else {
			for (PrincipalPermission permissionEntry : permissions) {
				if (permissionEntry.node.equals("")) {
					permissionString.append(format("\nGranted global permissions:'\n"));
					appendPermissions(permissionString, permissionEntry.permissions);
				} else {
					permissionString.append(format("\nGranted permissions on node '%s':\n", permissionEntry.node));
					appendPermissions(permissionString, permissionEntry.permissions);
				}
			}
		}
		return permissionString.toString();

	}

	private void appendPermissions(StringBuilder permissionString, List<String> permissions) {
		for (String permission : permissions) {
			permissionString.append(format("\t- %s\n", permission));
		}
	}

	@XmlRootElement
	private static class PrincipalPermission {
		@XmlElement
		private String node;
		
		@XmlElement
		private List<String> permissions = newArrayList();

		private PrincipalPermission(String node, List<String> permissions) {
			this.node = node;
			this.permissions = permissions;
		}

		public PrincipalPermission() {
		}
	}
}
