package com.xebialabs.deployit.core.api.resteasy.http.tunnel;

import java.io.StringWriter;
import java.net.URI;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.List;
import java.util.Locale;

import javax.ws.rs.core.CacheControl;
import javax.ws.rs.core.EntityTag;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.NewCookie;
import javax.ws.rs.core.Response;
import javax.ws.rs.core.Variant;
import javax.xml.bind.JAXBContext;
import javax.xml.bind.JAXBException;
import javax.xml.bind.Marshaller;
import javax.xml.bind.annotation.XmlRootElement;

import org.jboss.resteasy.core.Headers;
import org.jboss.resteasy.core.ServerResponse;
import org.jboss.resteasy.spi.HttpRequest;
import org.jboss.resteasy.spi.ResteasyProviderFactory;
import org.jboss.resteasy.util.DateUtil;
import org.jboss.resteasy.util.HttpHeaderNames;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 */
public class TunneledResponseBuilder extends Response.ResponseBuilder {

	private TunneledResponse tunneledResponse = new TunneledResponse();

	private Headers<Object> metadata = new Headers<Object>();

	private static final SimpleDateFormat dateFormatRFC822 = new SimpleDateFormat("EEE, dd MMM yyyy HH:mm:ss z");

	@Override
	public Response build() {
		return new ServerResponse(tunneledResponse, 200, metadata);
	}

	@Override
	public Response.ResponseBuilder clone() {
		final TunneledResponseBuilder trb = new TunneledResponseBuilder();
		trb.tunneledResponse = tunneledResponse;
		trb.metadata.putAll(metadata);
		return trb;
	}

	@Override
	public Response.ResponseBuilder status(final int status) {
		if (logger.isDebugEnabled())
			logger.debug("Setting status: {}", status);

		tunneledResponse.setStatus(status);
		return this;
	}

	@Override
	public Response.ResponseBuilder entity(final Object entity) {
		if (logger.isDebugEnabled())
			logger.debug("Setting entity: {}", entity);

        if (entity == null) {
            return this;
        }

		if (entity.getClass().getAnnotation(XmlRootElement.class) != null) {
			final StringWriter writer = new StringWriter();
			try {
				final JAXBContext jaxbContext = JAXBContext.newInstance(entity.getClass());
				final Marshaller marshaller = jaxbContext.createMarshaller();

				marshaller.setProperty(Marshaller.JAXB_FRAGMENT, true);

				marshaller.marshal(entity, writer);
			} catch (JAXBException e) {
				throw new IllegalArgumentException("Could not marshal: " + entity, e);
			}
			tunneledResponse.setEntity(writer.toString());
		} else {
			tunneledResponse.setEntity(entity.toString());
		}
		return this;
	}

	@Override
	public Response.ResponseBuilder type(final MediaType type) {
		return addSingleHeader(HttpHeaderNames.CONTENT_TYPE, type);
	}

	@Override
	public Response.ResponseBuilder type(final String type) {
		return addSingleHeader(HttpHeaderNames.CONTENT_TYPE, type);
	}

	@Override
	public Response.ResponseBuilder variant(final Variant variant) {
		if (variant.getMediaType() != null)
			type(variant.getMediaType());
		if (variant.getLanguage() != null)
			language(variant.getLanguage());
		if (variant.getEncoding() != null)
			metadata.putSingle(HttpHeaderNames.CONTENT_ENCODING, variant.getEncoding());
		return this;
	}

	@Override
	public Response.ResponseBuilder variants(final List<Variant> variants) {
		throw new UnsupportedOperationException("Ask Jeroen");
	}

	@Override
	public Response.ResponseBuilder language(final String language) {
		return addSingleHeader(HttpHeaderNames.CONTENT_LANGUAGE, language);
	}

	@Override
	public Response.ResponseBuilder language(final Locale language) {
		return addSingleHeader(HttpHeaderNames.CONTENT_LANGUAGE, language);
	}

	@Override
	public Response.ResponseBuilder location(final URI location) {
		return addSingleHeader(HttpHeaderNames.LOCATION, absolutize(location));
	}

	@Override
	public Response.ResponseBuilder contentLocation(final URI location) {
		return addSingleHeader(HttpHeaderNames.CONTENT_LOCATION, absolutize(location));
	}

	private URI absolutize(URI location) {
		if (!location.isAbsolute() && ResteasyProviderFactory.getContextData(HttpRequest.class) != null) {
			String path = location.toString();
			if (path.startsWith("/"))
				path = path.substring(1);
			URI baseUri = ResteasyProviderFactory.getContextData(HttpRequest.class).getUri().getBaseUri();
			location = baseUri.resolve(path);
		}

		return location;
	}

	@Override
	public Response.ResponseBuilder tag(final EntityTag tag) {
		return addSingleHeader(HttpHeaderNames.ETAG, tag);
	}

	@Override
	public Response.ResponseBuilder tag(final String tag) {
		return addSingleHeader(HttpHeaderNames.ETAG, tag);
	}

	@Override
	public Response.ResponseBuilder lastModified(final Date lastModified) {
		return addSingleHeader(HttpHeaderNames.LAST_MODIFIED, DateUtil.formatDate(lastModified));
	}

	@Override
	public Response.ResponseBuilder cacheControl(final CacheControl cacheControl) {
		return addSingleHeader(HttpHeaderNames.CACHE_CONTROL, cacheControl);
	}

	@Override
	public Response.ResponseBuilder expires(final Date expires) {
		return addSingleHeader(HttpHeaderNames.EXPIRES, dateFormatRFC822.format(expires));
	}

	@Override
	public Response.ResponseBuilder header(final String name, final Object value) {
		return addHeader(name, value);
	}

	@Override
	public Response.ResponseBuilder cookie(final NewCookie... cookies) {
		if (logger.isDebugEnabled())
			logger.debug("Setting cookies: {}", cookies);

		if (cookies == null) {
			metadata.remove(HttpHeaderNames.SET_COOKIE);
			return this;
		}
		for (NewCookie cookie : cookies) {
			metadata.add(HttpHeaderNames.SET_COOKIE, cookie);
		}
		return this;
	}

	private Response.ResponseBuilder addSingleHeader(final String headerName, final Object headerValue) {
		if (logger.isDebugEnabled())
			logger.debug("Adding header: {}: {}", headerName, headerValue);

		metadata.putSingle(headerName, headerValue);
		tunneledResponse.addSingleHeader(headerName, headerValue);
		return this;
	}

	private Response.ResponseBuilder addHeader(final String headerName, final Object headerValue) {
		if (logger.isDebugEnabled())
			logger.debug("Adding header: {}: {}", headerName, headerValue);

		metadata.add(headerName, headerValue);
		tunneledResponse.addHeader(headerName, headerValue);
		return this;
	}

	private static final Logger logger = LoggerFactory.getLogger(TunneledResponseBuilder.class);

}
