package com.xebialabs.deployit.core.rest.api.reports.widgets;

import java.util.*;

import org.joda.time.DateTime;

import com.xebialabs.deployit.core.api.dto.Report;
import com.xebialabs.deployit.core.api.dto.Report.ReportLine;
import com.xebialabs.deployit.jcr.grouping.Duration;
import com.xebialabs.deployit.jcr.grouping.GroupBy;
import com.xebialabs.deployit.task.ArchivedTaskSearchParameters;
import com.xebialabs.deployit.task.archive.JcrTaskArchive;

import static com.google.common.collect.Lists.newArrayList;
import static com.xebialabs.deployit.task.TaskType.INITIAL;
import static com.xebialabs.deployit.task.TaskType.UNDEPLOY;
import static com.xebialabs.deployit.task.TaskType.UPGRADE;

public class Top5LongestRunningDeployments extends DashboardWidgetBase {

    private static final String DURATIONLIST = "durationList";
    private static final String DURATION = "duration";
    private static final String ENVIRONMENT = "environment";
    private static final String APPLICATION_NAME = "application";

    public Top5LongestRunningDeployments(final JcrTaskArchive taskArchive) {
        super(taskArchive);
    }

    @Override
    public Report getReport(final DateTime beginDate, final DateTime endDate) {
        ArchivedTaskSearchParameters sp = getSearchParameters().thatAreOfType(EnumSet.of(INITIAL, UNDEPLOY, UPGRADE));
        sp.createdBetween(beginDate, endDate).thatCompleted();
        final Collection<Map<String, Object>> searchResult = taskArchive.searchTasksWithoutLoadingSteps(sp,
                new GroupBy(newArrayList(APPLICATION_NAME, ENVIRONMENT), new Duration(DURATIONLIST)));
        return generateDurationReport(searchResult);
    }

    Report generateDurationReport(final Collection<Map<String, Object>> resultList) {
        calculateAverageDurations(resultList);
        final List<Map<String, Object>> list = newArrayList(resultList);
        Collections.sort(list, new Comparator<Map<String, Object>>() {
            @Override
            public int compare(Map<String, Object> valueMap1, Map<String, Object> valueMap2) {
                final Long duration1 = (Long) valueMap1.get(DURATION);
                final Long duration2 = (Long) valueMap2.get(DURATION);
                return -(duration1.compareTo(duration2));
            }
        });

        Report report = new Report();
        for (Map<String, Object> line : list.subList(0, Math.min(5, resultList.size()))) {
            ReportLine reportLine = report.addLine();
            reportLine.addValue("application", line.get(APPLICATION_NAME));
            reportLine.addValue("environment", line.get(ENVIRONMENT));
            reportLine.addValue(DURATION, formatToMinsAndSecs((Long) line.get(DURATION)));
        }
        return report;
    }

    @SuppressWarnings("unchecked")
    private static void calculateAverageDurations(Collection<Map<String, Object>> resultList) {
        for (Map<String, Object> line : resultList) {
            List<Long> durationList = (List<Long>) line.get(DURATIONLIST);
            long totalDuration = 0l;
            for (Long duration : durationList) {
                totalDuration += duration;
            }
            line.put(DURATION, totalDuration / durationList.size());
        }
    }
}
