package com.xebialabs.deployit.core.rest.api.reports;

import java.util.*;

import com.google.common.collect.Lists;

import com.xebialabs.deployit.core.api.dto.Report;
import com.xebialabs.deployit.core.api.dto.Report.ReportLine;
import com.xebialabs.deployit.core.api.resteasy.Date;
import com.xebialabs.deployit.jcr.grouping.Function;
import com.xebialabs.deployit.jcr.grouping.GroupBy;
import com.xebialabs.deployit.task.ArchivedTaskSearchParameters;
import com.xebialabs.deployit.task.TaskType;
import com.xebialabs.deployit.task.archive.ArchivedTask;
import com.xebialabs.deployit.task.archive.JcrTaskArchive;

import static com.google.common.collect.Lists.newArrayList;
import static com.google.common.collect.Maps.newHashMap;

public class DeploymentsForEnvironmentReport extends TaskArchiveReport {

    // in this map we keep all previous tasks per application to decide
    // which app to show in case of a rollback
    private final Map<String,Map<String,String>> lastTask       = new HashMap<String, Map<String, String>>();

    // in this map we keep all before previous tasks per application to get
    // the correct version of the beployment before the upgrade
    private final Map<String,Map<String,String>> beforeLastTask = new HashMap<String, Map<String, String>>();

    public DeploymentsForEnvironmentReport(final JcrTaskArchive taskArchive) {
        super(taskArchive);
    }

    @SuppressWarnings("unchecked")
    public Report report() {

        final Report report = new Report();

        final Map<String, ArchivedTask> previous = new HashMap<String, ArchivedTask>();

        // results must be ordered for this algorithm to work. We do not want to safe all
        // intermediate tasks so the results need to be ordered.
        ArchivedTaskSearchParameters params = getSearchParameters().orderBy("startDate");

        Collection<Map<String, Object>> searchResult = taskArchive.searchTasksWithoutLoadingSteps(params,
                new GroupBy(Lists.<String> newArrayList("application"), new Function<Map<String, String>>() {
                    @Override
                    public Map<String, String> invoke(Object previousValue, ArchivedTask task) {
                        if (previousValue == null) {
                            return getTaskInfo(task);
                        } else {
                            Map<String, String> taskInfo = (Map<String, String>) previousValue;

                            String application = taskInfo.get("application");
                            Map<String, String> prevTask = lastTask.get(application);
                            if(task != null){
                                beforeLastTask.put(application, prevTask);
                            }
                            lastTask.put(application, taskInfo);
                            return getTaskInfo(task);
                        }
                    }

                    private Map<String, String> getTaskInfo(ArchivedTask task) {
                        Map<String, String> value = newHashMap();
                        value.put("taskId", task.getId());
                        value.put("application", task.getMetadata().get("application"));
                        value.put("deploymentType", task.getMetadata().get("taskType"));
                        value.put("user", task.getOwner());
                        value.put("date", ReportUtils.FORMATTER.print(task.getCompletionDate()));
                        value.put("version", task.getMetadata().get("version"));
                        return value;
                    }

                    @Override
                    public String getName() {
                        return "taskInfo";
                    }

                }));
        List<Map<String, Object>> results = newArrayList(searchResult);
        Collections.sort(results, new Comparator<Map<String, Object>>() {
            @Override
            public int compare(Map<String, Object> lhs, Map<String, Object> rhs) {
                Map<String, String> lhsTaskInfo = (Map<String, String>) lhs.get("taskInfo");
                Map<String, String> rhsTaskInfo = (Map<String, String>) rhs.get("taskInfo");
                Date lhsCompletionDate = new Date(lhsTaskInfo.get("date"));
                Date rhsCompletionDate = new Date(rhsTaskInfo.get("date"));
                return lhsCompletionDate.compareTo(rhsCompletionDate);
            }
        });

        // TODO : Sort me by date
        for (Map<String, Object> task : results) {
            Map<String, String> taskInfo = (Map<String, String>) task.get("taskInfo");
            /* if it is an undeploy, we should not list is
             * if it is a rollback, we should only show it if there was a previous version
             * this happens when it is a rollback on an upgrade.
             */
            if (notUndeployTask(taskInfo) && notRollbackOnInitial(taskInfo)) {
                if(rollbankOnUpgrade(taskInfo)){
                    taskInfo = beforeLastTask.get(task.get("application"));
                }
                ReportLine line = report.addLine();
                line.addValue("application", task.get("application"));
                line.addValue("taskId", taskInfo.get("taskId"));
                line.addValue("version", taskInfo.get("version"));
                line.addValue("user", taskInfo.get("user"));
                line.addValue("date", taskInfo.get("date"));
            }
        }
        return report;
    }

    private boolean rollbankOnUpgrade(Map<String, String> taskInfo) {

        Map<String, String> application = lastTask.get(taskInfo.get("application"));
        return TaskType.valueOf(taskInfo.get("deploymentType")).equals(TaskType.ROLLBACK) &&
                application != null &&
                TaskType.valueOf(application.get("deploymentType")).equals(TaskType.UPGRADE);

    }

    private boolean notRollbackOnInitial(Map<String, String> taskInfo) {

        Map<String, String> application = lastTask.get(taskInfo.get("application"));
        return !(TaskType.valueOf(taskInfo.get("deploymentType")).equals(TaskType.ROLLBACK) &&
                application != null &&
                TaskType.valueOf(application.get("deploymentType")).equals(TaskType.INITIAL));

    }

    private boolean notUndeployTask(Map<String, String> taskInfo) {
        return !TaskType.valueOf(taskInfo.get("deploymentType")).equals(TaskType.UNDEPLOY);
    }
}
