package com.xebialabs.deployit.core.rest.api;

import java.io.IOException;
import java.util.List;
import java.util.Properties;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.core.io.ClassPathResource;
import org.springframework.stereotype.Controller;

import com.xebialabs.deployit.booter.local.PluginVersions;
import com.xebialabs.deployit.core.rest.secured.AbstractSecuredResource;
import com.xebialabs.deployit.engine.api.ServerService;
import com.xebialabs.deployit.engine.api.dto.ServerInfo;
import com.xebialabs.deployit.engine.api.dto.ServerPluginInfo;
import com.xebialabs.deployit.engine.spi.event.RepositoryGarbageCollectedEvent;
import com.xebialabs.deployit.engine.spi.event.SystemStoppedEvent;
import com.xebialabs.deployit.engine.spi.exception.DeployitException;
import com.xebialabs.deployit.event.EventBusHolder;
import com.xebialabs.deployit.event.ShutdownEvent;
import com.xebialabs.deployit.security.permission.Permission;
import com.xebialabs.deployit.service.gc.GarbageCollectionService;

import static com.google.common.collect.Lists.newArrayList;

@Controller
public class ServerResource extends AbstractSecuredResource implements ServerService {

    @Autowired
    private GarbageCollectionService garbageCollectionService;

    @Override
    public void shutdown() {
        checkPermission(Permission.ADMIN);

        EventBusHolder.publish(new SystemStoppedEvent());

        EventBusHolder.publish(new ShutdownEvent());
    }

    @Override
    public ServerInfo getInfo() {
        ClassPathResource resource = new ClassPathResource("com/xebialabs/deployit/release.properties");
        Properties release = new Properties();
        try {
            release.load(resource.getInputStream());
            ServerInfo si = new ServerInfo();
            si.setVersion(release.getProperty("version"));
            for (String plugin : PluginVersions.getRegisteredPlugins()) {
                si.getPluginsInfo().add(new ServerPluginInfo(plugin, PluginVersions.getVersionFor(plugin)));
            }
            si.setClasspathEntries(getServerClassPath());
            return si;
        } catch (IOException e) {
            throw new DeployitException(e);
        }
    }

    @Override
    public void logout() {
        // Is taken care of by BasicAuthWithRememberMeFilter
    }

    private static List<String> getServerClassPath() {
        String classpath = System.getProperty("java.class.path");
        return newArrayList(classpath.split(System.getProperty("path.separator")));
    }

    @Override
    public void gc() {
        checkPermission(Permission.ADMIN);
        garbageCollectionService.runGarbageCollector();

        EventBusHolder.publish(new RepositoryGarbageCollectedEvent());
    }
}
