package com.xebialabs.deployit.core.rest.api;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;

import com.google.common.base.Function;
import com.google.common.collect.Iterables;

import com.xebialabs.deployit.core.rest.secured.AbstractSecuredResource;
import com.xebialabs.deployit.engine.api.RoleService;
import com.xebialabs.deployit.engine.spi.event.*;
import com.xebialabs.deployit.event.EventBusHolder;
import com.xebialabs.deployit.security.Permissions;
import com.xebialabs.deployit.security.Role;

import static com.google.common.collect.Lists.newArrayList;
import static com.xebialabs.deployit.security.permission.Permission.EDIT_SECURITY;

@Controller
public class RoleResource extends AbstractSecuredResource implements RoleService {

    @Autowired
    private com.xebialabs.deployit.security.RoleService service;

    @Override
    public List<String> list() {
        List<String> roles = new ArrayList<String>();
        for (Role role : service.getRoles()) {
            roles.add(role.getName());
        }
        return roles;
    }

    @Override
    public List<String> listMyRoles() {
        return newArrayList(Iterables.transform(service.getRolesFor(Permissions.getAuthentication()), new Function<Role, String>() {
            public String apply(final Role input) {
                return input.getName();
            }
        }));
    }

    @Override
    public List<String> listRoles(final String username) {
        checkPermission(EDIT_SECURITY);
        return newArrayList(Iterables.transform(service.getRolesFor(username), new Function<Role, String>() {
            public String apply(final Role input) {
                return input.getName();
            }
        }));
    }

    @Override
    public void create(String name) {
        checkPermission(EDIT_SECURITY);

        List<Role> roles = service.readRoleAssignments();

        // Do nothing if role is already there
        for (Role role : roles) {
            if (role.getName().equals(name)) {
                return;
            }
        }

        // Create a new role
        Role newRole = new Role(name);
        roles.add(newRole);

        service.writeRoleAssignments(roles);
        
        EventBusHolder.publish(new RoleCreatedEvent(name));
    }

    @Override
    public void assign(String name, String principal) {
        checkPermission(EDIT_SECURITY);

        List<Role> roles = service.readRoleAssignments();

        // Assign existing role
        boolean found = false;
        for (Role role : roles) {
            if (role.getName().equals(name)) {
                found = true;

                if (!role.getPrincipalsAssigned().contains(principal)) {
                    role.getPrincipalsAssigned().add(principal);
                }
            }
        }

        // Create a new role
        if (!found) {
            Role newRole = new Role(name);
            newRole.getPrincipalsAssigned().add(principal);
            roles.add(newRole);
        }

        service.writeRoleAssignments(roles);
        
        EventBusHolder.publish(new PrincipalRoleAssignmentCreatedEvent(principal, name));
    }

    @Override
    public void unassign(String name, String principal) {
        checkPermission(EDIT_SECURITY);

        List<Role> roles = service.readRoleAssignments();

        for (Role role :roles) {
            if (role.getName().equals(name)) {
                role.getPrincipalsAssigned().remove(principal);
            }
        }

        service.writeRoleAssignments(roles);
        
        EventBusHolder.publish(new PrincipalRoleAssignmentDeletedEvent(principal, name));
    }

    @Override
    public void rename(String name, String newName) {
        checkPermission(EDIT_SECURITY);

        List<Role> roles = service.readRoleAssignments();

        for (Role role :roles) {
            if (role.getName().equals(name)) {
                role.setName(newName);
            }
        }

        service.writeRoleAssignments(roles);

        EventBusHolder.publish(new RoleRenamedEvent(name, newName));
    }

    @Override
    public void delete(String name) {
        checkPermission(EDIT_SECURITY);

        List<Role> roles = service.readRoleAssignments();

        Iterator<Role>iter = roles.iterator();
        while(iter.hasNext()) {
            if (iter.next().getName().equals(name)) {
                iter.remove();
            }
        }

        service.writeRoleAssignments(roles);
        
        EventBusHolder.publish(new RoleDeletedEvent(name));
    }
}
