package com.xebialabs.deployit.core.rest.api;

import com.google.common.base.Function;
import com.google.common.base.Predicate;
import com.google.common.collect.FluentIterable;
import com.google.common.collect.Lists;
import com.google.common.io.ByteStreams;
import com.google.common.io.OutputSupplier;
import com.xebialabs.deployit.checks.Checks;
import com.xebialabs.deployit.core.rest.resteasy.Workdir;
import com.xebialabs.deployit.core.rest.resteasy.WorkdirHolder;
import com.xebialabs.deployit.core.rest.secured.AbstractSecuredResource;
import com.xebialabs.deployit.engine.api.RoleService;
import com.xebialabs.deployit.engine.api.dto.ArtifactAndData;
import com.xebialabs.deployit.engine.api.dto.ConfigurationItemId;
import com.xebialabs.deployit.engine.replacer.Placeholders;
import com.xebialabs.deployit.engine.spi.command.CreateCiCommand;
import com.xebialabs.deployit.engine.spi.command.CreateCisCommand;
import com.xebialabs.deployit.engine.spi.command.RepositoryBaseCommand;
import com.xebialabs.deployit.engine.spi.command.UpdateCiCommand;
import com.xebialabs.deployit.event.EventBusHolder;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import com.xebialabs.deployit.plugin.api.udm.artifact.Artifact;
import com.xebialabs.deployit.plugin.api.udm.artifact.FolderArtifact;
import com.xebialabs.deployit.plugin.api.udm.artifact.SourceArtifact;
import com.xebialabs.deployit.repository.ConfigurationItemData;
import com.xebialabs.deployit.repository.RepositoryService;
import com.xebialabs.deployit.repository.SearchParameters;
import com.xebialabs.deployit.repository.WorkDir;
import com.xebialabs.deployit.security.Permissions;
import com.xebialabs.deployit.security.permission.Permission;
import com.xebialabs.deployit.service.replacement.MustachePlaceholderScanner;
import com.xebialabs.deployit.service.replacement.PlaceholderScanningFailedException;
import com.xebialabs.deployit.service.validation.Validator;
import com.xebialabs.deployit.util.TFiles;
import com.xebialabs.overthere.local.LocalFile;
import nl.javadude.t2bus.event.strategy.ThrowingEventHandlerStrategy;
import nl.javadude.t2bus.event.strategy.ThrowingRuntimeExceptionHandlerStrategy;
import org.joda.time.DateTime;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;

import java.io.IOException;
import java.io.OutputStream;
import java.util.Collection;
import java.util.List;

import static com.google.common.collect.Collections2.filter;
import static com.google.common.collect.FluentIterable.from;
import static com.google.common.collect.Lists.newArrayList;
import static com.xebialabs.deployit.checks.Checks.checkArgument;
import static com.xebialabs.deployit.core.rest.api.DtoReader.ciDataToCiId;
import static com.xebialabs.deployit.core.rest.api.SearchParameterFactory.createSearchParam;

@Controller
public class RepositoryResource extends AbstractSecuredResource implements com.xebialabs.deployit.engine.api.RepositoryService {
    public static final Predicate<String> canRead = new Predicate<String>() {
        @Override
        public boolean apply(String input) {
            return Permission.READ.getPermissionHandler().hasPermission(input);
        }
    };

    public static final Predicate<ConfigurationItemData> hasReadRight = new Predicate<ConfigurationItemData>() {
        @Override
        public boolean apply(ConfigurationItemData input) {
            return Permission.READ.getPermissionHandler().hasPermission(input.getId());
        }
    };

    public static final Predicate<ConfigurationItem> canEdit = new Predicate<ConfigurationItem>() {
        @Override
        public boolean apply(ConfigurationItem input) {
            return Permission.EDIT_REPO.getPermissionHandler().hasPermission(input.getId());
        }
    };

    @Autowired
    private RepositoryService repositoryService;

    @Autowired
    private Validator validator;

    @Autowired
    private RoleService roleService;

    @Override
    public ConfigurationItem create(String id, ConfigurationItem ci) {
        checkPermission(Permission.EDIT_REPO, id);
        Checks.checkArgument(id.equals(ci.getId()), "The Configuration item id is [%s], but the id parameter is [%s]", ci.getId(), id);
        return createInternal(ci);
    }

    @Override
    @Workdir
    public ConfigurationItem create(String id, ArtifactAndData aad) {
        checkPermission(Permission.EDIT_REPO, id);
        Artifact artifact = aad.getArtifact();
        Checks.checkArgument(id.equals(artifact.getId()), "The Configuration item id is [%s], but the id parameter is [%s]", artifact.getId(), id);
        try {
            artifact.setFile(readArtifactData(aad, WorkdirHolder.get()));

            if (artifact instanceof FolderArtifact) {
                Checks.checkTrue(TFiles.isArchive(artifact.getFile().getPath()), "Folder artifact [%s] is not a valid ZIP archive", artifact.getFile().getName());
            }

            if (artifact instanceof SourceArtifact) {
                try {
                    Placeholders.scanPlaceholders((SourceArtifact) artifact, new MustachePlaceholderScanner());
                } catch (RuntimeException re) {
                    PlaceholderScanningFailedException.throwException(artifact.getId(), re);
                }
            }

            return createInternal(artifact);
        } finally {
            WorkdirHolder.get().delete();
        }
    }

    private static LocalFile readArtifactData(ArtifactAndData aad, WorkDir workDir) {
        checkArgument(aad.getFilename() != null, "The filename for the artifact should not be null");
        final LocalFile localFile = workDir.newFile(aad.getFilename());
        try {
            ByteStreams.write(aad.getData(), new OutputSupplier<OutputStream>() {
                @Override
                public OutputStream getOutput() throws IOException {
                    return localFile.getOutputStream();
                }
            });
        } catch (IOException e) {
            throw new RuntimeException("Could not write Artifact data for [" + aad.getArtifact().getId() + "] to [" + localFile + "]");
        }
        return localFile;
    }

    @Override
    public List<ConfigurationItem> create(List<ConfigurationItem> cis) {
        List<String> transform = Lists.transform(cis, new Function<ConfigurationItem, String>() {
            public String apply(ConfigurationItem from) {
                return from.getId();
            }
        });
        for (String s : transform) {
            checkPermission(Permission.EDIT_REPO, s);

        }
        return createInternal(cis);
    }

    @Override
    public ConfigurationItem read(String id) {
        checkPermission(Permission.READ, id);
        return repositoryService.read(id);
    }

    @Override
    public List<ConfigurationItem> read(List<String> ids) {
        List<ConfigurationItem> result = newArrayList();
        Collection<String> datas = filter(ids, canRead);
        for (String data : datas) {
            try {
                final ConfigurationItem entity = repositoryService.read(data);
                result.add(entity);
            } catch (RuntimeException re) {
                logger.error("Could not read configuration item [{}]", data);
                logger.error("Exception was: ", re);
            }
        }
        return result;
    }

    @Override
    public ConfigurationItem move(String id, String newLocation) {
        checkPermission(Permission.EDIT_REPO, id);
        ConfigurationItem entity = repositoryService.read(id);
        repositoryService.move(entity, newLocation);
        return read(newLocation);
    }

    @Override
    public ConfigurationItem rename(String id, String newName) {
        checkPermission(Permission.EDIT_REPO, id);
        repositoryService.rename(id, newName);
        return repositoryService.read(id.substring(0, id.lastIndexOf('/') + 1) + newName);
    }

    @Override
    public List<ConfigurationItemId> query(Type type, String parent, String namePattern, DateTime lastModifiedBefore, DateTime lastModifiedAfter, long page, long resultPerPage) {
        SearchParameters searchParam = createSearchParam(type, page, resultPerPage, parent, namePattern);
        FluentIterable<ConfigurationItemId> filter = from(repositoryService.list(searchParam)).filter(hasReadRight).transform(ciDataToCiId);
        return newArrayList(filter);
    }

    @Override
    public List<ConfigurationItem> validate(final List<ConfigurationItem> cis) {
        validator.validateCis(cis);
        return cis;
    }

    @Override
    public ConfigurationItem update(String id, ConfigurationItem ci) {
        checkPermission(Permission.EDIT_REPO, id);
        Checks.checkArgument(id.equals(ci.getId()), "The Configuration item id is [%s], but the id parameter is [%s]", ci.getId(), id);
        return updateInternal(ci);
    }

    @Override
    @Workdir
    public ConfigurationItem update(String id, ArtifactAndData aad) {
        checkPermission(Permission.EDIT_REPO, id);
        Checks.checkArgument(id.equals(aad.getArtifact().getId()), "The Configuration item id is [%s], but the id parameter is [%s]", aad.getArtifact().getId(), id);
        try {
            aad.getArtifact().setFile(readArtifactData(aad, WorkdirHolder.get()));
            return updateInternal(aad.getArtifact());
        } finally {
            WorkdirHolder.get().delete();
        }
    }

    @Override
    public List<ConfigurationItem> update(List<ConfigurationItem> configurationItems) {
        Collection<ConfigurationItem> filter = filter(configurationItems, canEdit);
        for (ConfigurationItem configurationItem : filter) {
            if (repositoryService.exists(configurationItem.getId())) {
                ConfigurationItem previous = repositoryService.read(configurationItem.getId(), false);
                publishCommand(new UpdateCiCommand(previous, configurationItem));
            } else {
                publishCommand(new CreateCiCommand(configurationItem));
            }
        }
        return reloadEntityAndCreateSuccessResponse(filter);
    }

    @Override
    public void delete(String id) {
        if (Permission.IMPORT_REMOVE.isApplicableTo(id)) {
            checkPermission(Permission.IMPORT_REMOVE, id);
        } else {
            checkPermission(Permission.EDIT_REPO, id);
        }
        repositoryService.delete(id);
    }

    @Override
    public Boolean exists(String id) {
        return repositoryService.exists(id);
    }

    private ConfigurationItem createInternal(final ConfigurationItem ci) {
        validator.validateCi(ci);
        publishCommand(new CreateCiCommand(ci));
        return repositoryService.read(ci.getId());
    }

    private List<ConfigurationItem> createInternal(List<ConfigurationItem> cis) {
        validator.validateCis(cis);
        RepositoryBaseCommand event = new CreateCisCommand(cis);
        publishCommand(event);

        return reloadEntityAndCreateSuccessResponse(cis);
    }

    private void publishCommand(final RepositoryBaseCommand event) {
        List<String> roles = roleService.listMyRoles();
        event.setSecurityContext(Permissions.getAuthenticatedUserName(), roles);
        EventBusHolder.publish(event, new ThrowingRuntimeExceptionHandlerStrategy());
    }

    private List<ConfigurationItem> reloadEntityAndCreateSuccessResponse(Collection<ConfigurationItem> cis) {
        List<ConfigurationItem> reloaded = Lists.newArrayList();

        for (ConfigurationItem ci : cis) {
            ConfigurationItem reloadedCi = repositoryService.read(ci.getId());
            reloaded.add(reloadedCi);
        }

        return reloaded;
    }

    private ConfigurationItem updateInternal(final ConfigurationItem ci) {
        validator.validateCi(ci);
        ConfigurationItem previous = repositoryService.read(ci.getId(), false);
        publishCommand(new UpdateCiCommand(previous, ci));
        return repositoryService.read(ci.getId());
    }

    private static final Logger logger = LoggerFactory.getLogger(RepositoryResource.class);
}
