package com.xebialabs.deployit.core.rest.api;


import com.google.common.base.Function;
import com.google.common.base.Predicate;
import com.google.common.collect.*;
import com.xebialabs.deployit.core.rest.resteasy.Workdir;
import com.xebialabs.deployit.core.rest.resteasy.WorkdirHolder;
import com.xebialabs.deployit.core.rest.secured.AbstractSecuredResource;
import com.xebialabs.deployit.engine.api.dto.ConfigurationItemId;
import com.xebialabs.deployit.engine.api.dto.Deployment;
import com.xebialabs.deployit.engine.api.dto.Deployment.DeploymentType;
import com.xebialabs.deployit.engine.api.dto.ValidatedConfigurationItem;
import com.xebialabs.deployit.engine.spi.event.TaskCreatedEvent;
import com.xebialabs.deployit.engine.spi.exception.DeployitException;
import com.xebialabs.deployit.engine.spi.exception.HttpResponseCodeResult;
import com.xebialabs.deployit.engine.tasker.Engine;
import com.xebialabs.deployit.engine.tasker.Task;
import com.xebialabs.deployit.engine.tasker.TaskSpecification;
import com.xebialabs.deployit.event.EventBusHolder;
import com.xebialabs.deployit.plugin.api.deployment.specification.DeltaSpecification;
import com.xebialabs.deployit.plugin.api.reflect.DescriptorRegistry;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.plugin.api.udm.*;
import com.xebialabs.deployit.plugin.api.validation.ValidationMessage;
import com.xebialabs.deployit.repository.RepositoryService;
import com.xebialabs.deployit.repository.WorkDir;
import com.xebialabs.deployit.repository.WorkDirFactory;
import com.xebialabs.deployit.security.permission.Permission;
import com.xebialabs.deployit.server.api.util.IdGenerator;
import com.xebialabs.deployit.service.deployment.ChangeSetBuilder;
import com.xebialabs.deployit.service.deployment.DeployedService;
import com.xebialabs.deployit.service.deployment.DeploymentService;
import com.xebialabs.deployit.service.deployment.RollbackService;
import com.xebialabs.deployit.service.validation.Validator;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;

import javax.ws.rs.PathParam;
import java.util.*;

import static com.google.common.collect.Lists.newArrayList;
import static com.xebialabs.deployit.checks.Checks.checkArgument;
import static com.xebialabs.deployit.checks.Checks.checkNotNull;

@Controller
public class DeploymentResource extends AbstractSecuredResource implements com.xebialabs.deployit.engine.api.DeploymentService {

    @Autowired
    private RepositoryService repositoryService;

    @Autowired
    private DeploymentService deploymentService;

    @Autowired
    private DeployedService deployedService;

    @Autowired
    private RollbackService rollbackService;

    @Autowired
    private Engine engine;

    @Autowired
    private Validator validator;

    @Autowired
    private WorkDirFactory workDirFactory;

    @Override
    public Deployment prepareInitial(String versionId, String environmentId) {
        logger.trace("prepareInitial {}, {}", versionId, environmentId);
        checkPermission(Permission.DEPLOY_INITIAL, environmentId);
        checkNotNull(versionId, "version");
        checkNotNull(environmentId, "environment");

        ConfigurationItem version = repositoryService.read(versionId);
        ConfigurationItem environment = repositoryService.read(environmentId);

        checkArgument(version instanceof Version, "%s is not a Version", versionId);
        checkArgument(environment instanceof Environment, "%s is not an Environment", environmentId);

        Deployment deployment = createDeployment((Version) version, (Environment) environment);

        return deployment;
    }

    @SuppressWarnings("rawtypes")
    @Override
    public Deployment prepareUpdate(String newVersionId, String deployedApplicationId) {
        logger.trace("prepareUpgrade {}, {}", newVersionId, deployedApplicationId);
        checkNotNull(newVersionId, "version");
        checkNotNull(deployedApplicationId, "deployedApplication");

        WorkDir existingWorkDir = workDirFactory.newWorkDir();
        WorkDir newWorkDir = workDirFactory.newWorkDir();
        try {
            ConfigurationItem deployedApplication = repositoryService.read(deployedApplicationId, existingWorkDir);
            checkArgument(deployedApplication instanceof DeployedApplication, "%s is not a DeployedApplication", deployedApplicationId);

            Environment env = ((DeployedApplication) deployedApplication).getEnvironment();
            checkPermission(Permission.DEPLOY_UPGRADE, env.getId());

            ConfigurationItem newVersion = repositoryService.read(newVersionId, newWorkDir);
            checkArgument(newVersion instanceof Version, "%s is not a Version", newVersionId);

            ListMultimap<Boolean, ConfigurationItem> upgradedDeployeds = deployedService.generateUpgradedDeployeds((Version) newVersion, (DeployedApplication) deployedApplication);

            DeployedApplication deplApp = (DeployedApplication) deployedApplication;
            deplApp.setDeployeds(Sets.<Deployed>newHashSet());
            deplApp.setVersion((Version) newVersion);
            Deployment deployment = createDeployment(deplApp, DeploymentType.UPDATE);
            deployment.setDeployeds(convertDeployeds(upgradedDeployeds));

            return deployment;
        } finally {
            existingWorkDir.delete();
            newWorkDir.delete();
        }
    }

    @Override
    public Deployment prepareUndeploy(String deployedApplicationId) {
        logger.trace("prepareUndeployApplication {}", deployedApplicationId);

        // Read from repository
        checkNotNull(deployedApplicationId, "deployedApplication");
        ConfigurationItem application = repositoryService.read(deployedApplicationId);
        checkArgument(application instanceof DeployedApplication, "%s is not a DeployedApplication", deployedApplicationId);
        DeployedApplication deployedApplication = (DeployedApplication) application;

        // Create deployment if allowed
        checkPermission(Permission.UNDEPLOY, deployedApplication.getEnvironment().getId());
        Deployment deployment = createDeployment(deployedApplication, DeploymentType.UNDEPLOYMENT);

        return deployment;
    }

    @Override
    public Deployment generateAllDeployeds(Deployment deployment) {
        logger.trace("generateAllDeployeds {}", deployment);
        checkNotNull(deployment, "deployment");
        DeployedApplication deployedApplication = (DeployedApplication) deployment.getDeployedApplication();
        checkPermission(Permission.DEPLOY_INITIAL, deployedApplication.getEnvironment().getId());
        Version version = deployedApplication.getVersion();
        Environment environment = deployedApplication.getEnvironment();

        ListMultimap<Boolean, ConfigurationItem> initialDeployeds = deployedService.generateAllDeployeds(version, environment);

        deployment.addAll(convertDeployeds(initialDeployeds));
        return deployment;
    }

    private static List<ConfigurationItem> convertDeployeds(ListMultimap<Boolean, ConfigurationItem> initialDeployeds) {
        List<ConfigurationItem> deployeds = newArrayList();
        deployeds.addAll(initialDeployeds.get(true));
        for (ConfigurationItem configurationItem : initialDeployeds.get(false)) {
            ValidatedConfigurationItem vci = new ValidatedConfigurationItem(configurationItem);
            vci.getValidations().add(new ValidationMessage(configurationItem.getId(), "source", "The deployable for this deployed is missing from the package."));
            deployeds.add(vci);
        }
        return deployeds;
    }

    @Override
    public Deployment generateSelectedDeployeds(List<String> deployableIds, Deployment deployment) {
        logger.trace("generateSelectedDeployeds {}, {}", deployableIds, deployment);
        checkNotNull(deployment, "deployment");
        DeployedApplication deployedApplication = (DeployedApplication) deployment.getDeployedApplication();
        checkPermission(Permission.DEPLOY_INITIAL, deployedApplication.getEnvironment().getId());
        Version version = deployedApplication.getVersion();
        Environment environment = deployedApplication.getEnvironment();
        checkArgument(deployableIds.size() > 0, "Should select at least one deployable to generate a deployed");

        Map<String, Deployable> deployables = Maps.uniqueIndex(version.getDeployables(), new Function<Deployable, String>() {
            public String apply(Deployable input) {
                return input.getId();
            }
        });

        final List<ConfigurationItem> deployableCIs = Lists.newArrayList();
        for (String id : deployableIds) {
            checkArgument(deployables.containsKey(id), "All sources should be from same package");
            deployableCIs.add(deployables.get(id));
        }

        ListMultimap<Boolean, ConfigurationItem> selectedDeployeds = deployedService.generateSelectedDeployeds(deployableCIs, environment);
        deployment.addAll(convertDeployeds(selectedDeployeds));
        return deployment;
    }

    @Override
    public Deployment generateSingleDeployed(final String deployableId, final String containerId, Type deployedType, Deployment deployment) {
        logger.debug("Creating explicit deployed for [{}] and [{}]", deployableId, containerId);
        checkNotNull(deployment, "deployment");
        DeployedApplication deployedApplication = (DeployedApplication) deployment.getDeployedApplication();
        checkPermission(Permission.DEPLOY_INITIAL, deployedApplication.getEnvironment().getId());
        Version version = deployedApplication.getVersion();
        Environment environment = deployedApplication.getEnvironment();

        checkNotNull(deployableId, "deployable");
        checkNotNull(containerId, "container");

        ConfigurationItem deployable = Iterables.find(version.getDeployables(), new Predicate<Deployable>() {
            @Override
            public boolean apply(Deployable input) {
                return input.getId().equals(deployableId);
            }
        });
        checkArgument(deployable instanceof Deployable, "%s should be a Deployable.", deployableId);
        ConfigurationItem container = Iterables.find(environment.getMembers(), new Predicate<Container>() {
            @Override
            public boolean apply(Container input) {
                return input.getId().equals(containerId);
            }
        });
        checkArgument(container instanceof Container, "%s should be a Container.", containerId);

        ListMultimap<Boolean, ConfigurationItem> deployeds = deployedService.createSelectedDeployed((Deployable) deployable, (Container) container, deployedType, environment);
        deployment.addAll(convertDeployeds(deployeds));
        return deployment;
    }

    @Override
    public Deployment validate(Deployment deployment) {
        if (!validateDeploymentWithValidator(deployment)) {
            throw new InvalidDeploymentException(deployment);
        }

        return deployment;
    }

    @Override
    @Workdir
    public String createTask(Deployment deployment) {
        checkNotNull(deployment, "deployment");
        checkPermission(deployment);

        if (!validateDeploymentWithValidator(deployment)) {
            throw new InvalidDeploymentException("The deployment has validation errors.");
        }

        logger.trace("Creating task for {}", deployment);
        try {
            TaskSpecification deploymentTask = createDeploymentTask(deployment);
            String taskId = engine.register(deploymentTask);
            logger.debug("Registered {} task {} for Deployed Application {}", new Object[]{deployment.getDeploymentType(), taskId, deployment.getDeployedApplication().getId()});

            EventBusHolder.publish(new TaskCreatedEvent(taskId, deployment.getDeploymentType().toString(), deployment.getDeployedApplication().getId()));

            return taskId;
        } catch (RuntimeException e) {
            WorkdirHolder.get().delete();
            throw e;
        }
    }

    @Override
    public String rollback(@PathParam("taskid") final String taskid) {
        Task retrieve = engine.retrieve(taskid);
        TaskSpecification rollback = rollbackService.rollback(retrieve);
        return engine.register(rollback);
    }


    private void checkPermission(Deployment deployment) {
        String environment = ((DeployedApplication) deployment.getDeployedApplication()).getEnvironment().getId();
        switch (deployment.getDeploymentType()) {
            case INITIAL:
                checkPermission(Permission.DEPLOY_INITIAL, environment);
                break;
            case UPDATE:
                checkPermission(Permission.DEPLOY_UPGRADE, environment);
                break;
            case UNDEPLOYMENT:
                checkPermission(Permission.UNDEPLOY, environment);
                break;
        }
    }

    private TaskSpecification createDeploymentTask(Deployment deployment) {
        switch (deployment.getDeploymentType()) {
            case INITIAL:
                return createInitialDeploymentTask(deployment);
            case UPDATE:
                return createUpdateTask(deployment);
            case UNDEPLOYMENT:
                return createUndeploymentTask(deployment);
            default:
                throw new IllegalArgumentException("Unknown deployment type: " + deployment.getDeploymentType());
        }
    }

    private TaskSpecification createInitialDeploymentTask(Deployment deployment) {
        DeployedApplication deployedApp = (DeployedApplication) deployment.getDeployedApplication();
        deployedApp.addDeployeds(getDeployeds(deployment));
        DeltaSpecification deltaSpecification = deploymentService.prepareInitialDeployment(deployedApp);
        repositoryService.checkReferentialIntegrity(ChangeSetBuilder.determineChanges(deltaSpecification));
        return deploymentService.getTaskSpecification(deltaSpecification, WorkdirHolder.get());
    }

    private TaskSpecification createUpdateTask(Deployment deployment) {
        DeployedApplication deployedApp = (DeployedApplication) deployment.getDeployedApplication();
        deployedApp.addDeployeds(getDeployeds(deployment));
        // TODO in case of an upgrade, we need to verify that a user on the CLI didn't break the security contract by manually adding extra deployeds...

        WorkDir workDirForExistingDeployment = workDirFactory.newWorkDir();
        try {
            DeployedApplication existingDeployedApplication = repositoryService.read(deployment.getDeployedApplication().getId(), workDirForExistingDeployment);
            DeltaSpecification deltaSpecification = deploymentService.prepareUpgradeDeployment(deployedApp, existingDeployedApplication);
            repositoryService.checkReferentialIntegrity(ChangeSetBuilder.determineChanges(deltaSpecification));
            return deploymentService.getTaskSpecification(deltaSpecification, workDirForExistingDeployment, WorkdirHolder.get());
        } catch (RuntimeException e) {
            workDirForExistingDeployment.delete();
            WorkdirHolder.get().delete();
            throw e;
        }
    }

    private TaskSpecification createUndeploymentTask(Deployment deployment) {
        DeployedApplication deployedApplication = (DeployedApplication) deployment.getDeployedApplication();

        checkPermission(Permission.UNDEPLOY, deployedApplication.getEnvironment().getId());

        DeltaSpecification deltaSpecification = deploymentService.prepareUndeployment(deployedApplication);
        repositoryService.checkReferentialIntegrity(ChangeSetBuilder.determineChanges(deltaSpecification));
        return deploymentService.getTaskSpecification(deltaSpecification, WorkdirHolder.get());
    }

    @SuppressWarnings("rawtypes")
    private static Collection<Deployed> getDeployeds(Deployment deployment) {
        Collection<ConfigurationItem> deployedEntities = deployment.getDeployeds();
        return Collections2.transform(deployedEntities, new Function<ConfigurationItem, Deployed>() {
            @Override
            public Deployed apply(ConfigurationItem input) {
                checkArgument(input instanceof Deployed, "%s should be a Deployed", input.getId());
                return (Deployed) input;
            }
        });
    }

    private boolean validateDeploymentWithValidator(Deployment deployment) {
        ConfigurationItem deployedApplicationEntity = deployment.getDeployedApplication();
        List<ValidationMessage> messages = validator.validate(deployedApplicationEntity, Collections.EMPTY_LIST);
        //filter out any deployed property error messages on the deployed application
        Iterable<ValidationMessage> filteredMessages = Iterables.filter(messages, new Predicate<ValidationMessage>() {
            @Override
            public boolean apply(ValidationMessage input) {
                return !input.getPropertyName().equals("deployeds");
            }
        });
        if (Iterables.size(filteredMessages) > 0) {
            ValidatedConfigurationItem validatedConfigurationItem = new ValidatedConfigurationItem(deployment.getDeployedApplication());
            validatedConfigurationItem.setValidations(newArrayList(filteredMessages));
            deployment.setDeployedApplication(validatedConfigurationItem);
            return false;
        }

        return validateDeployedsWithValidator(deployment.getDeployeds());
    }

    private boolean validateDeployedsWithValidator(final List<ConfigurationItem> deployeds) {
        boolean allValid = true;
        List<ConfigurationItem> toBeValidated = newArrayList(deployeds);
        deployeds.clear();
        for (ConfigurationItem validate : toBeValidated) {
            List<ValidationMessage> messages = validator.validate(validate, toBeValidated);
            if (!messages.isEmpty()) {
                ValidatedConfigurationItem vci = new ValidatedConfigurationItem(validate);
                vci.setValidations(messages);
                deployeds.add(vci);
                allValid = false;
            } else {
                deployeds.add(validate);
            }
        }
        return allValid;
    }

    private static Deployment createDeployment(Version version, Environment env) {

        DeployedApplication deployedApplication = DescriptorRegistry.getDescriptor(Type.valueOf(DeployedApplication.class)).newInstance();
        deployedApplication.setVersion(version);
        deployedApplication.setEnvironment(env);
        IdGenerator.generateId(deployedApplication);

        return createDeployment(deployedApplication, DeploymentType.INITIAL);
    }

    private static Deployment createDeployment(DeployedApplication deployedApplication, DeploymentType type) {
        Deployment deployment = new Deployment();
        deployment.setDeploymentType(type);
        deployment.setDeployedApplication(deployedApplication);

        deployment.setDeployables(sort(newArrayList(Iterables.transform(deployedApplication.getVersion().getDeployables(), DtoReader.ciToCiId))));
        deployment.setContainers(sort(newArrayList(Iterables.transform(deployedApplication.getEnvironment().getMembers(), DtoReader.ciToCiId))));

        return deployment;
    }

    private static List<ConfigurationItemId> sort(List<ConfigurationItemId> toBeSorted) {
        Collections.sort(toBeSorted, new Comparator<ConfigurationItemId>() {
            @Override
            public int compare(ConfigurationItemId o1, ConfigurationItemId o2) {
                return o1.getId().compareTo(o2.getId());
            }
        });
        return toBeSorted;
    }

    private static final Logger logger = LoggerFactory.getLogger(DeploymentResource.class);

    @SuppressWarnings("serial")
    @HttpResponseCodeResult(statusCode = 400)
    private class InvalidDeploymentException extends DeployitException {

        public InvalidDeploymentException(Deployment deployment) {
            super(deployment);
        }

        public InvalidDeploymentException(String message) {
            super(message);
        }
    }
}
