package com.xebialabs.deployit.core.rest.api.reports.widgets;

import static com.google.common.collect.Lists.newArrayList;

import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.concurrent.TimeUnit;

import com.google.common.base.Function;
import com.google.common.collect.Maps;
import com.xebialabs.deployit.core.api.dto.Report;
import com.xebialabs.deployit.core.api.dto.Report.ReportLine;
import com.xebialabs.deployit.core.api.resteasy.Date;
import com.xebialabs.deployit.task.ArchivedTaskSearchParameters;
import com.xebialabs.deployit.task.DeploymentTaskInfo;
import com.xebialabs.deployit.task.TaskArchive;
import com.xebialabs.deployit.task.jcrarchive.JcrTaskArchive.TaskCallback;

public class DeploymentTrendsPercentileWidget extends DashboardWidgetBase {

	private static final int EIGHTIETH_PERCENTILE = 80;
	
	private static enum ReportTimeScale {
		SECS, MINS, HRS;
	}

	private DeploymentTrendsPercentileWidget(final TaskArchive taskArchive) {
		this.taskArchive = taskArchive;
	}

	public final static Widget getInstance(final TaskArchive taskArchive){
		return new DeploymentTrendsPercentileWidget(taskArchive);
	};
	
	@Override
	public Report getReport(final Date begin, final Date end) {
		Map<String, List<DeploymentTaskInfo>> groupedTasks = groupSuccesfulTasksByMonth(begin, end);
		Map<String, Long> monthPercentileMap = calculatePercentiles(groupedTasks);
		return generateReportData(monthPercentileMap);
	}

	private Map<String, List<DeploymentTaskInfo>> groupSuccesfulTasksByMonth(Date begin, Date end) {
		ArchivedTaskSearchParameters params = new ArchivedTaskSearchParameters().createdBetween(begin.getCalendar(), end.getCalendar()).thatCompleted();
		final Map<String, List<DeploymentTaskInfo>> groupedTasks = initializeMap(begin, end, new Function<String, List<DeploymentTaskInfo>>() {
			public List<DeploymentTaskInfo> apply(String input) {
				return newArrayList();
			}
		});

		taskArchive.searchTasksWithoutLoadingSteps(params, new TaskCallback() {
			public void doWithTask(DeploymentTaskInfo task) {
				String month = monthFormat.format(task.getCompletionDate().getTime());
				List<DeploymentTaskInfo> dataPerMonth = groupedTasks.get(month);
				if(dataPerMonth != null) {
					dataPerMonth.add(task);
				}
			}
		});

		return groupedTasks;
	}

	private Map<String, Long> calculatePercentiles(Map<String, List<DeploymentTaskInfo>> groupedTasks) {
		return Maps.transformValues(groupedTasks, new Function<List<DeploymentTaskInfo>, Long>() {
			@Override
			public Long apply(List<DeploymentTaskInfo> input) {
				return getPercentile(input, EIGHTIETH_PERCENTILE);
			}
		});
	}

	/**
	 * Calculates percentile by given ratio by using nearest rank algorithm.
	 * 
	 * @see http://en.wikipedia.org/wiki/Percentile
	 * @param tasks
	 * @param percentileRatio
	 * @return
	 */
	private long getPercentile(List<DeploymentTaskInfo> tasks, int percentileRatio) {
		if (tasks.size() == 0)
			return 0;
		Collections.sort(tasks, new Comparator<DeploymentTaskInfo>() {
			@Override
			public int compare(DeploymentTaskInfo o1, DeploymentTaskInfo o2) {
				return (int) (o1.getDurationInMillis() - o2.getDurationInMillis());
			}
		});
		int indexOfPercentile = Long.valueOf(Math.round(((double)percentileRatio * tasks.size() / 100) + 0.5)).intValue();
		if (indexOfPercentile > 0)
			indexOfPercentile--;
		final DeploymentTaskInfo task = tasks.get(indexOfPercentile);
		return task.getDurationInMillis();
	}

	private Report generateReportData(Map<String, Long> monthPercentileMap) {
		final Report report = new Report();
		ReportTimeScale timeScale = findTimeScale(monthPercentileMap);
		for (String month : monthPercentileMap.keySet()) {
			ReportLine line = report.addLine();
			line.addValue("month", month);
			line.addValue("timeScale", timeScale.toString());
			switch (timeScale) {
			case SECS:
				line.addValue("deploymentTime", formatToSecs(monthPercentileMap.get(month)));
				break;
			case MINS:
				line.addValue("deploymentTime", formatToMins(monthPercentileMap.get(month)));
				break;
			default:
				line.addValue("deploymentTime", formatToHours(monthPercentileMap.get(month)));
			}
		}
		return report;
	}

	private ReportTimeScale findTimeScale(Map<String, Long> monthPercentileMap) {
		Long maxDeploymentTime = Collections.max(monthPercentileMap.values());
		if (TimeUnit.MILLISECONDS.toSeconds(maxDeploymentTime) < 600) {
			return ReportTimeScale.SECS;
		} else if (TimeUnit.MILLISECONDS.toMinutes(maxDeploymentTime) < 600) {
			return ReportTimeScale.MINS;
		}
		return ReportTimeScale.HRS;
	}
}
